﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/pcv/pcv.h>
#include <nn/os.h>
#include <nn/spl/spl_Types.h>

namespace nn {
namespace pcv {
namespace driver {
namespace detail {

// エラーレポートで送信するモジュールごとの情報を保持するテーブル
class ModuleStateTable
{
public:
    void Initialize() NN_NOEXCEPT;
    void GetModuleStates(ModuleState* pOutModuleStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT;
    void SetPowerEnabled(Module moduleId, bool enabled) NN_NOEXCEPT;
    void SetClockEnabled(Module moduleId, bool enabled) NN_NOEXCEPT;
    void SetClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT;
    void SetMinVClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT;
    void SetReset(Module moduleId, bool asserted) NN_NOEXCEPT;

private:
    ModuleState m_ModuleStates[ModuleStateTableSize];
};

// エラーレポートで送信する電源ドメインごとの情報を保持するテーブル
class PowerDomainStateTable
{
public:
    void Initialize(nn::spl::Regulator) NN_NOEXCEPT;
    void GetPowerDomainStates(PowerDomainState* pOutPowerDomainStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT;
    void SetVoltageEnabled(PowerDomain powerDomain, bool enabled) NN_NOEXCEPT;
    void SetVoltageValue(PowerDomain powerDomain, MicroVolt microVolt) NN_NOEXCEPT;

private:
    Result GetRegisterValueFromLookUpTable(uint32_t* pOutReigsterValue) NN_NOEXCEPT;
    uint32_t GetMax77621CpuVoltageMicroVolt() NN_NOEXCEPT;
    uint32_t GetMax77812CpuVoltageMicroVolt() NN_NOEXCEPT;

private:
    const PowerDomain* m_pPowerDomainListForErrorReport;
    int m_PowerDomainStateTableSize;
    PowerDomainState m_PowerDomainStates[PowerDomainStateTableSizeMax];
    uintptr_t m_DvfsBase;
};

// エラーレポートで送信する DVFS テーブル
class DvfsTable
{
public:
    void Initialize() NN_NOEXCEPT;
    Result GetDvfsTable(uint32_t* pOutClocks, int32_t* pOutVoltages, int32_t* pOutCount, Module moduleId, int32_t maxCount) NN_NOEXCEPT;

private:
    struct DvfsTableModule
    {
        int32_t count; // 有効な clocks / voltages の数
        uint32_t clocks[nn::pcv::DvfsTableMaxDataCount];
        int32_t voltages[nn::pcv::DvfsTableMaxDataCount];
    };

    DvfsTableModule m_Cpu;
    DvfsTableModule m_Gpu;
    DvfsTableModule m_Emc;
};

// チップ情報
class FuseInfo
{
public:
    void Initialize() NN_NOEXCEPT;
    void GetFuseInfo(uint32_t* pOutValues, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT;

private:
    // チップ情報を取得するための FUSE のレジスタアドレス
    const uint64_t FusePhysicalAddress = 0x7000f800ull;
    const uint32_t FuseAddressSpaceSize = 0x400;

    uint32_t m_Values[nn::pcv::FuseValueCount];
};

// Dram情報
class DramInfo
{
public:
    void Initialize(uint32_t dramId) NN_NOEXCEPT;
    void GetDramId(uint32_t* pOutDramId) NN_NOEXCEPT;

private:
    uint32_t m_DramId;
};

} // namespace detail
} // namespace driver
} // namespace pcv
} // namespace nn
