﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <utility>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/pcv/detail/pcv_Log.h>
#include <nn/pcv/pcv_Modules.h>
#include <nn/pcv/pcv_PowerDomains.h>
#include <nn/pcv/pcv_Result.h>
#include <nn/spl/spl_Types.h>

#include <nne/max7762x/max7762x_results.h>
#include <nne/max7762x/max7762x_regulator_Types.h>
#include <nne/vcc/ipids.h>
#include <nne/vcc/vcc_dram_ids.h>
#include <nne/vcc/vcc_results.h>

#include "pcv_DramIdMap-hardware.nx.h"
#include "pcv_ModuleMap-hardware.nx.h"
#include "pcv_PowerDomainMap-hardware.nx.h"

namespace nn { namespace pcv { namespace driver { namespace detail {

// ConvertModuleToIp - static/private function to map ModuleIDs to ids used by VCC
::IP ConvertModuleToIp(nn::pcv::Module module) NN_NOEXCEPT
{
    for ( const auto& pair : ModuleIpPairs )
    {
        if ( pair.first == module )
        {
            return pair.second;
        }
    }

    NN_SDK_ASSERT(false, "Never reached here.\n");
    NN_DETAIL_PCV_WARN("Could not find module ID: %d.\n", static_cast<int>(module));

    return IP_NUM;
}

// Map PCV power domains to PMIC's regulators
nne::max7762x::RegulatorHandle ConvertPowerDomainToRegulatorHandle(nn::pcv::PowerDomain powerDomain) NN_NOEXCEPT
{
    for ( const auto& pair : PowerDomainRegulatorHandlePairs )
    {
        if ( pair.first == powerDomain )
        {
            return pair.second;
        }
    }

    NN_SDK_ASSERT(false, "Never reached here.\n");
    NN_DETAIL_PCV_WARN("Could not find power domain: %d.\n", static_cast<int>(powerDomain));

    return nne::max7762x::REG_MAX_HANDLE;;
}

nn::Result MakeResult(nne::vcc::Result vccResult) NN_NOEXCEPT
{
    nn::Result result;

    switch ( vccResult )
    {
    case nne::vcc::Result_Vcc_Success:            result = nn::ResultSuccess(); break;
    case nne::vcc::Result_Vcc_Id_Invalid:         result = nn::pcv::ResultInvalidModuleId(); break;
    case nne::vcc::Result_Vcc_Op_Not_Implemented: result = nn::pcv::ResultNotImplemented(); break;
    case nne::vcc::Result_Vcc_Op_Failed:          result = nn::pcv::ResultFailed(); break;

    // TORIAEZU(hotfix for HOS r8.1)
    default: result = nn::pcv::ResultFailed(); break;
    }

    return result;
}

nn::Result MakeResult(nne::max7762x::Result max7762xResult) NN_NOEXCEPT
{
    nn::Result result;

    switch ( max7762xResult )
    {
    case nne::max7762x::Result_Success:            result = nn::ResultSuccess(); break;
    case nne::max7762x::Result_i2C_BusNoAck:       result = nn::pcv::ResultPMICI2CBusNoAck(); break;
    case nne::max7762x::Result_i2C_BusBusy:        result = nn::pcv::ResultPMICI2CBusBusy(); break;
    case nne::max7762x::Result_Pull_Error:         result = nn::pcv::ResultPMICPullError(); break;
    case nne::max7762x::Result_Debounce_Range_Err: result = nn::pcv::ResultPMICPullError(); break;

    default: result = nn::pcv::ResultPMICOther(); break;
    }

    return result;
}

nne::vcc::DramId ConvertDramId(nn::spl::DramId dramId) NN_NOEXCEPT
{
    for ( const auto& pair : SplDramIdVccDramIdPairs )
    {
        if ( pair.first == dramId )
        {
            return pair.second;
        }
    }

    NN_DETAIL_PCV_WARN("Unsupported dram id! EMC DVFS setting will be skipped.\n");

    // 対応する VCC の DramId が存在しない場合 EMC の DVFS 設定をしないために無効な値を返す
    // TODO: 最終的には無効な値として InvalidDramId とか定義してもらうのが良い
    return static_cast<nne::vcc::DramId>(-1);
}

}}}} // namespace nn::pcv::driver::detail
