﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/pctl/pctl_PairingInfo.h>
#include <nn/pctl/pctl_ApiWatcher.h>
#include <nn/pctl/detail/pctl_PairingInfoBase.h>
#include <nn/pctl/detail/pctl_ResultHandler.h>
#include <nn/pctl/detail/ipc/pctl_IpcSession.h>

#include <nn/nifm/nifm_ApiClientManagement.h>

#include <cstring>
#include <algorithm>

namespace nn { namespace pctl {

namespace
{
    void WaitSystemEventPairingInfo(nn::sf::NativeHandle& nativeHandle) NN_NOEXCEPT
    {
        nn::os::SystemEventType eventObject;
        nn::os::AttachReadableHandleToSystemEvent(&eventObject,
            nativeHandle.GetOsHandle(),
            nativeHandle.IsManaged(),
            nn::os::EventClearMode_ManualClear);
        nativeHandle.Detach();
        nn::os::WaitSystemEvent(&eventObject);
        nn::os::DestroySystemEvent(&eventObject);
    }
}

PairingAccountInfo::PairingAccountInfo() NN_NOEXCEPT
{
    NN_UNUSED(m_Reserved);

    detail::ToHolder(this)->Initialize();
}

PairingAccountInfo::~PairingAccountInfo() NN_NOEXCEPT
{
    detail::ToHolder(this)->Invalidate();
}

bool PairingAccountInfo::GetAccountState(PairingAccountState* outState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInstanceValid(), "This instance is not valid");
    NN_SDK_REQUIRES_NOT_NULL(outState);

    int state = static_cast<int>(PairingAccountState::PairingAccountState_Active);
    auto result = detail::ipc::GetServiceForWatcher()->GetAccountState(&state,
        *detail::ToHolder(this));
    if (nn::pctl::ResultNeedsRefresh::Includes(result))
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    *outState = static_cast<PairingAccountState>(state);
    return true;
}

size_t PairingAccountInfo::GetNickname(char* pNickname, size_t nicknameSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInstanceValid(), "This instance is not valid");
    NN_SDK_REQUIRES_NOT_NULL(pNickname);

    uint32_t returnSize = 0;
    auto result = detail::ipc::GetServiceForWatcher()->GetAccountNickname(&returnSize,
        nn::sf::OutArray<char>(pNickname, nicknameSize),
        *detail::ToHolder(this));
    if (nn::pctl::ResultNeedsRefresh::Includes(result))
    {
        return 0;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return static_cast<size_t>(returnSize);
}

nn::Result PairingAccountInfo::GetMiiImage(size_t* pActualSize, void* pImage, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInstanceValid(), "This instance is not valid");
    NN_SDK_REQUIRES_NOT_NULL(pActualSize);

    bool accepted = nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId());
    NN_RESULT_THROW_UNLESS(accepted, nn::pctl::ResultConnectionNotAccepted());

    uint32_t actualSize = 0;
    detail::AsyncData data;
    nn::sf::NativeHandle eventHandle;
    auto result = detail::ipc::GetServiceForWatcher()->GetAccountMiiImageAsync(
        &data,
        &eventHandle,
        &actualSize,
        nn::sf::OutBuffer(reinterpret_cast<char*>(pImage), bufferSize),
        *detail::ToHolder(this)
        );
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);

    // InvalidHandle の場合は通信処理なしに完了しているのでそのまま終了する
    if (eventHandle.GetOsHandle() == nn::os::InvalidNativeHandle)
    {
        *pActualSize = static_cast<size_t>(actualSize);
        NN_RESULT_SUCCESS;
    }

    WaitSystemEventPairingInfo(eventHandle);

    result = detail::ipc::GetServiceForWatcher()->FinishGetAccountMiiImage(
        &actualSize,
        nn::sf::OutBuffer(reinterpret_cast<char*>(pImage), bufferSize),
        data
        );
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);

    *pActualSize = static_cast<size_t>(actualSize);
    NN_RESULT_SUCCESS;
}

nn::Result PairingAccountInfo::GetMiiImageContentType(size_t* pActualLength, char* outContentType, size_t maxLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInstanceValid(), "This instance is not valid");
    NN_SDK_REQUIRES_NOT_NULL(pActualLength);

    bool accepted = nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId());
    NN_RESULT_THROW_UNLESS(accepted, nn::pctl::ResultConnectionNotAccepted());

    uint32_t actualLength = 0;
    detail::AsyncData data;
    nn::sf::NativeHandle eventHandle;
    auto result = detail::ipc::GetServiceForWatcher()->GetAccountMiiImageContentTypeAsync(
        &data,
        &eventHandle,
        &actualLength,
        nn::sf::OutArray<char>(outContentType, maxLength),
        *detail::ToHolder(this)
        );
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);

    // InvalidHandle の場合は通信処理なしに完了しているのでそのまま終了する
    if (eventHandle.GetOsHandle() == nn::os::InvalidNativeHandle)
    {
        *pActualLength = static_cast<size_t>(actualLength);
        NN_RESULT_SUCCESS;
    }

    WaitSystemEventPairingInfo(eventHandle);

    result = detail::ipc::GetServiceForWatcher()->FinishGetAccountMiiImageContentType(
        &actualLength,
        nn::sf::OutArray<char>(outContentType, maxLength),
        data
        );
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);

    *pActualLength = static_cast<size_t>(actualLength);
    NN_RESULT_SUCCESS;
}

bool PairingAccountInfo::IsInstanceValid() const NN_NOEXCEPT
{
    return detail::ToHolder(this)->IsValid();
}

////////////////////////////////////////////////////////////////////////////////

PairingInfo::PairingInfo() NN_NOEXCEPT
{
    NN_UNUSED(m_Reserved);

    detail::ToHolder(this)->Initialize();
}

PairingInfo::~PairingInfo() NN_NOEXCEPT
{
    detail::ToHolder(this)->Invalidate();
}

bool PairingInfo::GetAccountInfo(PairingAccountInfo* pAccountInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInstanceValid(), "This instance is not valid");
    NN_SDK_REQUIRES(GetState() != PairingState_None, "The pairing is not active");

    auto result = detail::ipc::GetServiceForWatcher()->GetPairingAccountInfo(
        detail::ToHolder(pAccountInfo),
        *detail::ToHolder(this));
    if (nn::pctl::ResultNeedsRefresh::Includes(result))
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return true;
}

PairingState PairingInfo::GetState() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInstanceValid(), "This instance is not valid");

    return static_cast<PairingState>(detail::ToHolder(this)->state);
}

bool PairingInfo::IsInstanceValid() const NN_NOEXCEPT
{
    return detail::ToHolder(this)->IsValid();
}

}}
