﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/pctl/pctl_ApiWatcher.h>
#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_EventObjectHolder.h>
#include <nn/pctl/detail/pctl_PairingInfoBase.h>
#include <nn/pctl/detail/pctl_ResultHandler.h>
#include <nn/pctl/detail/ipc/pctl_IpcSession.h>

#include <nn/nifm/nifm_ApiClientManagement.h>

#include <nn/util/util_LockGuard.h>

namespace nn { namespace pctl {

namespace
{

    void WaitSystemEvent(nn::sf::NativeHandle& nativeHandle) NN_NOEXCEPT
    {
        nn::os::SystemEventType eventObject;
        nn::os::AttachReadableHandleToSystemEvent(&eventObject,
            nativeHandle.GetOsHandle(),
            nativeHandle.IsManaged(),
            nn::os::EventClearMode_ManualClear);
        nativeHandle.Detach();
        nn::os::WaitSystemEvent(&eventObject);
        nn::os::DestroySystemEvent(&eventObject);
    }
}

// 見守り機能サーバー連携処理関連関数

bool IsPairingActive() NN_NOEXCEPT
{
    bool value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->IsPairingActive(&value));
    return value;
}

bool GetSettingsLastUpdated(nn::time::PosixTime* outTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTime);

    auto result = detail::ipc::GetServiceForWatcher()->GetSettingsLastUpdated(outTime);
    if (nn::pctl::ResultNotFound::Includes(result))
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return true;
}

nn::Result RetrievePairingInfo(PairingInfo* pInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pInfo);

    bool accepted = nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId());
    NN_RESULT_THROW_UNLESS(accepted, nn::pctl::ResultConnectionNotAccepted());

    detail::AsyncData data;
    nn::sf::NativeHandle eventHandle;
    auto result = detail::ipc::GetServiceForWatcher()->RetrievePairingInfoAsync(&data, &eventHandle);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);
    // InvalidHandle の場合は何も処理をしていないのでそのまま終了する
    if (eventHandle.GetOsHandle() == nn::os::InvalidNativeHandle)
    {
        detail::ToHolder(pInfo)->id = 0;
        detail::ToHolder(pInfo)->state = PairingState::PairingState_None;
        NN_RESULT_SUCCESS;
    }

    WaitSystemEvent(eventHandle);

    result = detail::ipc::GetServiceForWatcher()->FinishRetrievePairingInfo(detail::ToHolder(pInfo), data);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);

    NN_RESULT_SUCCESS;
}

nn::Result UnlinkPairing(bool force) NN_NOEXCEPT
{
    bool accepted = nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId());
    NN_RESULT_THROW_UNLESS(accepted, nn::pctl::ResultConnectionNotAccepted());

    detail::AsyncData data;
    nn::sf::NativeHandle eventHandle;
    auto result = detail::ipc::GetServiceForWatcher()->UnlinkPairingAsync(&data, &eventHandle, force);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);
    // InvalidHandle の場合は何も処理をしていないのでそのまま終了する
    if (eventHandle.GetOsHandle() == nn::os::InvalidNativeHandle)
    {
        NN_RESULT_SUCCESS;
    }

    WaitSystemEvent(eventHandle);

    result = detail::ipc::GetServiceForWatcher()->FinishUnlinkPairing(data, force);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);

    NN_RESULT_SUCCESS;
}

void DeletePairing() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->DeletePairing());
}

// 見守り機能同期処理関数

nn::Result SynchronizeParentalControlSettings(nn::time::PosixTime* outLastUpdatedTime) NN_NOEXCEPT
{
    bool accepted = nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId());
    NN_RESULT_THROW_UNLESS(accepted, nn::pctl::ResultConnectionNotAccepted());

    detail::AsyncData data;
    nn::sf::NativeHandle eventHandle;
    auto result = detail::ipc::GetServiceForWatcher()->SynchronizeParentalControlSettingsAsync(&data, &eventHandle);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);
    // InvalidHandle の場合は何も処理をしていないのでそのまま終了する
    if (eventHandle.GetOsHandle() == nn::os::InvalidNativeHandle)
    {
        NN_RESULT_SUCCESS;
    }

    WaitSystemEvent(eventHandle);

    nn::time::PosixTime lastUpdatedTime;
    result = detail::ipc::GetServiceForWatcher()->FinishSynchronizeParentalControlSettingsWithLastUpdated(&lastUpdatedTime, data);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    if (outLastUpdatedTime != nullptr)
    {
        *outLastUpdatedTime = lastUpdatedTime;
    }

    return result;
}

nn::Result RequestUpdateExemptionList(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(applicationId, nn::ncm::ApplicationId::GetInvalidId());

    bool accepted = nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId());
    NN_RESULT_THROW_UNLESS(accepted, nn::pctl::ResultConnectionNotAccepted());

    detail::AsyncData data;
    nn::sf::NativeHandle eventHandle;
    auto result = detail::ipc::GetServiceForWatcher()->RequestUpdateExemptionListAsync(&data, &eventHandle, applicationId, true);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_DO(result);
    // InvalidHandle の場合は何も処理をしていないのでそのまま終了する
    if (eventHandle.GetOsHandle() == nn::os::InvalidNativeHandle)
    {
        NN_RESULT_SUCCESS;
    }

    WaitSystemEvent(eventHandle);

    nn::time::PosixTime lastUpdatedTime;
    result = detail::ipc::GetServiceForWatcher()->FinishSynchronizeParentalControlSettingsWithLastUpdated(&lastUpdatedTime, data);
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);

    return result;
}

nn::Result RequestPostEvent(int* pOutCount, detail::service::watcher::EventData* pOutEventData, int eventDataSize) NN_NOEXCEPT
{
    bool accepted = nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId());
    NN_RESULT_THROW_UNLESS(accepted, nn::pctl::ResultConnectionNotAccepted());

    nn::sf::OutArray<detail::service::watcher::EventData> eventDataArray(pOutEventData, eventDataSize);
    auto result = detail::ipc::GetServiceForWatcher()->RequestPostEvents(pOutCount, eventDataArray);

    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);
    NN_RESULT_THROW(result);
}

nn::os::SystemEvent* GetSynchronizationEvent() NN_NOEXCEPT
{
    // プロセス単位のインスタンスとする
    NN_FUNCTION_LOCAL_STATIC(detail::EventObjectHolder, eventSynchronization);

    NN_UTIL_LOCK_GUARD(eventSynchronization);

    if (!eventSynchronization.IsInitialized())
    {
        nn::sf::NativeHandle eventHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->GetSynchronizationEvent(&eventHandle));
        eventSynchronization.InitializeWithReadableHandle(
            eventHandle.GetOsHandle(),
            eventHandle.IsManaged(),
            nn::os::EventClearMode::EventClearMode_AutoClear
            );
        eventHandle.Detach();
    }
    return eventSynchronization.GetSystemEvent();
}

nn::os::SystemEvent* GetUnlinkedEvent() NN_NOEXCEPT
{
    // プロセス単位のインスタンスとする
    NN_FUNCTION_LOCAL_STATIC(detail::EventObjectHolder, eventUnlinked);

    NN_UTIL_LOCK_GUARD(eventUnlinked);

    if (!eventUnlinked.IsInitialized())
    {
        nn::sf::NativeHandle eventHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->GetUnlinkedEvent(&eventHandle));
        eventUnlinked.InitializeWithReadableHandle(
            eventHandle.GetOsHandle(),
            eventHandle.IsManaged(),
            nn::os::EventClearMode::EventClearMode_ManualClear
            );
        eventHandle.Detach();
    }
    return eventUnlinked.GetSystemEvent();
}

void ClearUnlinkedEvent() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->ClearUnlinkedEvent());
}

// プレイタイマー処理関数

void StartPlayTimer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->StartPlayTimer());
}

void StopPlayTimer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->StopPlayTimer());
}

bool IsRestrictedByPlayTimer() NN_NOEXCEPT
{
    bool value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->IsRestrictedByPlayTimer(&value));
    return value;
}

void GetPlayTimerSettings(PlayTimerSettings* pSettings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSettings);

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->GetPlayTimerSettings(pSettings));
}

nn::os::SystemEvent* GetPlayTimerEventToRequestSuspension() NN_NOEXCEPT
{
    // プロセス単位のインスタンスとする
    NN_FUNCTION_LOCAL_STATIC(detail::EventObjectHolder, eventSuspend);

    NN_UTIL_LOCK_GUARD(eventSuspend);

    if (!eventSuspend.IsInitialized())
    {
        nn::sf::NativeHandle eventHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->GetPlayTimerEventToRequestSuspension(&eventHandle));
        eventSuspend.InitializeWithReadableHandle(
            eventHandle.GetOsHandle(),
            eventHandle.IsManaged(),
            nn::os::EventClearMode::EventClearMode_AutoClear
            );
        eventHandle.Detach();
    }
    return eventSuspend.GetSystemEvent();
}

bool IsPlayTimerAlarmDisabled() NN_NOEXCEPT
{
    bool value;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->IsPlayTimerAlarmDisabled(&value));
    return value;
}

void SetPlayTimerSettingsForDebug(const PlayTimerSettings& settings) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->SetPlayTimerSettingsForDebug(settings));
}

void SetPlayTimerAlarmDisabledForDebug(bool isDisabled) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->SetPlayTimerAlarmDisabledForDebug(isDisabled));
}

// その他のサーバー連携関連関数

void CancelNetworkRequest() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetServiceForWatcher()->CancelNetworkRequest());
}

}}
