﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_WatcherEventStorage.h>

#include <nn/fs/fs_Result.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_FileSystem.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/time/time_TimeZoneApi.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

namespace
{
    static const char EventDataFileName[] = "eventlog.dat";
}

WatcherEventStorage::WatcherEventStorage() NN_NOEXCEPT :
    m_TimeZoneRule(),
    m_IsCommitSkipped(false),
    m_IsProcessingPostData(false),
    m_TotalEventCountFromBoot(0),
    m_EventCountOnPost(0)
{
    // header のみ初期化する(storage は不要)
    m_Data.header.startIndex = 0;
    m_Data.header.lastIndex = 0;
    m_Data.header.lastIndexOnLoad = 0;
    m_Data.header.isAnyRecorded = false;
    m_Data.header.isAnyRecordedOnLoad = false;

    m_WriteDataInfo.isWritingEventsNecessary = false;
    m_WriteDataInfo.writeStartIndex = 0;
    m_WriteDataInfo.writeLastIndex = 0;
}

void WatcherEventStorage::Initialize() NN_NOEXCEPT
{
    LoadDataFromFile();
}

void WatcherEventStorage::AddEvent(EventType eventType, const EventData::Payload* pPayload) NN_NOEXCEPT
{
    nn::time::PosixTime timestamp;
    // 時刻が取得できない場合は記録しない
    if (!common::GetNetworkTime(&timestamp))
    {
        return;
    }

    AddEvent(eventType, timestamp, pPayload);
}

void WatcherEventStorage::AddEvent(EventType eventType, const nn::time::PosixTime& timestamp, const EventData::Payload* pPayload) NN_NOEXCEPT
{
    // Watcher機能無効状態の場合は記録しない
    if (!IsWatcherAvailable())
    {
        return;
    }
    // ペアコン機能無効の場合は記録しない
    if (g_pMain->GetSettingsManager().IsAllFeaturesDisabled())
    {
        return;
    }
    // サーバー連携が行われていない場合は記録しない
    if (!g_pWatcher->GetNetworkManager().IsPairingActive())
    {
        return;
    }

    nn::time::LocationName locationName = g_pWatcher->GetWatcherEventManager().GetTimeLocationName();

    NN_UTIL_LOCK_GUARD(m_MutexData);
    AddEventInternalNoLock(eventType, timestamp, locationName, pPayload);
}

void WatcherEventStorage::InsertEvent(EventType eventType, const EventData::Payload* pPayload, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    // Watcher機能無効状態の場合は記録しない
    // (g_pWatcher が使えないのでここで判定)
    if (!IsWatcherAvailable())
    {
        return;
    }
    nn::time::LocationName locationName = g_pWatcher->GetWatcherEventManager().GetTimeLocationName();
    InsertEvent(eventType, locationName, pPayload, timeAt);
}

void WatcherEventStorage::InsertEvent(EventType eventType, const nn::time::LocationName& locationName, const EventData::Payload* pPayload, const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    // Watcher機能無効状態の場合は記録しない
    if (!IsWatcherAvailable())
    {
        return;
    }
    // ペアコン機能無効の場合は記録しない
    if (g_pMain->GetSettingsManager().IsAllFeaturesDisabled())
    {
        return;
    }
    // サーバー連携が行われていない場合は記録しない
    if (!g_pWatcher->GetNetworkManager().IsPairingActive())
    {
        return;
    }

    NN_UTIL_LOCK_GUARD(m_MutexData);
    InsertEventInternalNoLock(eventType, timeAt, locationName, pPayload);
}

uint16_t WatcherEventStorage::GetPartialDataForPost(EventData* outDataArray, uint16_t maxDataCount) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexData);

    // 既に処理が行われているので何もしない
    if (m_IsProcessingPostData)
    {
        return 0;
    }

    // 出力すべきデータがない場合は何もしない
    if (!m_Data.header.isAnyRecorded)
    {
        return 0;
    }

    int availableCount = static_cast<int>(m_Data.header.lastIndex) - static_cast<int>(m_Data.header.startIndex) + 1;
    if (availableCount <= 0)
    {
        availableCount += MaxEventCount;
    }
    if (availableCount >= static_cast<int>(maxDataCount))
    {
        availableCount = static_cast<int>(maxDataCount);
    }
    int lastIndex = static_cast<int>(m_Data.header.startIndex) + availableCount - 1;
    if (lastIndex >= MaxEventCount)
    {
        lastIndex -= MaxEventCount;
        int partialCount = MaxEventCount - static_cast<int>(m_Data.header.startIndex);
        std::memcpy(outDataArray, &m_Data.storage[m_Data.header.startIndex],
            sizeof(EventData) * partialCount);
        std::memcpy(&outDataArray[partialCount], m_Data.storage,
            sizeof(EventData) * (lastIndex + 1));
    }
    else
    {
        std::memcpy(outDataArray, &m_Data.storage[m_Data.header.startIndex],
            sizeof(EventData) * availableCount);
    }

    m_IsProcessingPostData = true;
    // m_TotalEventCountFromBoot がオーバーフローしていない限り
    // m_TotalEventCountFromBoot >= static_cast<uint32_t>(availableCount)
    m_EventCountOnPost = m_TotalEventCountFromBoot - static_cast<uint32_t>(availableCount);

    return static_cast<uint16_t>(availableCount);
}

void WatcherEventStorage::FinishPostPartialData(uint16_t postCount) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexData);

    if (!m_IsProcessingPostData)
    {
        return;
    }
    m_IsProcessingPostData = false;
    if (postCount == 0)
    {
        return;
    }
    // 送信処理中にデータがクリアされていたら調整不要のため関数を抜ける
    if (!m_Data.header.isAnyRecorded)
    {
        return;
    }

    // 上書きされているかどうかを考慮して postCount の数だけ startIndex を進める処理を行う

    // MaxEventCount + postCount を超えている場合はすべて上書きされているので
    // インデックスの更新操作が不要になる
    if (m_TotalEventCountFromBoot - m_EventCountOnPost > MaxEventCount + static_cast<uint32_t>(postCount))
    {
        return;
    }
    // 上記以外で MaxEventCount を超えている場合は一部上書きされている
    else if (m_TotalEventCountFromBoot - m_EventCountOnPost > MaxEventCount)
    {
        // 上書きされた数
        auto overwrittenCount = (m_TotalEventCountFromBoot - m_EventCountOnPost - MaxEventCount);
        // 上書きされていない時点の startIndex に postCount を加えたいので
        // postCount から上書きされた数を引いた値を加える
        // (postCount < static_cast<uint16_t>(overwrittenCount) の場合は最初の if に引っかかる)
        m_Data.header.startIndex += (postCount - static_cast<uint16_t>(overwrittenCount));
        if (m_Data.header.startIndex >= MaxEventCount)
        {
            m_Data.header.startIndex -= MaxEventCount;
        }
    }
    // それ以外の場合は残りの数に注意して単純に startIndex を進める
    else
    {
        // (この時点では送信前に蓄えられているイベントも含んだ数が計算される)
        int availableCount = static_cast<int>(m_Data.header.lastIndex) - static_cast<int>(m_Data.header.startIndex) + 1;
        // (startIndex == lastIndex のとき合計1つで常に1以上の値になるため、0以下の場合はループしている)
        if (availableCount <= 0)
        {
            availableCount += MaxEventCount;
        }
        // 現在の残りの数が送信した数以下である場合はすべて送信済みとなる
        if (availableCount <= static_cast<int>(postCount))
        {
            m_Data.header.startIndex = m_Data.header.lastIndex;
            m_Data.header.isAnyRecorded = false;
        }
        else
        {
            // 現在位置を進める
            m_Data.header.startIndex += postCount;
            if (m_Data.header.startIndex >= MaxEventCount)
            {
                m_Data.header.startIndex -= MaxEventCount;
            }
            m_Data.header.isAnyRecorded = true;
        }
    }

    // ここではヘッダーのみ更新する
    PostWriteDataToFileNoLock(true, 0, 0);
}

void WatcherEventStorage::DiscardEventsSinceInitialize() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexData);

    if (m_Data.header.isAnyRecordedOnLoad)
    {
        // 初期化時点の「lastIndex」(= lastIndexOnLoad)まで巻き戻すとイベントを消すことになる
        // (ただし古いデータが上書きされている可能性があるので startIndex の位置はそのまま)
        m_Data.header.lastIndex = m_Data.header.lastIndexOnLoad;
    }
    else
    {
        // 初期化時点では何もなかったのですべて破棄するのみで良い
        m_Data.header.startIndex = 0;
        m_Data.header.lastIndex = 0;
        m_Data.header.isAnyRecorded = false;
    }
    // データが増えたわけではないのでヘッダーのみ更新する
    PostWriteDataToFileNoLock(true, 0, 0);
}

void WatcherEventStorage::DiscardAllEvents() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexData);

    // 何も記録されていなければ処理不要
    if (!m_Data.header.isAnyRecorded && !m_Data.header.isAnyRecordedOnLoad)
    {
        return;
    }

    m_Data.header.startIndex = 0;
    m_Data.header.lastIndex = 0;
    m_Data.header.lastIndexOnLoad = 0;
    m_Data.header.isAnyRecorded = false;
    m_Data.header.isAnyRecordedOnLoad = false;

    // データが増えたわけではないのでヘッダーのみ更新する
    PostWriteDataToFileNoLock(true, 0, 0);
}

void WatcherEventStorage::WriteDataToFile() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexData);
    WriteDataToFileNoLock(!m_WriteDataInfo.isWritingEventsNecessary, m_WriteDataInfo.writeStartIndex, m_WriteDataInfo.writeLastIndex);
    m_WriteDataInfo.isWritingEventsNecessary = false;
}

void WatcherEventStorage::ApplyStoredEventsToPlayState(PlayState* pPlayState) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexData);

    if (!m_Data.header.isAnyRecorded)
    {
        return;
    }

    int16_t index = m_Data.header.startIndex;
    while (NN_STATIC_CONDITION(true))
    {
        auto& data = m_Data.storage[index];
        pPlayState->UpdatePlayStateFromEvent(data);

        if (index == m_Data.header.lastIndex)
        {
            break;
        }
        ++index;
        if (index == MaxEventCount)
        {
            index = 0;
        }
    }
}

void WatcherEventStorage::AddManagedEvent(bool isAdd, const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    // Watcher機能無効状態の場合は記録しない
    // (g_pWatcher が使えないのでここで判定)
    if (!IsWatcherAvailable())
    {
        return;
    }
    // ペアコン機能無効の場合は記録しない
    if (g_pMain->GetSettingsManager().IsAllFeaturesDisabled())
    {
        return;
    }
    // サーバー連携が行われていない場合は記録しない
    if (!g_pWatcher->GetNetworkManager().IsPairingActive())
    {
        return;
    }

    nn::time::PosixTime timestamp;
    // 時刻が取得できない場合は記録しない
    if (!common::GetNetworkTime(&timestamp))
    {
        return;
    }

    NN_UTIL_LOCK_GUARD(m_MutexData);

    // 過去イベントをさかのぼり、同種かつ同一 applicationId のイベントがあれば追加しない
    // (ただし見つかったイベントよりも後(時系列的に新しい方向)に isAdd と逆のイベントがあれば追加する)
    if (m_Data.header.isAnyRecorded)
    {
        uint16_t index = m_Data.header.lastIndex;
        bool isFound = false;
        while (NN_STATIC_CONDITION(true))
        {
            const auto& data = m_Data.storage[index];
            // Add/Remove のイベントで applicationId が一致するものがあればそこで検索を止める
            // その上で、Add/Remove と isAdd が一致すれば「既にイベントがある」として扱う
            if (data.eventType == EventType::DidAddNewManagedApplication && data.payload.addManagedApplication.applicationId == applicationId)
            {
                if (isAdd)
                {
                    isFound = true;
                }
                break;
            }
            else if (data.eventType == EventType::DidRemoveManagedApplication && data.payload.removeManagedApplication.applicationId == applicationId)
            {
                if (!isAdd)
                {
                    isFound = true;
                }
                break;
            }

            if (index == m_Data.header.startIndex)
            {
                break;
            }
            if (index == 0)
            {
                index = MaxEventCount - 1;
            }
            else
            {
                --index;
            }
        }
        if (isFound)
        {
            return;
        }
    }
    // イベントが無かったので通常通り追加処理を行う
    nn::time::LocationName locationName = g_pWatcher->GetWatcherEventManager().GetTimeLocationName();
    EventData::Payload payload;
    if (isAdd)
    {
        payload.addManagedApplication.applicationId = applicationId;
        AddEventInternalNoLock(EventType::DidAddNewManagedApplication, timestamp, locationName, &payload);
    }
    else
    {
        payload.removeManagedApplication.applicationId = applicationId;
        AddEventInternalNoLock(EventType::DidRemoveManagedApplication, timestamp, locationName, &payload);
    }
}

void WatcherEventStorage::AddEventInternalNoLock(EventType eventType, const nn::time::PosixTime& timestamp, const nn::time::LocationName& locationName, const EventData::Payload* pPayload) NN_NOEXCEPT
{
    uint16_t insertPosition;
    if (m_Data.header.isAnyRecorded)
    {
        insertPosition = m_Data.header.lastIndex + 1;
        if (insertPosition == MaxEventCount)
        {
            insertPosition = 0;
        }
    }
    else
    {
        // 何も記録されていない場合は m_Data.header.startIndex を挿入位置とする
        insertPosition = m_Data.header.startIndex;
    }
    InsertEventInternalNoLock(insertPosition, eventType, timestamp, locationName, pPayload);
}

void WatcherEventStorage::InsertEventInternalNoLock(EventType eventType, const nn::time::PosixTime& timestamp, const nn::time::LocationName& locationName, const EventData::Payload* pPayload) NN_NOEXCEPT
{
    uint16_t insertPosition;
    if (m_Data.header.isAnyRecorded)
    {
        bool foundInsertPosition = false;
        insertPosition = m_Data.header.lastIndex + 1; // timestamp が最も遅い時刻の場合は末尾
        if (insertPosition == MaxEventCount)
        {
            insertPosition = 0;
        }
        for (auto i = m_Data.header.startIndex; ; ++i)
        {
            if (i == MaxEventCount)
            {
                i = 0;
            }
            if (m_Data.storage[i].timestamp > timestamp)
            {
                insertPosition = i;
                foundInsertPosition = true;
                break;
            }
            if (i == m_Data.header.lastIndex)
            {
                break;
            }
        }
        // insertPosition が先頭になり、すでに許容データ量いっぱいの
        // イベントが蓄えられている場合は、追加しようとしている
        // イベント自身があふれるので記録しない
        if (foundInsertPosition && insertPosition == m_Data.header.startIndex &&
            (m_Data.header.lastIndex < MaxEventCount - 1 ?
                m_Data.header.lastIndex + 1 == m_Data.header.startIndex : // lastIndexがstartIndexに追いつく手前かどうか
                m_Data.header.startIndex == 0) // 0 == lastIndex + 1 - MaxEventCount (lastIndex < MaxEventCount)
            )
        {
            NN_DETAIL_PCTL_TRACE("[pctl] InsertEventInternalNoLock: new event data will be dropped.\n");
            return;
        }
    }
    else
    {
        // 何も記録されていないので m_Data.header.startIndex から始める
        insertPosition = m_Data.header.startIndex;
    }

    InsertEventInternalNoLock(insertPosition, eventType, timestamp, locationName, pPayload);
}

void WatcherEventStorage::InsertEventInternalNoLock(uint16_t insertPosition, EventType eventType, const nn::time::PosixTime& timestamp, const nn::time::LocationName& locationName, const EventData::Payload* pPayload) NN_NOEXCEPT
{
    auto result = nn::time::LoadTimeZoneRule(&m_TimeZoneRule, locationName);
    if (result.IsFailure())
    {
        NN_DETAIL_PCTL_WARN("Failed to load time-zone rule for event: 0x%08lX\n", result.GetInnerValueForDebug());
        return;
    }
    // タイムゾーンの実際のオフセット値などを取得
    nn::time::CalendarAdditionalInfo calInfo;
    {
        nn::time::CalendarTime calTime;
        result = nn::time::ToCalendarTime(&calTime, &calInfo, timestamp, m_TimeZoneRule);
        if (result.IsFailure())
        {
            NN_DETAIL_PCTL_WARN("Failed to get calendar time for event: 0x%08lX\n", result.GetInnerValueForDebug());
            return;
        }
        NN_UNUSED(calTime);
    }

    EventData* pDataPosition;
    if (m_Data.header.isAnyRecorded)
    {
        ++m_Data.header.lastIndex;
        // 最大数まで到達したら先頭に戻る
        if (m_Data.header.lastIndex >= MaxEventCount)
        {
            m_Data.header.lastIndex = 0;
        }
        // lastIndex が startIndex に追いついたら startIndex を進める
        // (その分古いデータが消えることになる)
        if (m_Data.header.lastIndex == m_Data.header.startIndex)
        {
            ++m_Data.header.startIndex;
            // lastIndexOnLoad を超える = ロード時に残っていたイベントデータがすべて上書きされた
            // (startIndex が MaxEventCount のときは 0 に戻さずに計算した方が都合が良い)
            // ※ lastIndex > lastIndexOnLoad のときはまだ上書きされていない
            if (m_Data.header.isAnyRecordedOnLoad &&
                m_Data.header.lastIndex <= m_Data.header.lastIndexOnLoad &&
                m_Data.header.startIndex > m_Data.header.lastIndexOnLoad)
            {
                // すべて上書きされたのでフラグを落とす
                m_Data.header.isAnyRecordedOnLoad = false;
            }
            if (m_Data.header.startIndex == MaxEventCount)
            {
                m_Data.header.startIndex = 0;
            }
        }
    }
    else
    {
        // 何も記録されていないので m_Data.header.startIndex から始める
        m_Data.header.lastIndex = m_Data.header.startIndex;
    }

    if (insertPosition == m_Data.header.lastIndex)
    {
        // 何もしない
    }
    else if (insertPosition < m_Data.header.lastIndex)
    {
        std::memmove(&m_Data.storage[insertPosition + 1], &m_Data.storage[insertPosition],
            sizeof(EventData) * (m_Data.header.lastIndex - insertPosition));
    }
    else
    {
        if (m_Data.header.lastIndex > 0)
        {
            std::memmove(&m_Data.storage[1], &m_Data.storage[0],
                sizeof(EventData) * m_Data.header.lastIndex);
        }
        std::memmove(&m_Data.storage[0], &m_Data.storage[MaxEventCount - 1],
            sizeof(EventData));
        std::memmove(&m_Data.storage[insertPosition + 1], &m_Data.storage[insertPosition],
            sizeof(EventData) * (MaxEventCount - 1 - insertPosition));
    }
    pDataPosition = &m_Data.storage[insertPosition];
    ++m_TotalEventCountFromBoot;

    StoreEventData(pDataPosition, eventType, timestamp, locationName, calInfo, pPayload);

    NN_DETAIL_PCTL_TRACE("Pctl-Event inserted: type = %d, index = %d, last-index = %d, timestamp = %lld\n",
        static_cast<int>(eventType), static_cast<int>(insertPosition),
        static_cast<int>(m_Data.header.lastIndex), timestamp.value);

    m_Data.header.isAnyRecorded = true;
    // 挿入位置から末尾までが更新対象
    PostWriteDataToFileNoLock(false, insertPosition, m_Data.header.lastIndex);
}

void WatcherEventStorage::StoreEventData(EventData* pOutData, EventType eventType, const nn::time::PosixTime& timestamp, const nn::time::LocationName& locationName,
    const nn::time::CalendarAdditionalInfo& calInfo, const EventData::Payload* pPayload) NN_NOEXCEPT
{
    pOutData->timestamp = timestamp;
    pOutData->isDaylightSavingTime = calInfo.timeZone.isDaylightSavingTime;
    pOutData->timezoneOffsetSeconds = calInfo.timeZone.utcOffsetSeconds;
    NN_STATIC_ASSERT(
        (std::is_same<
            decltype(pOutData->timezoneName),
            std::remove_const<std::remove_reference<decltype(locationName)>::type>::type
        >::value)
    );
    std::memcpy(&pOutData->timezoneName, &locationName, sizeof(pOutData->timezoneName));
    pOutData->dataVersion = EventDataVersionCurrent;

    pOutData->eventType = eventType;
    if (pPayload != nullptr)
    {
        std::memcpy(&pOutData->payload, pPayload, sizeof(pOutData->payload));
    }
}

void WatcherEventStorage::LoadDataFromFile() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexData);
    size_t sizeRead = 0;
    {
        common::FileStream stream;
        auto result = common::FileSystem::OpenRead(&stream, EventDataFileName);
        NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result),
            "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
        if (result.IsSuccess())
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Read(&sizeRead, 0, &m_Data, sizeof(m_Data)));
        }
    }
    // 1. 少なくともヘッダーサイズ分は必要
    // 2. 最大では m_Data 分読み込まれる
    // 3, 4. ヘッダーデータ上のインデックスが問題ないか
    // 5. lastIndex が指す位置までは取得されている必要あり
    // (5 に 1 が含まれるが、3, 4, 5 は header を参照するため 1 に依存する)
    if (sizeRead >= sizeof(m_Data.header) &&
        sizeRead <= sizeof(m_Data) &&
        m_Data.header.startIndex < MaxEventCount &&
        m_Data.header.lastIndex < MaxEventCount &&
        sizeRead >= sizeof(m_Data.header) + sizeof(m_Data.storage[0]) * (m_Data.header.lastIndex + 1))
    {
        m_Data.header.lastIndexOnLoad = m_Data.header.lastIndex;
        m_Data.header.isAnyRecordedOnLoad = m_Data.header.isAnyRecorded;
        if (!m_Data.header.isAnyRecorded)
        {
            m_TotalEventCountFromBoot = 0;
        }
        else if (m_Data.header.lastIndex >= m_Data.header.startIndex)
        {
            m_TotalEventCountFromBoot = m_Data.header.lastIndex - m_Data.header.startIndex + 1;
        }
        else
        {
            m_TotalEventCountFromBoot = MaxEventCount - m_Data.header.startIndex + m_Data.header.lastIndex;
        }
    }
    else
    {
        m_Data.header.startIndex = 0;
        m_Data.header.lastIndex = 0;
        m_Data.header.lastIndexOnLoad = 0;
        m_Data.header.isAnyRecorded = false;
        m_Data.header.isAnyRecordedOnLoad = false;
        m_TotalEventCountFromBoot = 0;
    }

    /// FOR TEST
#if !defined(NN_SDK_BUILD_RELEASE)
    NN_DETAIL_PCTL_TRACE("[pctl] Event dump:\n");
    for (uint32_t u = m_Data.header.startIndex; u <= static_cast<uint32_t>(m_Data.header.lastIndex); ++u)
    {
        auto& e = m_Data.storage[u];
        if (e.eventType == EventType::DidUserOpen || e.eventType == EventType::DidUserClose)
        {
            auto& uid = e.payload.openedUser.uid;
            NN_DETAIL_PCTL_TRACE("  [% 3lu] type = %d, timestamp = %lld, uid = %08x_%08x_%08x_%08x\n",
                u, static_cast<int>(e.eventType), e.timestamp.value,
                static_cast<uint32_t>(uid._data[0] >> 32),
                static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull),
                static_cast<uint32_t>(uid._data[1] >> 32),
                static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull));
        }
        else if (e.eventType == EventType::DidApplicationLaunch || e.eventType == EventType::DidApplicationSuspend ||
            e.eventType == EventType::DidAddNewManagedApplication || e.eventType == EventType::DidRemoveManagedApplication ||
            e.eventType == EventType::DidApplicationPlay ||
            e.eventType == EventType::DidApplicationResume || e.eventType == EventType::DidApplicationTerminate ||
            e.eventType == EventType::DidApplicationDownloadStart)
        {
            NN_DETAIL_PCTL_TRACE("  [% 3lu] type = %d, timestamp = %lld, appId = 0x%016llX\n",
                u, static_cast<int>(e.eventType), e.timestamp.value,
                e.payload.applicationLaunch.applicationId.value);
        }
        else
        {
            NN_DETAIL_PCTL_TRACE("  [% 3lu] type = %d, timestamp = %lld\n",
                u, static_cast<int>(e.eventType), e.timestamp.value);
        }
    }
#endif
}

void WatcherEventStorage::PostWriteDataToFileNoLock(bool isHeaderOnly, uint16_t writeStartIndex, uint16_t writeLastIndex) NN_NOEXCEPT
{
    if (!isHeaderOnly)
    {
        // m_WriteDataInfo.isWritingEventsNecessary が false のときは
        // まだ writeStartIndex / writeLastIndex が設定されていない
        if (!m_WriteDataInfo.isWritingEventsNecessary)
        {
            m_WriteDataInfo.isWritingEventsNecessary = true;
            m_WriteDataInfo.writeStartIndex = writeStartIndex;
            m_WriteDataInfo.writeLastIndex = writeLastIndex;
        }
        else
        {
            // 原則として writeStartIndex から writeLastIndex の範囲の OR を取る
            // (それぞれ writeStartIndex > writeLastIndex の可能性もある)

            // 計算しやすくするため startIndex <= lastIndex になるように変換
            uint32_t tempOldLastIndex = static_cast<uint32_t>(m_WriteDataInfo.writeLastIndex);
            uint32_t tempNewLastIndex = static_cast<uint32_t>(writeLastIndex);
            if (m_WriteDataInfo.writeStartIndex > m_WriteDataInfo.writeLastIndex)
            {
                tempOldLastIndex += MaxEventCount;
            }
            if (writeStartIndex > writeLastIndex)
            {
                tempNewLastIndex += MaxEventCount;
            }

            // S = min(m_WriteDataInfo.writeStartIndex, writeStartIndex)
            // T = max(tempOldLastIndex, tempNewLastIndex)
            // m_WriteDataInfo.writeStartIndex = S
            // tempNewLastIndex = T
            // (S <= T はそのまま)
            if (m_WriteDataInfo.writeStartIndex > writeStartIndex)
            {
                m_WriteDataInfo.writeStartIndex = writeStartIndex;
            }
            if (tempNewLastIndex < tempOldLastIndex)
            {
                tempNewLastIndex = tempOldLastIndex;
            }
            // S + MaxEventCount <= T の場合は S から T までの個数が MaxEventCount になるように
            // S の値を進める
            if (tempNewLastIndex - m_WriteDataInfo.writeStartIndex >= MaxEventCount)
            {
                uint32_t tempWriteStartIndex = (tempNewLastIndex - MaxEventCount + 1);
                if (tempWriteStartIndex >= MaxEventCount) // 現実的には tempWriteStartIndex == MaxEventCount になる可能性がある
                {
                    tempWriteStartIndex -= MaxEventCount;
                }
                m_WriteDataInfo.writeStartIndex = static_cast<uint16_t>(tempWriteStartIndex);
            }
            // ここまで S <= T となるように T (← tempOldLastIndex または tempNewLastIndex)を補正していたので
            // 元の範囲に戻してストアする
            if (tempNewLastIndex >= MaxEventCount)
            {
                tempNewLastIndex -= MaxEventCount;
            }
            m_WriteDataInfo.writeLastIndex = static_cast<uint16_t>(tempNewLastIndex);
        }
    }
    if (IsWatcherAvailable())
    {
        g_pWatcher->GetWatcherEventManager().PostSerializeEvents();
    }
}

void WatcherEventStorage::WriteDataToFileNoLock(bool isHeaderOnly, uint16_t writeStartIndex, uint16_t writeLastIndex) NN_NOEXCEPT
{
    {
        common::FileStream stream;
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::OpenWrite(&stream, EventDataFileName, sizeof(m_Data)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Write(0, &m_Data.header, sizeof(m_Data.header)));
        // m_Data.header.isAnyRecorded が false の場合はデータなしなので
        // データ書き込み部分を省略する
        if (!isHeaderOnly && m_Data.header.isAnyRecorded)
        {
            if (writeStartIndex <= writeLastIndex)
            {
                // writeStartIndex から (writeLastIndex - writeStartIndex + 1) の要素分書き込む
                NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Write(
                    sizeof(m_Data.header) + sizeof(m_Data.storage[0]) * writeStartIndex,
                    &m_Data.storage[writeStartIndex],
                    sizeof(m_Data.storage[0]) * (writeLastIndex - writeStartIndex + 1)
                ));
            }
            else
            {
                // ループしている場合は writeStartIndex から末尾までと
                // 先頭から writeLastIndex までを書き込む
                NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Write(
                    sizeof(m_Data.header),
                    &m_Data.storage[0],
                    sizeof(m_Data.storage[0]) * (writeLastIndex + 1)
                ));
                NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Write(
                    sizeof(m_Data.header) + sizeof(m_Data.storage[0]) * writeStartIndex,
                    &m_Data.storage[0],
                    sizeof(m_Data.storage[0]) * (MaxEventCount - writeStartIndex)
                ));
            }
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Flush());
    }
    if (!m_IsCommitSkipped)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    }
}

}}}}}
