﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_WatcherEventManager.h>

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs/fs_Result.h>
#include <nn/nifm/nifm_ApiInternetConnectionStatus.h>
#include <nn/os/os_Tick.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ApplicationStateManager.h>
#include <nn/pctl/detail/service/pctl_PlayState.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_Constants.h>
#include <nn/pctl/detail/service/common/pctl_FileSystem.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/pctl/detail/service/common/pctl_TimeUtil.h>
#include <nn/pctl/detail/service/overlay/pctl_OverlaySender.h>
#include <nn/pctl/detail/service/watcher/pctl_DeviceUser.h>
#include <nn/pctl/detail/service/watcher/pctl_IntermittentOperation.h>
#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

// 定数定義
namespace
{
    static const char PlayTimerSettingsFileName[] = "pt.dat";
    static const char PlayStateFileName[] = "pstate.dat";

    // 残り時間を通知するタイミング(秒)
    // ※ 大きい順に並べる
    static const int16_t TimeForNotifyRemainings[] = {
        60 * 60,
        30 * 60,
        5 * 60,
        4 * 60,
        3 * 60,
        2 * 60,
        1 * 60,
        0
    };
    // 超過時間を通知するタイミング(秒)
    // ※ 小さい順に並べる
    static const int16_t TimeForNotifyExceedings[] = {
        15 * 60,
        30 * 60,
        60 * 60,
        90 * 60,
        120 * 60
    };
    // bedtime の終了時刻
    static const struct
    {
        uint8_t hour;
        uint8_t minute;
    } EndBedtimeValue = { 6, 0 };

    // 試行回数に応じた、設定取得を行うまでの遅延時間(秒)
    static const int16_t TimeForRetrieveSettings[] = {
        0,          // 1回目(即時)
        2 * 60,     // 2回目(2分)
        5 * 60,     // 3回目(5分)
        20 * 60,    // 4回目(20分)
        60 * 60,    // 5回目(1時間)
        3 * 60 * 60 // 6回目(3時間)
        // 7回目以降はもうリトライしない
    };

    int32_t GetPostEventIntervalSeconds() NN_NOEXCEPT
    {
        int32_t value = 0;
        auto size = nn::settings::fwdbg::GetSettingsItemValue(&value, sizeof(value), "pctl", "post_event_interval_seconds");
        NN_DETAIL_PCTL_TRACE("[pctl] GetPostEventIntervalSeconds(): setting value = %d\n", static_cast<int>(value));
        // 10秒未満は短すぎるため既定値を用いるようにする
        if (size != sizeof(value) || value < 10)
        {
            value = static_cast<int32_t>(TimeSpanForPostEvents);
        }
        NN_DETAIL_PCTL_TRACE("[pctl] GetPostEventIntervalSeconds(): using value = %d\n", static_cast<int>(value));
        return value;
    }
}

// 補助関数
namespace
{
    #define IS_SAME_ENUM_VALUE(v1, v2) (static_cast<int>((v1)) == static_cast<int>((v2)))
    NN_STATIC_ASSERT(
        IS_SAME_ENUM_VALUE(Week::Week_Sunday, nn::time::DayOfWeek::DayOfWeek_Sunday) &&
        IS_SAME_ENUM_VALUE(Week::Week_Monday, nn::time::DayOfWeek::DayOfWeek_Monday) &&
        IS_SAME_ENUM_VALUE(Week::Week_Tuesday, nn::time::DayOfWeek::DayOfWeek_Tuesday) &&
        IS_SAME_ENUM_VALUE(Week::Week_Wednesday, nn::time::DayOfWeek::DayOfWeek_Wednesday) &&
        IS_SAME_ENUM_VALUE(Week::Week_Thursday, nn::time::DayOfWeek::DayOfWeek_Thursday) &&
        IS_SAME_ENUM_VALUE(Week::Week_Friday, nn::time::DayOfWeek::DayOfWeek_Friday) &&
        IS_SAME_ENUM_VALUE(Week::Week_Saturday, nn::time::DayOfWeek::DayOfWeek_Saturday)
        );
    #undef IS_SAME_ENUM_VALUE

    inline bool operator == (const PlayTimerDaySettings& lhs, const PlayTimerDaySettings& rhs) NN_NOEXCEPT
    {
        return (
            lhs.isBedtimeEnabled == rhs.isBedtimeEnabled &&
            lhs.isLimitTimeEnabled == rhs.isLimitTimeEnabled &&
            (!lhs.isBedtimeEnabled || (lhs.bedtimeHour == rhs.bedtimeHour && lhs.bedtimeMinute == rhs.bedtimeMinute)) &&
            (!lhs.isLimitTimeEnabled || (lhs.limitTime == rhs.limitTime))
            );
    }
    inline bool operator != (const PlayTimerDaySettings& lhs, const PlayTimerDaySettings& rhs) NN_NOEXCEPT
    {
        return !(operator == (lhs, rhs));
    }
    inline Week GetPreviousDayOfWeek(Week dayOfWeek) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(static_cast<int>(Week::Week_Sunday) == 0);
        NN_STATIC_ASSERT(static_cast<int>(Week::Week_Monday) == 1);
        NN_STATIC_ASSERT(static_cast<int>(Week::Week_Tuesday) == 2);
        NN_STATIC_ASSERT(static_cast<int>(Week::Week_Wednesday) == 3);
        NN_STATIC_ASSERT(static_cast<int>(Week::Week_Thursday) == 4);
        NN_STATIC_ASSERT(static_cast<int>(Week::Week_Friday) == 5);
        NN_STATIC_ASSERT(static_cast<int>(Week::Week_Saturday) == 6);
        return dayOfWeek == Week::Week_Sunday ? Week::Week_Saturday : (static_cast<Week>(static_cast<int>(dayOfWeek) - 1));
    }
}

WatcherEventManager::WatcherEventManager(void* threadStackBuffer, size_t stackBufferSize) NN_NOEXCEPT :
    m_TaskThread(),
    m_MultiWait(),
    m_EventHolderExit(),
    m_EventHolderRefreshTimer(),
    m_EventHolderUpdateTimerSettings(),
    m_TimerEventNextCheck(),
    m_TimerEventPostEvents(),
    m_TimerEventPostRemainingEvents(),
    m_TimerEventNotify(),
    m_EventHolderNotifyExceeded(),
    m_TimerEventPreSuspend(),
    m_TimerEventSuspend(),
    m_TimerEventDayChanged(),
    m_TimerEventBedtimeDayChanged(),
    m_EventLocationNameChanged(),
    m_EventHolderSaveDeviceEvents(),
    m_EventHolderSaveTimerStatus(),
    m_EventHolderPowerAwake(),
    m_TimerEventRetrieveSettings(),
    m_EventRetrieveSettingsFromNpns(),
    m_EventHolderStartIntermittentTask(),
    m_EventHolderSavePlayState(),
    m_EventHolderSaveDeviceUserData(),
    m_SysEventSynchronization(nn::os::EventClearMode::EventClearMode_AutoClear, true),
    m_SysEventPinCodeChanged(nn::os::EventClearMode::EventClearMode_AutoClear, true),
    m_SysEventSuspend(nn::os::EventClearMode::EventClearMode_AutoClear, true),
    m_LastPlayTimerStatusUpdated(),
    m_TimeSecondsValueForLastNotification(nn::util::nullopt),
    m_TimeSecondsValueForNextNotification(0),
    m_TimeSecondsValueForExceededNotification(0),
    m_TryCountForRetrieveSettings(0),
    m_IsLastNetworkTimeAvailable(false),
    m_IsTimeExceedNotified(false),
    m_IsSuspendEventTriggered(false),
    m_IsSuspended(false),
    m_RefreshTimerPattern(RefreshTimerPattern::Normal),
    m_IsTimerOnceRun(false),
    m_IsTimerStartedManually(false),
    m_IsThreadRunning(false),
    m_IsOnline(false),
    m_IsSleeping(false)
{
    std::memset(&m_PlayTimerStatus, 0, sizeof(m_PlayTimerStatus));

    nn::os::InitializeMultiWait(&m_MultiWait);

    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderExit.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderRefreshTimer.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderUpdateTimerSettings.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TimerEventNextCheck.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TimerEventPostEvents.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TimerEventPostRemainingEvents.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TimerEventNotify.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderNotifyExceeded.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TimerEventPreSuspend.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TimerEventSuspend.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TimerEventDayChanged.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TimerEventBedtimeDayChanged.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventLocationNameChanged.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderSaveDeviceEvents.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderSaveTimerStatus.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderPowerAwake.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_TimerEventRetrieveSettings.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventRetrieveSettingsFromNpns.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderStartIntermittentTask.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderSavePlayState.m_WaitHolder);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderSaveDeviceUserData.m_WaitHolder);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_TaskThread, StaticThreadProc, this,
        threadStackBuffer, stackBufferSize, NN_SYSTEM_THREAD_PRIORITY(pctl, WatcherEventManager)));
    nn::os::SetThreadNamePointer(&m_TaskThread, NN_SYSTEM_THREAD_NAME(pctl, WatcherEventManager));
}

WatcherEventManager::~WatcherEventManager() NN_NOEXCEPT
{
    StopThread();
    nn::os::DestroyThread(&m_TaskThread);

    nn::os::UnlinkAllMultiWaitHolder(&m_MultiWait);
    nn::os::FinalizeMultiWait(&m_MultiWait);
}

void WatcherEventManager::Initialize() NN_NOEXCEPT
{
    auto pTempPlayState = GetTempPlayStateData();

    LoadSettings();
    InitializePlayState(pTempPlayState);
    InitializeDeviceUserData();

    if (!g_pMain->GetSettingsManager().IsAllFeaturesDisabled() && g_pWatcher->GetNetworkManager().IsPairingActive())
    {
        // 蓄積済みのデータを pTempPlayState に反映させて時間を計算する
        g_pWatcher->GetWatcherEventStorage().ApplyStoredEventsToPlayState(pTempPlayState);

        OnDeviceLaunch(pTempPlayState);

        // 計算させた結果で情報を更新する(上書きするため timestamp を m_LastPlayTimerStatusUpdated にする)
        // ※ 必ず IgnorableTime 区間内になる
        if (m_IsTimerOnceRun)
        {
            ApplyPlayTimerStatus(true,
                m_LastPlayTimerStatusUpdated,
                pTempPlayState->lastElapsedTimeSeconds,
                pTempPlayState->playTimerStoppedTime);
        }
    }
}

void WatcherEventManager::StartThread() NN_NOEXCEPT
{
    nn::os::StartThread(&m_TaskThread);
    m_IsThreadRunning = true;
}

void WatcherEventManager::StopThread() NN_NOEXCEPT
{
    if (m_IsThreadRunning)
    {
        m_EventHolderExit.m_Event.Signal();
        nn::os::WaitThread(&m_TaskThread);
        m_IsThreadRunning = false;
    }
}

void WatcherEventManager::PostSavePlayState() NN_NOEXCEPT
{
    m_EventHolderSavePlayState.m_Event.Signal();
}

void WatcherEventManager::PostSaveDeviceUserData() NN_NOEXCEPT
{
    m_EventHolderSaveDeviceUserData.m_Event.Signal();
}

void WatcherEventManager::ClearDeviceUserData() NN_NOEXCEPT
{
    UpdateDeviceUserExecutor::ClearDeviceUserData();
}

void WatcherEventManager::ResetTimeStatus() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    nn::time::PosixTime now;
    if (common::GetNetworkTime(&now))
    {
        UpdateSpendTime(now);
    }

    // イベントをシグナルして再度 RefreshTimerEvents を呼び出させるようにする
    SignalRefreshTimerEventNoLock(RefreshTimerPattern::Normal);
}

void WatcherEventManager::RefreshTimeStatus() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    // イベントをシグナルして再度 RefreshTimerEvents を呼び出させるようにするだけ
    SignalRefreshTimerEventNoLock(RefreshTimerPattern::Normal);
}

bool WatcherEventManager::UpdatePlayTimerSettings(const PlayTimerSettings& settings) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    // 無効化状態であればタイマー設定の更新処理をしない
    if (g_pMain->GetSettingsManager().IsAllFeaturesDisabled())
    {
        return false;
    }

    NN_STATIC_ASSERT(
        (std::is_same<decltype(m_PlayTimerStatus.settings), std::remove_const< std::remove_reference<decltype(settings)>::type >::type >::value)
        );
    // 先にSignalしてこのイベントのハンドルフローに入るようにする
    m_EventHolderUpdateTimerSettings.m_Event.Signal();
    bool result = (
        m_PlayTimerStatus.settings.playTimerMode != settings.playTimerMode ||
        m_PlayTimerStatus.settings.isWeekSettingsUsed != settings.isWeekSettingsUsed ||
        m_PlayTimerStatus.settings.dailySettings != settings.dailySettings ||
        m_PlayTimerStatus.settings.weekSettings[Week_Sunday] != settings.weekSettings[Week_Sunday] ||
        m_PlayTimerStatus.settings.weekSettings[Week_Monday] != settings.weekSettings[Week_Monday] ||
        m_PlayTimerStatus.settings.weekSettings[Week_Tuesday] != settings.weekSettings[Week_Tuesday] ||
        m_PlayTimerStatus.settings.weekSettings[Week_Wednesday] != settings.weekSettings[Week_Wednesday] ||
        m_PlayTimerStatus.settings.weekSettings[Week_Thursday] != settings.weekSettings[Week_Thursday] ||
        m_PlayTimerStatus.settings.weekSettings[Week_Friday] != settings.weekSettings[Week_Friday] ||
        m_PlayTimerStatus.settings.weekSettings[Week_Saturday] != settings.weekSettings[Week_Saturday]
        );
    std::memcpy(&m_PlayTimerStatus.settings, &settings, sizeof(PlayTimerSettings));

    // 曜日チェック(ネットワーク時計が利用できない場合は isEnabled は false のままとしておく)
    m_PlayTimerStatus.settings.isEnabled = false;
    CheckDayEnabled(false); // 別途通知するのでここでは通知不要

    // タイマーの開始・停止状態は変わらないので OnTimerRunningStatusChanged は呼び出さない

    return result;
}

void WatcherEventManager::ClearSpentTime() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);

    // 経過時間に使う値を現在の時刻にセットし直す
    nn::time::PosixTime posixTime;
    if (common::GetNetworkTime(&posixTime))
    {
        nn::time::CalendarTime calTime;
        nn::time::CalendarAdditionalInfo calInfo;

        nn::time::ToCalendarTime(&calTime, &calInfo, posixTime);
        NN_UNUSED(calInfo);
        UpdateSpendTime(posixTime, posixTime, posixTime, calTime);
    }
    else
    {
        m_PlayTimerStatus.startTimeForSpend = nn::time::PosixTime();
        m_PlayTimerStatus.stopTimeForSpend = nn::time::PosixTime();
        m_LastPlayTimerStatusUpdated = nn::time::PosixTime();
        m_IsTimerOnceRun = false;
    }

    // 保存イベントを発行(遅延保存)
    m_EventHolderSaveTimerStatus.m_Event.Signal();
}

void WatcherEventManager::SetPlayTimerEnabled(bool isEnabled) NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        if (m_IsTimerStartedManually == isEnabled)
        {
            return;
        }
        // 無効化状態であればタイマー有効化処理をしない
        if (isEnabled && g_pMain->GetSettingsManager().IsAllFeaturesDisabled())
        {
            return;
        }
        m_IsTimerStartedManually = isEnabled;
        // 一時解除中は稼働させない
        auto temporaryUnlocked = g_pMain->GetStates().temporaryUnlocked;
        m_PlayTimerStatus.isTimerRunning = isEnabled && !temporaryUnlocked;

        // ロックされていなければ必ず isTimerRunning の値が変化しているので
        // 状態更新処理を行う
        if (!temporaryUnlocked)
        {
            OnTimerRunningStatusChanged(m_PlayTimerStatus.isTimerRunning);
        }

        // イベントをシグナルして再度 RefreshTimerEvents を呼び出させるようにする
        SignalRefreshTimerEventNoLock(RefreshTimerPattern::SettingChanged);
    }
    // 一時解除状態にかかわらず、プレイタイマー自信を開始・停止させたかでイベントを追加する
    // (Mutexのロックを解いた状態で実行する)
    if (isEnabled)
    {
        AddPlayTimerStartedEvent();
    }
    else
    {
        AddPlayTimerStoppedEvent();
    }
}

bool WatcherEventManager::IsPlayTimerEnabled() const NN_NOEXCEPT
{
    return m_PlayTimerStatus.isTimerRunning;
}

nn::Result WatcherEventManager::GetTimeToResetAlarmDisable(nn::time::PosixTime* outTime, const nn::time::PosixTime& now) NN_NOEXCEPT
{
    return common::GetStartTimeOfNextDate(outTime, now);
}

bool WatcherEventManager::IsAlarmDisabled() const NN_NOEXCEPT
{
    return m_PlayTimerStatus.alarmDisableExpirationTime.value != 0;
}

bool WatcherEventManager::SetAlarmDisabled(bool isDisabled, const nn::time::PosixTime& now) NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        nn::time::PosixTime time;
        nn::time::PosixTime expirationTime = {0};
        if (isDisabled)
        {
            if (!common::GetNetworkTime(&time))
            {
                // 時刻が取得できなかった場合は現在の状態を返す(非0 == 無効状態)
                return m_PlayTimerStatus.alarmDisableExpirationTime.value != 0;
            }

            nn::time::CalendarTime calTime;
            nn::time::ToCalendarTime(&calTime, nullptr, now);
            GetTimeToResetAlarmDisable(&expirationTime, now);
            // 指定された時刻が既に GetStartTimeOfNextDate で得られる時刻に到達していたら
            // 無効化して終わる
            if (expirationTime <= time)
            {
                isDisabled = false;
                expirationTime.value = 0;
            }
        }
        NN_DETAIL_PCTL_TRACE("[pctl] WatcherEventManager::SetAlarmDisabled: alarmDisableExpirationTime = %lld (now = %lld)\n",
            expirationTime.value, now.value);
        // 結果同じ設定になる場合は何もしない
        if (expirationTime == m_PlayTimerStatus.alarmDisableExpirationTime)
        {
            return isDisabled;
        }
        m_PlayTimerStatus.alarmDisableExpirationTime = expirationTime;
        // アラーム設定用の通知を発行
        nn::ovln::format::PctlAlarmSettingFlag flag = isDisabled ?
            nn::ovln::format::PctlAlarmSettingFlag_Disabled :
            nn::ovln::format::PctlAlarmSettingFlag_Enabled;
        overlay::NotifyAlarmSettingChanged(flag);
        m_EventHolderSaveTimerStatus.m_Event.Signal();

        // イベントをシグナルして RefreshTimerEvents を呼び出させるようにする
        // (設定変更のパターンで更新する)
        SignalRefreshTimerEventNoLock(RefreshTimerPattern::SettingChanged);

        // 状態変更をハンドルできるようにシステムイベントもSignalする
        m_SysEventSynchronization.Signal();
    }
    // (mutexがロックされていない状態で呼び出す)
    if (isDisabled)
    {
        AddAlarmDisabledEvent();
    }
    else
    {
        AddAlarmEnabledEvent();
    }
    return isDisabled;
}

nn::TimeSpan WatcherEventManager::GetRemainingTime(PlayTimerMode* outTimerMode) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);

    nn::TimeSpan tm = nn::TimeSpan(0);
    nn::time::PosixTime posixTime;
    // (ネットワーク時計が使用できない場合は 0 のままとする)
    if (common::GetNetworkTime(&posixTime))
    {
        GetRemainingTimeNoLock(&tm, outTimerMode, posixTime);
    }

    return tm;
}

void WatcherEventManager::GetPlayTimerStatus(PlayTimerStatus* outStatus) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    std::memcpy(outStatus, &m_PlayTimerStatus, sizeof(PlayTimerStatus));
}

void WatcherEventManager::GetPlayTimerSettings(PlayTimerSettings* outSettings) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    std::memcpy(outSettings, &m_PlayTimerStatus.settings, sizeof(PlayTimerSettings));
}

bool WatcherEventManager::IsRestrictedByPlayTimer() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);

    // SuspendEvent 発行のためのタイマー待機中は true を返す
    if (m_IsSuspendEventTriggered)
    {
        return true;
    }

    // 設定が無効の場合は false を返す
    if (!m_PlayTimerStatus.settings.isEnabled)
    {
        return false;
    }

    nn::time::PosixTime posixTime;
    // ネットワーク時計が使用できない場合は false とする
    if (!common::GetNetworkTime(&posixTime))
    {
        return false;
    }

    // アラーム無効状態の場合は false を返す(制限させない)
    if (m_PlayTimerStatus.alarmDisableExpirationTime.value != 0 &&
        posixTime < m_PlayTimerStatus.alarmDisableExpirationTime)
    {
        return false;
    }

    PlayTimerMode mode;
    nn::TimeSpan tm;
    if (!GetRemainingTimeNoLock(&tm, &mode, posixTime))
    {
        return false;
    }
    return mode == PlayTimerMode::PlayTimerMode_Suspend && tm <= nn::TimeSpan::FromSeconds(0);
}

void WatcherEventManager::UpdateUnlockRestrictionStatus() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    // 一時解除状態の場合はタイマーを動作させない
    auto isRunning = (m_IsTimerStartedManually && !g_pMain->GetStates().temporaryUnlocked);
    if (m_PlayTimerStatus.isTimerRunning != isRunning)
    {
        m_PlayTimerStatus.isTimerRunning = isRunning;
        OnTimerRunningStatusChanged(isRunning);
        SignalRefreshTimerEventNoLock(RefreshTimerPattern::SettingChanged);
    }
}

void WatcherEventManager::DisableEventProcess() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        // 先にSignalしてこのイベントのハンドルフローに入るようにする
        m_EventHolderUpdateTimerSettings.m_Event.Signal();
        // 設定がないものとして扱うため、現時点で有効である場合は有効状態が変わる処理を行う
        bool runningStatusChanged = IsPlayTimerEnabled();
        if (runningStatusChanged)
        {
            OnTimerRunningStatusChanged(false);
        }
        std::memset(&m_PlayTimerStatus, 0, sizeof(m_PlayTimerStatus));
        m_IsTimerStartedManually = false;
        m_SysEventSuspend.Clear();
        m_IsSuspendEventTriggered = false;
        m_TimerEventSuspend.m_Event.Stop();
        m_TimerEventSuspend.m_Event.Clear();

        m_TimerEventRetrieveSettings.m_Event.Stop();
        m_TimerEventRetrieveSettings.m_Event.Clear();

        m_TimerEventPostRemainingEvents.m_Event.Stop();
        m_TimerEventPostRemainingEvents.m_Event.Clear();
    }

    // スレッドを停止する
    StopThread();
}

nn::TimeSpan WatcherEventManager::GetSpentTime() const NN_NOEXCEPT
{
    nn::time::PosixTime posixTime;
    // ネットワーク時計が使用できない場合は 0 を返す
    if (!common::GetNetworkTime(&posixTime))
    {
        return nn::TimeSpan(0);
    }

    nn::time::CalendarTime calTime;
    nn::time::CalendarAdditionalInfo calInfo;
    nn::time::ToCalendarTime(&calTime, &calInfo, posixTime);

    nn::TimeSpan spentTime = nn::TimeSpan();

    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    CalculateSpentTime(&spentTime, calTime, posixTime);

    return spentTime;
}

void WatcherEventManager::WaitForEventsForTest() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        if (m_EventHolderExit.m_Event.TryWait())
        {
            break;
        }
        // m_MultiWait は複数スレッドで同時に使えないので
        // (ここではテスト用なので)個別にチェックする
        if (!m_EventHolderRefreshTimer.m_Event.TryWait() &&
            !m_EventHolderUpdateTimerSettings.m_Event.TryWait() &&
            !m_TimerEventNextCheck.m_Event.TryWait() &&
            !m_TimerEventNotify.m_Event.TryWait() &&
            !m_TimerEventPreSuspend.m_Event.TryWait() &&
            !m_TimerEventSuspend.m_Event.TryWait() &&
            !m_TimerEventDayChanged.m_Event.TryWait() &&
            !m_TimerEventBedtimeDayChanged.m_Event.TryWait() &&
            !m_EventLocationNameChanged.m_Event.TryWait())
        {
            // どれも非シグナルなので抜ける
            break;
        }
        // 1フレームほど待つ
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(17));
    }
}

void WatcherEventManager::HandlePowerStateEvent(PowerTransition transition) NN_NOEXCEPT
{
    // スレッドが動作していない場合は何もしない
    if (!m_IsThreadRunning)
    {
        return;
    }
    // NOTE: m_MutexMessage はロックされていない(原則ロックしない)
    switch (transition)
    {
        case PowerTransition_None:
            break;
        case PowerTransition_FullAwakeToMinimumAwake:
            // ※ スリープに入るための処理は同期的に行う
            {
                // タイマーを止める
                SetPlayTimerEnabled(false);
                AddSleepEvent();

                if (IsWatcherAvailable())
                {
                    g_pWatcher->GetNetworkManager().OnEnterSleep();
                }
            }
            m_IsSleeping = true;
            break;
        case PowerTransition_ToFullAwake:
            m_IsSleeping = false;
            m_EventHolderPowerAwake.m_Event.Signal();
            break;
        case PowerTransition_ToShutdown:
            // タイマーを止める
            SetPlayTimerEnabled(false);
            // シャットダウンイベントを記録
            g_pWatcher->GetWatcherEventStorage().AddShutdownEvent();
            // Shutdownでは遅延させずに保存を行う
            ProcessSerializeData();
            // (特にイベントはない)
            break;
        default:
            break;
    }
}

void WatcherEventManager::PostSerializeEvents() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    m_EventHolderSaveDeviceEvents.m_Event.Signal();
}

void WatcherEventManager::RequestRetrieveSettingsBackground(int tryCount, bool fromNpns) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    // 既にリトライ待ち中の場合は一旦リセットする
    m_TimerEventRetrieveSettings.m_Event.Stop();
    m_TimerEventRetrieveSettings.m_Event.Clear();
    m_EventRetrieveSettingsFromNpns.m_Event.Clear();

    if (tryCount < 1)
    {
        tryCount = 1;
    }
    else if (tryCount > static_cast<int>(std::extent<decltype(TimeForRetrieveSettings)>::value))
    {
        // 試行可能回数を超えている場合はリトライしない
        NN_DETAIL_PCTL_TRACE("[pctl] Exceeded retry count for retrieve settings (tryCount = %d)\n",
            tryCount);
        return;
    }

    auto timeSec = TimeForRetrieveSettings[tryCount - 1];
    m_TryCountForRetrieveSettings = tryCount;
    NN_DETAIL_PCTL_TRACE("[pctl] Request retrieve settings background (tryCount = %d, time = %d [sec])\n",
        tryCount, static_cast<int>(timeSec));
    if (timeSec == 0)
    {
        if (fromNpns)
        {
            // NPNSからの設定変更としてイベントをSignalする
            // (こちらのイベントは半起床中でも待ち受ける)
            m_EventRetrieveSettingsFromNpns.m_Event.Signal();
        }
        else
        {
            // 0秒の場合は即時Signal
            // (※ 半起床中は待ち受けない)
            m_TimerEventRetrieveSettings.m_Event.Signal();
        }
    }
    else
    {
        // 指定秒数遅延させる
        m_TimerEventRetrieveSettings.m_Event.StartOneShot(nn::TimeSpan::FromSeconds(static_cast<int64_t>(timeSec)));
    }
}

void WatcherEventManager::RequestPostRemainingEvents() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    m_TimerEventPostRemainingEvents.m_Event.StartOneShot(nn::TimeSpan::FromSeconds(static_cast<int64_t>(TimeSpanForPostRemainingEvents)));
}

void WatcherEventManager::ProcessPairingActivated(nn::time::PosixTime timeCurrent) NN_NOEXCEPT
{
    bool isPlayTimerEnabled;
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        isPlayTimerEnabled = m_IsTimerStartedManually;
    }
    // 現時点での状態に基づいてイベントデータを更新
    // (連携完了時点からのデータとする)
    // ※ 要:mutexの外で実行
    auto& storage = g_pWatcher->GetWatcherEventStorage();
    storage.InsertPairingActivatedEvent(timeCurrent);
    if (isPlayTimerEnabled)
    {
        storage.InsertPlayTimerStartedEvent(timeCurrent);
    }
    if (g_pMain->GetStates().temporaryUnlocked)
    {
        storage.InsertUnlockEvent(timeCurrent);
    }
    // 仕様上アプリケーションは必ず終了しているので以下の処理は不要
#if 0
    nn::ncm::ApplicationId applicationId;
    if (ApplicationStateManager::GetCurrentApplicationInfo(&applicationId))
    {
        storage.InsertApplicationLaunchEvent(applicationId, timeCurrent);
    }
    // MEMO: Openなユーザーの一覧を得て storage.InsertUserOpenedEvent を呼び出す
#endif
}

void WatcherEventManager::ProcessPairingDeleted() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);

    // プレイタイマーは連携中にしか使わないので消去
    ClearPlayTimerSettingsNoLock();
    // 設定同期のバックグラウンド待機を停止させる
    ClearWaitingForRetrieveSettingsBackgroundNoLock();
}

nn::Result WatcherEventManager::RequestStartIntermittentTask() NN_NOEXCEPT
{
    // 無効化中は無視してキャンセル扱いにする
    NN_RESULT_THROW_UNLESS(!m_EventHolderExit.m_Event.TryWait(), nn::pctl::ResultCanceled());
    // 未連携時は何もしない
    NN_RESULT_THROW_UNLESS(g_pWatcher->GetNetworkManager().IsPairingActive(), nn::pctl::ResultCanceled());

    NN_RESULT_DO(watcher::IntermittentOperation::NotifyTaskStarting());

    // 別スレッドで具体的なタスクを開始させ、呼び出し元にはすぐに戻るようにする
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        m_EventHolderStartIntermittentTask.m_Event.Signal();
    }

    NN_RESULT_SUCCESS;
}

void WatcherEventManager::CancelIntermittentTask() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        // シグナル済みだがハンドルされていない場合は、ClearしたうえでFinishを宣言する
        // (ハンドルされている場合は別途Finish宣言が行われる)
        if (m_EventHolderStartIntermittentTask.m_Event.TryWait())
        {
            m_EventHolderStartIntermittentTask.m_Event.Clear();
            watcher::IntermittentOperation::NotifyTaskFinished();
        }
    }
    g_pWatcher->GetNetworkManager().CancelIntermittentOperation();
}

void WatcherEventManager::ProcessSerializeData() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_TRACE("[pctl] WatcherEventManager::ProcessSerializeData()\n");
    // 5つとも内部で Mutex を利用
    OnSaveDeviceEvents();
    OnSaveDeviceUserData();
    OnSavePlayState();
    OnUpdatePlayTimerSettings(); // 必要に応じて OnSaveTimerStatus の処理が必要な状態にする
    OnSaveTimerStatus();
    NN_DETAIL_PCTL_TRACE("[pctl] WatcherEventManager::ProcessSerializeData() done.\n");
}

void WatcherEventManager::ProcessSerializeEventsIfNeededForTest() NN_NOEXCEPT
{
    OnSaveDeviceEvents();
}

void WatcherEventManager::ApplyPlayTimerStatus(bool isIgnorableTime,
    const nn::time::PosixTime& lastTimestampForElapsedTime,
    int64_t lastElapsedTimeSeconds,
    const nn::time::PosixTime& lastTimerStoppedTime) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    NN_DETAIL_PCTL_TRACE("[pctl] In WatcherEventManager::ApplyPlayTimerStatus\n");

    // 適用しようとしたデータが古い場合は何もしない
    if (m_LastPlayTimerStatusUpdated != nn::time::PosixTime() &&
        m_LastPlayTimerStatusUpdated > lastTimestampForElapsedTime)
    {
        NN_DETAIL_PCTL_TRACE("[pctl]  - data is old [%lld vs. %lld]\n",
            m_LastPlayTimerStatusUpdated.value, lastTimestampForElapsedTime.value);
        return;
    }

    // 引数のデータをもとに startTimeForSpend と stopTimeForSpend を計算する
    nn::time::PosixTime startTimeForSpend = (isIgnorableTime ? lastTimerStoppedTime : lastTimestampForElapsedTime)
        - nn::TimeSpan::FromSeconds(lastElapsedTimeSeconds);
    nn::time::PosixTime stopTimeForSpend = lastTimerStoppedTime;
    bool isChanged;
    if (isIgnorableTime)
    {
        // 時間計算を無視する期間にいる場合は差分が異なるかだけを見る
        isChanged = (stopTimeForSpend - startTimeForSpend != m_PlayTimerStatus.stopTimeForSpend - m_PlayTimerStatus.startTimeForSpend);
    }
    else
    {
        // 時間計算を行う期間にいる場合は開始点の違いを見る
        isChanged = (m_PlayTimerStatus.startTimeForSpend != startTimeForSpend);
    }

    if (isChanged)
    {
        NN_DETAIL_PCTL_TRACE("[pctl] Update times for spend: start = %lld -> %lld, stop = %lld -> %lld\n",
            m_PlayTimerStatus.startTimeForSpend.value, startTimeForSpend.value,
            m_PlayTimerStatus.stopTimeForSpend.value, stopTimeForSpend.value);
        m_PlayTimerStatus.startTimeForSpend = startTimeForSpend;
        m_PlayTimerStatus.stopTimeForSpend = stopTimeForSpend;

        // 保存イベントを発行(遅延保存)
        m_EventHolderSaveTimerStatus.m_Event.Signal();
        // 時刻が想定と変わっていた場合はイベントをシグナルして再度 RefreshTimerEvents を呼び出させるようにする
        // (強制中断タイミングのリセットも一部必要なので(日付が変わっている可能性もあるので)日付変更パターンで更新する)
        SignalRefreshTimerEventNoLock(RefreshTimerPattern::DayChanged);
        m_SysEventSynchronization.Signal();
    }
    else
    {
        NN_DETAIL_PCTL_TRACE("[pctl]  - data is same (start = %lld vs. %lld, stop = %lld vs. %lld)\n",
            m_PlayTimerStatus.startTimeForSpend.value, startTimeForSpend.value,
            m_PlayTimerStatus.stopTimeForSpend.value, stopTimeForSpend.value);
    }

    m_LastPlayTimerStatusUpdated = lastTimestampForElapsedTime;
}

void WatcherEventManager::PreparePostEvent() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        m_TimerEventPostEvents.m_Event.Clear();
        // OnPostEvents 時は残っていたイベントもまとめて送信されるので
        // m_TimerEventPostRemainingEvents のトリガーも不要
        m_TimerEventPostRemainingEvents.m_Event.Stop();
        m_TimerEventPostRemainingEvents.m_Event.Clear();

        // Post要求前に日付変更チェックを行う
        CheckDayChangedNoLock();
    }

    // 以降イベントを取り扱うが、その際 PostSerializeEvents が行われるので、
    // m_MutexMessage をロックし続けるとデッドロックになるため外側で実行

    ApplicationStateManager::RecordPlayDataBasedEvents();

    // 蓄えたデータの送信をトリガーする
    // ※ この処理は定期処理であるため、定期的にイベントを発生させることも兼ねて
    //    常にIdleイベントとPlayedイベントを積んでから実行する
    AddIdleEvent();
    AddApplicationPlayedEvent(); // 実行している場合のみイベントが積まれる
}

void WatcherEventManager::LoadSettings() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    // 機能制限無効化中はデータを読み込まない
    if (g_pMain->GetSettingsManager().IsAllFeaturesDisabled())
    {
        m_IsTimerOnceRun = false;
    }
    else
    // PlayTimerStatus の読み込み
    {
        m_IsTimerOnceRun = LoadSettings(&m_PlayTimerStatus);
    }
    if (m_IsTimerOnceRun)
    {
        // 初期状態は必ず false にする
        // もし true のまま終了していたら、今の時刻を停止時刻とする
        // (ただし日付を跨いでいたらリセットする)
        if (m_PlayTimerStatus.isTimerRunning)
        {
            if (!common::GetNetworkTime(&m_PlayTimerStatus.stopTimeForSpend))
            {
                // ネットワーク時計が取れなかったらリセットする
                // (他の時刻関連計算もできないので補正などは行わない)
                m_PlayTimerStatus.startTimeForSpend = m_PlayTimerStatus.stopTimeForSpend = nn::time::PosixTime();
                m_LastPlayTimerStatusUpdated = nn::time::PosixTime();
            }
            else
            {
                nn::time::CalendarTime calTime;
                nn::time::CalendarAdditionalInfo calInfo;

                nn::time::ToCalendarTime(&calTime, &calInfo, m_PlayTimerStatus.stopTimeForSpend);
                NN_UNUSED(calInfo);
                if (m_PlayTimerStatus.lastCheckDate.year != calTime.year ||
                    m_PlayTimerStatus.lastCheckDate.month != calTime.month ||
                    m_PlayTimerStatus.lastCheckDate.day != calTime.day)
                {
                    // 開始時刻を停止時刻に揃える
                    m_PlayTimerStatus.startTimeForSpend = m_PlayTimerStatus.stopTimeForSpend;
                }
                m_PlayTimerStatus.lastCheckDate = calTime;
                m_LastPlayTimerStatusUpdated = m_PlayTimerStatus.stopTimeForSpend;
            }
        }
        m_PlayTimerStatus.isTimerRunning = false;
        // イベント発行をトリガー
        g_pWatcher->GetNetworkManager().RequestPostEventsBackground();
    }
    else
    {
        // セーブデータが読み込めなかった時の初期値を設定
        nn::time::GetDeviceLocationName(&m_PlayTimerStatus.lastTimeLocationName);
    }

    InitializeTimerForPostEvents();
    RefreshTimerEvents(RefreshTimerPattern::SettingChanged);
    // 現状に対して保存イベントを発行(遅延保存)
    m_EventHolderSaveTimerStatus.m_Event.Signal();
}

bool WatcherEventManager::LoadSettings(PlayTimerStatus* outStatus) NN_NOEXCEPT
{
    PlayTimerStatus data;
    size_t size = 0;
    std::memset(&data, 0, sizeof(PlayTimerStatus));
    {
        common::FileStream stream;
        auto result = common::FileSystem::OpenRead(&stream, PlayTimerSettingsFileName);
        NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result),
            "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
        if (result.IsSuccess())
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Read(&size, 0, &data, sizeof(data)));
        }
    }
    if (size == sizeof(data))
    {
        NN_DETAIL_PCTL_TRACE("[pctl] PlayTimerStatusVer3 loaded.\n");
        std::memcpy(outStatus, &data, sizeof(PlayTimerStatus));
    }
    else if (size == sizeof(PlayTimerStatusVer1))
    {
        NN_DETAIL_PCTL_TRACE("[pctl] PlayTimerStatusVer1 loaded.\n");
        std::memcpy(outStatus, &data, sizeof(PlayTimerStatusVer1));
        // Ver1時点では存在しないデータの初期化
        outStatus->alarmDisableExpirationTime.value = 0;
    }
    else
    {
        NN_DETAIL_PCTL_TRACE("[pctl] Data size unknown for PlayTimerStatus. (size = %d)\n", static_cast<int>(size));
        return false;
    }
    // この値は初期値は常に false (データ管理の都合上セーブデータに入るのでリセットしておく)
    outStatus->settings.isEnabled = false;
    return true;
}

void WatcherEventManager::SaveSettings(const PlayTimerStatus& status) NN_NOEXCEPT
{
    {
        common::FileStream stream;
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::OpenWrite(&stream, PlayTimerSettingsFileName, sizeof(PlayTimerStatus)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Write(0, &status, sizeof(PlayTimerStatus)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Flush());
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
}

void WatcherEventManager::ClearPlayTimerSettingsFromSaveData() NN_NOEXCEPT
{
    if (IsWatcherAvailable())
    {
        // インスタンスの状態が更新できるようにインスタンスのメソッドを呼び出し
        auto& watcherManager = g_pWatcher->GetWatcherEventManager();

        // プレイタイマー設定を削除(ペアコン設定が無いと強制中断モードで中断を解除できなくなるため)
        watcherManager.ClearPlayTimerSettings();

        // 「一時解除したまま設定削除」した場合に状態変化がありうるため更新を実行しておく
        watcherManager.UpdateUnlockRestrictionStatus();
    }
    else
    {
        // セーブデータを直接変更する(データを読み込めたときだけ設定を書き換える)
        watcher::PlayTimerStatus status;
        if (LoadSettings(&status))
        {
            std::memset(&status.settings, 0, sizeof(status.settings));
            // 設定値に加えて以下の値もリセットする
            status.startTimeForSpend = nn::time::PosixTime();
            status.stopTimeForSpend = nn::time::PosixTime();
            status.alarmDisableExpirationTime = nn::time::PosixTime();
            SaveSettings(status);
        }
    }
}

void WatcherEventManager::InitializePlayState(PlayState* outPlayState) NN_NOEXCEPT
{
    auto holder = PlayStateHolder::Acquire();

    size_t size = 0;
    {
        common::FileStream stream;
        auto result = common::FileSystem::OpenRead(&stream, PlayStateFileName);
        NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result),
            "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
        if (result.IsSuccess())
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Read(&size, 0, holder.Get(), sizeof(*holder.Get())));
        }
    }
    if (size != sizeof(*holder.Get()))
    {
        holder.Get()->InitializeData();
    }
    std::memcpy(outPlayState, holder.Get(), sizeof(*holder.Get()));
}

void WatcherEventManager::InitializeDeviceUserData() NN_NOEXCEPT
{
    UpdateDeviceUserExecutor::InitializeAndLoadDeviceUserData();
}

void WatcherEventManager::InitializeTimerForPostEvents() NN_NOEXCEPT
{
    // 機能制限無効化中はイベントを発行しない
    if (g_pMain->GetSettingsManager().IsAllFeaturesDisabled())
    {
        return;
    }

    int32_t interval = GetPostEventIntervalSeconds();

    m_TimerEventPostEvents.m_Event.StartPeriodic(
        nn::TimeSpan::FromSeconds(interval),
        nn::TimeSpan::FromSeconds(interval)
    );
}

void WatcherEventManager::OnDeviceLaunch(PlayState* pTempPlayState) NN_NOEXCEPT
{
    nn::time::PosixTime now;
    // ネットワーク時刻が取得できなければ何もしない
    if (!common::GetNetworkTime(&now))
    {
        return;
    }

    // この時点で起動中の状態になっている場合はシャットダウンイベントがないことになる
    // → 強制電源断が起きている
    if (pTempPlayState->IsDeviceRunning())
    {
        nn::time::PosixTime lastEventTime = pTempPlayState->lastTimestamp;
        auto span = now - lastEventTime;
        const auto maxSpan = nn::TimeSpan::FromSeconds(watcher::MaxElapsedTimeSpanOnUnexpectedShutdown);
        if (span > maxSpan)
        {
            span = maxSpan;
        }
        // (DeviceLaunch のイベントの前に挿入する)
        AddUnexpectedShutdownOccurEvent(lastEventTime + span);
    }

    // Initialize はプロセス起動時に呼び出されるのでこのタイミングをデバイス起動タイミングとする
    AddDeviceLaunchEvent(now);
    {
        EventData data;
        auto r = g_pWatcher->GetWatcherEventStorage().GetLastEventData(&data);
        if (r && data.eventType == EventType::DidDeviceLaunch)
        {
            pTempPlayState->UpdatePlayStateFromEvent(data);
        }
    }
}

const PlayTimerDaySettings* WatcherEventManager::GetDaySettingsForBedtime(const nn::time::CalendarTime& calTimeNow, Week dayOfWeek) const NN_NOEXCEPT
{
    if (m_PlayTimerStatus.settings.isWeekSettingsUsed)
    {
        // 今の時刻が bedtime の終了時刻より前である場合は前日の設定を見る
        // (bedtime は EndBedtimeValue の時刻が日付の境界線になる)
        if (!(calTimeNow.hour > EndBedtimeValue.hour || (calTimeNow.hour == EndBedtimeValue.hour && calTimeNow.minute >= EndBedtimeValue.minute)))
        {
            return &m_PlayTimerStatus.settings.weekSettings[GetPreviousDayOfWeek(dayOfWeek)];
        }
        else
        {
            return &m_PlayTimerStatus.settings.weekSettings[dayOfWeek];
        }
    }
    else
    {
        return &m_PlayTimerStatus.settings.dailySettings;
    }
}

const PlayTimerDaySettings* WatcherEventManager::GetDaySettingsForLimitTime(Week dayOfWeek) const NN_NOEXCEPT
{
    if (m_PlayTimerStatus.settings.isWeekSettingsUsed)
    {
        // limitTime は 0 時が境界線になるので dayOfWeek をそのまま使う
        return &m_PlayTimerStatus.settings.weekSettings[dayOfWeek];
    }
    else
    {
        return &m_PlayTimerStatus.settings.dailySettings;
    }
}

bool WatcherEventManager::GetRemainingTimeNoLock(nn::TimeSpan* outRemainingTime, PlayTimerMode* outTimerMode, nn::time::PosixTime now) const NN_NOEXCEPT
{
    if (outTimerMode != nullptr)
    {
        *outTimerMode = m_PlayTimerStatus.settings.playTimerMode;
    }

    nn::time::CalendarTime calTime;
    nn::time::CalendarAdditionalInfo calInfo;
    nn::time::ToCalendarTime(&calTime, &calInfo, now);

    nn::TimeSpan spentTime;
    CalculateSpentTime(&spentTime, calTime, now);

    nn::time::PosixTime limitTime;
    // nn::time::DayOfWeek → Week への変換可否はソースコード冒頭のstatic assertでチェック済み
    if (!GetNextLimitTime(&limitTime, now, calTime,
        static_cast<Week>(calInfo.dayOfWeek), spentTime))
    {
        // 制限時刻がないので 0 を返す
        *outRemainingTime = nn::TimeSpan(0);
        return false;
    }

    *outRemainingTime = (limitTime - now);
    return true;
}

void WatcherEventManager::ClearPlayTimerSettingsNoLock() NN_NOEXCEPT
{
    // 設定値を0クリアする
    std::memset(&m_PlayTimerStatus.settings, 0, sizeof(m_PlayTimerStatus.settings));
    m_PlayTimerStatus.alarmDisableExpirationTime.value = 0;
    m_IsTimeExceedNotified = false;
    // 先にSignalしてこのイベントのハンドルフローに入るようにする
    m_EventHolderUpdateTimerSettings.m_Event.Signal();
}

void WatcherEventManager::RefreshTimerEvents(RefreshTimerPattern refreshTimerPattern) NN_NOEXCEPT
{
    m_TimerEventNotify.m_Event.Stop();
    m_TimerEventDayChanged.m_Event.Stop();
    m_TimerEventBedtimeDayChanged.m_Event.Stop();
    if (refreshTimerPattern != RefreshTimerPattern::Normal)
    {
        m_TimerEventNextCheck.m_Event.Stop();
        m_TimerEventPreSuspend.m_Event.Stop();
    }
    m_TimeSecondsValueForNextNotification = 0;

    // 無効化状態であればタイマーイベントの更新処理をしない
    if (g_pMain->GetSettingsManager().IsAllFeaturesDisabled())
    {
        return;
    }

    if (refreshTimerPattern != RefreshTimerPattern::Normal)
    {
        m_TimerEventNextCheck.m_Event.StartPeriodic(nn::TimeSpan::FromSeconds(TimeSpanForCheck), nn::TimeSpan::FromSeconds(TimeSpanForCheck));
    }

    nn::time::PosixTime posixTime;
    // ネットワーク時計が使用できない場合は再チェックさせるだけとする
    if (!common::GetNetworkTime(&posixTime))
    {
        if (m_IsLastNetworkTimeAvailable)
        {
            NN_DETAIL_PCTL_TRACE("[pctl] RefreshTimerEvents: Network time not available.\n");
            m_IsLastNetworkTimeAvailable = false;
            // MEMO: ここでは m_IsTimerOnceRun は false にリセットしない。
            //   以前ネットワーク時計が利用できていたので、その情報を開始時刻などに残しておけば
            //   数日後に利用可能になった際に経過時間計算上は
            //   「前回ストップ時から数日開いている」扱いにできる
            //   (OnTimerRunningStatusChanged() 内の処理)
        }
        return;
    }
    nn::time::CalendarTime calTime;
    nn::time::CalendarAdditionalInfo calInfo;
    nn::time::ToCalendarTime(&calTime, &calInfo, posixTime);

    if (!m_IsLastNetworkTimeAvailable)
    {
        NN_DETAIL_PCTL_TRACE("[pctl] RefreshTimerEvents: Network time available now (again).\n");
        m_IsLastNetworkTimeAvailable = true;
        // タイマー開始がトリガーされてからネットワーク時計が利用できるようになった場合、
        // ネットワーク時計利用不可の際にタイマー開始時処理を省略しているため
        // 改めてタイマーが開始されたときの処理を行う
        // (これによりタイマー開始時刻などの計算が行われるようになる)
        if (IsPlayTimerEnabled())
        {
            OnTimerRunningStatusChanged(true);
        }
    }

    // 既に期限が切れていたら無効化する
    if (m_PlayTimerStatus.alarmDisableExpirationTime.value != 0 &&
        m_PlayTimerStatus.alarmDisableExpirationTime <= posixTime)
    {
        m_PlayTimerStatus.alarmDisableExpirationTime.value = 0;
        m_SysEventSynchronization.Signal();
    }

    // OnLocationNameChanged からもタイマーイベント更新がトリガーされるが、
    // その場合 CheckLocationNameChanged 内での判定は「変更なし」に
    // なる場合がほとんどであるため、特別対応は行わない
    CheckLocationNameChanged();

    // 曜日の再チェック
    CheckDayEnabled(posixTime, calTime, static_cast<Week>(calInfo.dayOfWeek), true);

    {
        nn::time::PosixTime timeDayChange;
        common::GetStartTimeOfNextDate(&timeDayChange, posixTime);
        NN_SDK_ASSERT(timeDayChange > posixTime);
        // 日付変更イベントを設定
        m_TimerEventDayChanged.m_Event.StartOneShot(timeDayChange - posixTime);
    }
    {
        nn::time::PosixTime timeBedtimeDayChange;
        common::GetTimeOfNextClockPoint(&timeBedtimeDayChange, posixTime,
            static_cast<int>(EndBedtimeValue.hour), static_cast<int>(EndBedtimeValue.minute));
        NN_SDK_ASSERT(timeBedtimeDayChange > posixTime);
        // Bedtime用日付変更イベントを設定
        m_TimerEventBedtimeDayChanged.m_Event.StartOneShot(timeBedtimeDayChange - posixTime);
    }

    // 無効である場合はこれ以上何もしない
    if (!IsPlayTimerEnabled())
    {
        return;
    }

    // 時間制限関連のイベントは設定値が有効である時のみセットする
    if (m_PlayTimerStatus.settings.isEnabled)
    {
        // nn::time::DayOfWeek → Week への変換可否はソースコード冒頭のstatic assertでチェック済み
        SetNextTimerEventForLimit(posixTime, calTime, static_cast<Week>(calInfo.dayOfWeek), refreshTimerPattern);
    }
}

void WatcherEventManager::SetNextTimerEventForLimit(nn::time::PosixTime now, const nn::time::CalendarTime& calTime, Week dayOfWeek, RefreshTimerPattern refreshTimerPattern) NN_NOEXCEPT
{
    nn::TimeSpan spentTime = nn::TimeSpan();
    CalculateSpentTime(&spentTime, calTime, now);

    // 残り時間/超過時間を取得
    nn::TimeSpan timeSpan;
    {
        nn::time::PosixTime nextLimitTime;
        if (!GetNextLimitTime(&nextLimitTime, now, calTime, dayOfWeek, spentTime))
        {
            // 制限時間が取得できない(設定がない)場合は何もしない
            return;
        }
        timeSpan = nextLimitTime - now;
    }

    auto isTimeExceed = (timeSpan <= nn::TimeSpan(0));
    auto isAlarmDisabled = (m_PlayTimerStatus.alarmDisableExpirationTime.value != 0 && now < m_PlayTimerStatus.alarmDisableExpirationTime);
    auto isTimeExceedNotifiedPrevious = m_IsTimeExceedNotified;
    // イベント更新ごとに常に最新の状態を反映させる
    // (日付変更時やタイムゾーン変更時なども含む)
    // ※ アラーム無効状態のときは通知面では時間超過していない扱いとする
    m_IsTimeExceedNotified = (isTimeExceed && !isAlarmDisabled);

    // オーバーレイ通知が行われていない旨を設定 (m_TimeSecondsValueForLastNotification の無効化)
    auto lastNotifiedTime = m_TimeSecondsValueForLastNotification;
    m_TimeSecondsValueForLastNotification = nn::util::nullopt;

    // 無効化中は通知用のイベント設定をしない
    // (以降の処理はイベント通知の設定処理のみ)
    // ※ 通知の有無が変わっているので上記の m_TimeSecondsValueForLastNotification の
    //    変更は行われるようにする(ためにこの位置で抜ける)
    if (isAlarmDisabled)
    {
        // 強制中断していない状態に戻す
        m_IsSuspended = false;
        m_IsSuspendEventTriggered = false;
        m_TimerEventPreSuspend.m_Event.Stop();
        m_TimerEventPreSuspend.m_Event.Clear();
        m_TimerEventSuspend.m_Event.Stop();
        m_TimerEventSuspend.m_Event.Clear();
        return;
    }

    if (m_PlayTimerStatus.settings.playTimerMode == PlayTimerMode::PlayTimerMode_Suspend)
    {
        // 既に 0 である場合は即強制中断するかどうかをチェックする
        if (isTimeExceed) // 時間超過かどうかを見るため m_IsTimeExceedNotified を使わない(実際には同値)
        {
            NN_DETAIL_PCTL_TRACE("[pctl] No time on suspend mode. Not need to set timer event for time notification.\n");
            // 設定変更(→中断イベントの更新が必要)があり、
            // まだ強制中断していない場合は残り時間がないので即強制中断を発行する
            if (refreshTimerPattern != RefreshTimerPattern::Normal && (!m_IsSuspendEventTriggered && !m_IsSuspended))
            {
                m_TimerEventSuspend.m_Event.Stop();
                m_TimerEventSuspend.m_Event.Clear();
                if (refreshTimerPattern == RefreshTimerPattern::DayChanged)
                {
                    // 日付変更の場合は「残り時間 0」を通知し、一定の猶予を置いて強制中断する
                    NN_DETAIL_PCTL_TRACE("[pctl] ** Not suspended now and day changed, so trigger suspend with a few seconds.\n");
                    m_TimeSecondsValueForNextNotification = 0;
                    m_TimerEventNotify.m_Event.Signal();
                    m_TimerEventPreSuspend.m_Event.Signal();
                }
                else
                {
                    // それ以外(設定変更など)の場合は即強制中断とする
                    NN_DETAIL_PCTL_TRACE("[pctl] ** Not suspended now, so trigger suspend immediately.\n");
                    m_TimeSecondsValueForNextNotification = 0;
                    m_IsSuspendEventTriggered = true;
                    m_TimerEventSuspend.m_Event.Signal();
                }
            }
            // これ以上の処理(通知イベントタイマー設定など)は不要
            return;
        }
        // (設定変更などがない場合はイベントの再設定は不要)
        if (refreshTimerPattern != RefreshTimerPattern::Normal)
        {
            m_TimerEventPreSuspend.m_Event.StartOneShot(timeSpan);
            // 逆に中断イベントを発生させようとしているのであればそれは止める
            m_IsSuspendEventTriggered = false;
            m_TimerEventSuspend.m_Event.Stop();
            m_TimerEventSuspend.m_Event.Clear();
            // 残り時間もまだあるので強制中断していない扱いとする
            m_IsSuspended = false;
        }
    }
    else
    {
        // アラームモード中は強制中断していない状態に戻す
        // (アラームモードから強制中断モードに切り替わった際に強制中断をトリガーできるようにする)
        m_IsSuspended = false;
        m_IsSuspendEventTriggered = false;
        m_TimerEventPreSuspend.m_Event.Stop();
        m_TimerEventPreSuspend.m_Event.Clear();
        m_TimerEventSuspend.m_Event.Stop();
        m_TimerEventSuspend.m_Event.Clear();

        // 時間超過しており、まだ残り時間が 0 になっていなかったときは時間超過のイベントを発行する
        // (ただし、設定変更などを伴うイベント更新の場合のみとし、
        // 通常のカウントダウンによって時間が 0 になった場合はすでに
        // タイマーイベント処理で通知が発行されているので不要)
        if (refreshTimerPattern != RefreshTimerPattern::Normal &&
            m_IsTimeExceedNotified && !isTimeExceedNotifiedPrevious)
        {
            NN_DETAIL_PCTL_TRACE("[pctl] No time on alarm mode. Make event to notify remaining (exceeding) time.\n");
            m_TimeSecondsValueForExceededNotification = -timeSpan.GetSeconds(); // 超過時間として正の値にする
            m_EventHolderNotifyExceeded.m_Event.Signal();
            // 通常の通知イベント分も発行できるように return しない
            //return;
        }
    }

    SetNextNotifyTimeEvent(timeSpan, lastNotifiedTime);
}

void WatcherEventManager::SetNextNotifyTimeEvent(nn::TimeSpan timeSpan, const nn::util::optional<int16_t>& lastNotifiedTime) NN_NOEXCEPT
{
    int64_t timeSpanSeconds = timeSpan.GetSeconds();
    if (timeSpanSeconds > 0)
    {
        // 残り時間チェック
        for (auto& value : TimeForNotifyRemainings)
        {
            if (timeSpanSeconds >= value)
            {
                if (lastNotifiedTime == nn::util::nullopt || value != *lastNotifiedTime)
                {
                    // 残り時間 - 通知タイミング の値を次のイベント発生時刻にする
                    m_TimerEventNotify.m_Event.StartOneShot(nn::TimeSpan::FromSeconds(timeSpanSeconds - value));
                    m_TimeSecondsValueForNextNotification = value;
                    break;
                }
            }
        }
    }
    else
    {
        // 超過時間チェック
        for (auto& value : TimeForNotifyExceedings)
        {
            // ※ value は正数
            if (timeSpanSeconds >= -value)
            {
                if (lastNotifiedTime == nn::util::nullopt || -value != *lastNotifiedTime)
                {
                    // 超過時間(負数) - 通知タイミング の値を次のイベント発生時刻にする
                    m_TimerEventNotify.m_Event.StartOneShot(nn::TimeSpan::FromSeconds(timeSpanSeconds + value));
                    m_TimeSecondsValueForNextNotification = -value;
                    break;
                }
            }
        }
    }
}

void WatcherEventManager::CalculateSpentTime(nn::TimeSpan* outSpentTime, const nn::time::CalendarTime& calTime, nn::time::PosixTime now) const NN_NOEXCEPT
{
    // 過去に一度もタイマーが稼働していない場合は消費時間 0 とする
    if (!m_IsTimerOnceRun || (m_PlayTimerStatus.startTimeForSpend == nn::time::PosixTime() && m_PlayTimerStatus.stopTimeForSpend == nn::time::PosixTime()))
    {
        *outSpentTime = nn::TimeSpan();
        return;
    }

    // ローカル時計に基づいて日付部分のみ変化があるかチェックする
    bool dayChanged = (m_PlayTimerStatus.lastCheckDate.year != calTime.year ||
        m_PlayTimerStatus.lastCheckDate.month != calTime.month ||
        m_PlayTimerStatus.lastCheckDate.day != calTime.day);

    // 消費時間をタイマー開始時刻からの差分で計算
    nn::TimeSpan spentTime;
    if (IsPlayTimerEnabled())
    {
        // 動いている場合
        if (dayChanged)
        {
            // 「消費時間＝00:00からの経過時刻」なので単純に現在時刻を秒換算したものを設定する
            spentTime = nn::TimeSpan::FromSeconds(
                static_cast<int64_t>(calTime.hour) * 60 * 60 + static_cast<int64_t>(calTime.minute) * 60 + static_cast<int64_t>(calTime.second)
            );
        }
        else
        {
            // 現在の時刻を用いる
            spentTime = now - m_PlayTimerStatus.startTimeForSpend;
        }
    }
    else
    {
        // 止まっている場合
        if (dayChanged)
        {
            // 止まっているので消費時間は 0
            spentTime = nn::TimeSpan();
        }
        else
        {
            // 前回の停止時刻を用いる
            spentTime = m_PlayTimerStatus.stopTimeForSpend - m_PlayTimerStatus.startTimeForSpend;
        }
    }


    // 大幅に差分がある場合はひとまず 0 を返す
    if (nn::util::IsIntValueRepresentable<uint32_t>(spentTime.GetSeconds()))
    {
        *outSpentTime = spentTime;
    }
    else
    {
        *outSpentTime = 0;
    }
}

bool WatcherEventManager::GetNextLimitTime(nn::time::PosixTime* outLimitTime, nn::time::PosixTime now, const nn::time::CalendarTime& calTime, Week dayOfWeek, nn::TimeSpan spentTime) const NN_NOEXCEPT
{
    const PlayTimerDaySettings* pDaySettingForBedtime = GetDaySettingsForBedtime(calTime, dayOfWeek);
    const PlayTimerDaySettings* pDaySettingForLimitTime = GetDaySettingsForLimitTime(dayOfWeek);

    if (!pDaySettingForLimitTime->isLimitTimeEnabled && !pDaySettingForBedtime->isBedtimeEnabled)
    {
        return false;
    }

    nn::time::PosixTime limitTime = nn::time::PosixTime();
    if (pDaySettingForLimitTime->isLimitTimeEnabled)
    {
        nn::TimeSpan remainingTimeWithLimit = nn::TimeSpan::FromSeconds(static_cast<int>(pDaySettingForLimitTime->limitTime) * 60) - spentTime;
        limitTime = now + remainingTimeWithLimit;
    }
    if (pDaySettingForBedtime->isBedtimeEnabled)
    {
        nn::time::PosixTime timeStartBedtime;
        nn::time::PosixTime timeEndBedtime;
        // calTimeEndBedtime に日付時刻の値を得る
        common::GetTimeOfNextClockPoint(&timeEndBedtime, nullptr, now,
            static_cast<int>(EndBedtimeValue.hour), static_cast<int>(EndBedtimeValue.minute));
        // EndBedtime の手前の時刻を得る
        common::GetTimeOfPreviousClockPoint(&timeStartBedtime, timeEndBedtime,
            static_cast<int>(pDaySettingForBedtime->bedtimeHour), static_cast<int>(pDaySettingForBedtime->bedtimeMinute));
        // 結果、timeStartBedtime の時刻が制限時刻になる
        // (timeStartBedtime から timeEndBedtime までの期間が制限される期間)

        // 両方の制限が働く場合は制限が近い方を用いる
        if (!pDaySettingForBedtime->isLimitTimeEnabled || timeStartBedtime < limitTime)
        {
            limitTime = timeStartBedtime;
        }
    }
    *outLimitTime = limitTime;
    return true;
}

void WatcherEventManager::OnTimerRunningStatusChanged(bool isRunning) NN_NOEXCEPT
{
    NN_DETAIL_PCTL_TRACE("[pctl] Timer status changed: running = %s\n", isRunning ? "true" : "false");

    nn::time::PosixTime posixTime;
    // ネットワーク時計利用可否状態も更新する
    // ここで false になった場合は OnNextCheck → RefreshTimerEvents で再チェックが行われる
    m_IsLastNetworkTimeAvailable = common::GetNetworkTime(&posixTime);
    NN_DETAIL_PCTL_TRACE("[pctl]   network time available = %s\n", m_IsLastNetworkTimeAvailable ? "true" : "false");
    if (m_IsLastNetworkTimeAvailable)
    {
        nn::time::CalendarTime calTime;
        nn::time::CalendarAdditionalInfo calInfo;

        nn::time::ToCalendarTime(&calTime, &calInfo, posixTime);
        NN_UNUSED(calInfo);
        if (isRunning)
        {
            // Disable -> Enable の場合は startTimeForSpend を更新する
            if (m_IsTimerOnceRun)
            {
                // 前回の確認時刻と現在の時刻で日付が異なる場合は差分を取らずに現在の時刻を設定する
                if (calTime.year != m_PlayTimerStatus.lastCheckDate.year ||
                    calTime.month != m_PlayTimerStatus.lastCheckDate.month ||
                    calTime.day != m_PlayTimerStatus.lastCheckDate.day)
                {
                    UpdateSpendTime(posixTime, posixTime, posixTime, calTime);
                }
                else
                {
                    nn::time::PosixTime newStartTime;
                    //int64_t diff = posixTime.value - m_PlayTimerStatus.stopTimeForSpend.value;
                    newStartTime.value = m_PlayTimerStatus.startTimeForSpend.value +
                        (posixTime.value - m_PlayTimerStatus.stopTimeForSpend.value);
                    UpdateSpendTime(posixTime, newStartTime, posixTime, calTime);
                }
            }
            else
            {
                UpdateSpendTime(posixTime, posixTime, posixTime, calTime);
                m_IsTimerOnceRun = true;
            }
        }
        else
        {
            // 先に日付変更をチェックする
            // (nn::time::DayOfWeek → Week への変換可否はソースコード冒頭のstatic assertでチェック済み)
            CheckDayChangedNoLock(posixTime, calTime, static_cast<Week>(calInfo.dayOfWeek));

            // Enable -> Disable の場合は stopTimeForSpend を更新する
            UpdateSpendTime(posixTime, m_PlayTimerStatus.startTimeForSpend, posixTime, calTime);
            // 強制中断はタイマー稼動中にのみ発生するので、
            // 次にタイマーが稼働した際に発生できるようにここでフラグを落とす
            m_IsSuspended = false;
            m_IsSuspendEventTriggered = false;
            m_TimerEventSuspend.m_Event.Stop();
            m_TimerEventSuspend.m_Event.Clear();
            // 次に再開したときに(超過していれば)「残り時間 0」通知を出せるようにフラグを落とす
            m_IsTimeExceedNotified = false;
        }

        // 保存イベントを発行(遅延保存)
        m_EventHolderSaveTimerStatus.m_Event.Signal();
    }
}

void WatcherEventManager::UpdateSpendTime(nn::time::PosixTime now) NN_NOEXCEPT
{
    nn::time::CalendarTime calTime;
    nn::time::CalendarAdditionalInfo calInfo;

    nn::time::ToCalendarTime(&calTime, &calInfo, now);
    NN_UNUSED(calInfo);
    UpdateSpendTime(now, now, now, calTime);
}

void WatcherEventManager::UpdateSpendTime(nn::time::PosixTime now, nn::time::PosixTime startTime, nn::time::PosixTime stopTime, const nn::time::CalendarTime& calTimeChecked) NN_NOEXCEPT
{
    m_PlayTimerStatus.startTimeForSpend = startTime;
    m_PlayTimerStatus.stopTimeForSpend = stopTime;
    m_PlayTimerStatus.lastCheckDate = calTimeChecked;
    m_LastPlayTimerStatusUpdated = now;
}

bool WatcherEventManager::CheckLocationNameChanged() NN_NOEXCEPT
{
    nn::time::LocationName locationName;
    nn::time::GetDeviceLocationName(&locationName);
    // タイムゾーンが変わっていればそのイベントを発行しておく
    if (m_PlayTimerStatus.lastTimeLocationName != locationName)
    {
        std::memcpy(&m_PlayTimerStatus.lastTimeLocationName, &locationName, sizeof(locationName));
        m_EventLocationNameChanged.m_Event.Signal();
        return true;
    }
    return false;
}

void WatcherEventManager::CheckDeviceSettingsChanged() NN_NOEXCEPT
{
    bool isUpdateDeviceNecessary = CheckLocationNameChanged();

    // その他設定が変わっていればネットワークでの更新を要求しておく
    if (IsWatcherAvailable())
    {
        auto& manager = g_pWatcher->GetNetworkManager();
        if (manager.IsPairingActive())
        {
            if (manager.RefreshDeviceStatus())
            {
                isUpdateDeviceNecessary = true;
            }
            if (isUpdateDeviceNecessary)
            {
                manager.RequestUpdateDeviceBackground();
            }
        }
    }
}

void WatcherEventManager::CheckOnlineStatus() NN_NOEXCEPT
{
    nn::nifm::InternetConnectionStatus status;
    auto result = nn::nifm::GetInternetConnectionStatus(&status);
    bool isOnline = false;
    if (result.IsSuccess())
    {
        // 「インターネットに接続できる」をオンライン状態とする
        isOnline = (status.internetAvailability == nn::nifm::InternetAvailability_Confirmed);
    }

    if (m_IsOnline != isOnline)
    {
        // イベント発行(m_MutexMessage はロックしてはいけない)
        if (isOnline)
        {
            AddOnlineEvent();
        }
        else
        {
            AddOfflineEvent();
        }
        m_IsOnline = isOnline;
    }
}

void WatcherEventManager::CheckDayChangedNoLock() NN_NOEXCEPT
{
    nn::time::PosixTime posixTime;
    if (common::GetNetworkTime(&posixTime))
    {
        nn::time::CalendarTime calTime;
        nn::time::CalendarAdditionalInfo calInfo;
        nn::time::ToCalendarTime(&calTime, &calInfo, posixTime);
        // (nn::time::DayOfWeek → Week への変換可否はソースコード冒頭のstatic assertでチェック済み)
        CheckDayChangedNoLock(posixTime, calTime, static_cast<Week>(calInfo.dayOfWeek));
    }
}

void WatcherEventManager::CheckDayChangedNoLock(nn::time::PosixTime timeNow, const nn::time::CalendarTime& calTimeNow, Week dayOfWeek) NN_NOEXCEPT
{
    // 前回の確認時刻と現在の時刻で日付が異なる場合は、
    // 現在の日付の0時を開始時刻・終了時刻に設定する
    if (calTimeNow.year != m_PlayTimerStatus.lastCheckDate.year ||
        calTimeNow.month != m_PlayTimerStatus.lastCheckDate.month ||
        calTimeNow.day != m_PlayTimerStatus.lastCheckDate.day)
    {
        nn::time::PosixTime time = {};
        auto r = common::GetTimeOfPreviousClockPoint(&time, timeNow, 0, 0);
        // 取得に失敗した場合は現在時刻を用いる
        if (r.IsFailure())
        {
            time = timeNow;
        }
        // 開始時刻＝停止時刻とすることで、以下のどちらの場合にも対応
        // ・タイマー稼動中: 停止時刻は使われない(稼働中に日付変更→その日の0時から指定時刻までの経過時間が消費時間)
        // ・タイマー停止中: 開始時刻＝停止時刻なのでその日の消費時間が 0 になる
        UpdateSpendTime(time, time, time, calTimeNow);
        NN_DETAIL_PCTL_TRACE("[pctl] Day change detected. (date: %04d/%02d/%02d, diff sec. from 00:00 = %lld)\n",
            static_cast<int>(calTimeNow.year), static_cast<int>(calTimeNow.month), static_cast<int>(calTimeNow.day),
            (timeNow - time).GetSeconds());

        RefreshEventsForDayChangedNoLock(timeNow, calTimeNow, dayOfWeek);
    }
}

void WatcherEventManager::RefreshEventsForDayChangedNoLock(nn::time::PosixTime timeNow, const nn::time::CalendarTime& calTimeNow, Week dayOfWeek) NN_NOEXCEPT
{
    // 一時無効をリセットする
    m_PlayTimerStatus.alarmDisableExpirationTime.value = 0;

    // イベントをシグナルして再度 RefreshTimerEvents を呼び出させるようにする
    // (強制中断タイミングのリセットも一部必要なので日付変更パターンで更新する)
    SignalRefreshTimerEventNoLock(RefreshTimerPattern::DayChanged);

    // 曜日も合わせてチェックする
    // (通知は常に行うのでこのチェック内では不要)
    CheckDayEnabled(timeNow, calTimeNow, dayOfWeek, false);

    // 保存イベントを発行(遅延保存)
    m_EventHolderSaveTimerStatus.m_Event.Signal();

    // 日付変更により使う設定が変わるので設定変更を通知
    // (一時無効リセットの通知も含む)
    m_SysEventSynchronization.Signal();
}

// NOTE: m_MutexMessage はロックされた状態
void WatcherEventManager::CheckDayEnabled(bool notifyIfChanged) NN_NOEXCEPT
{
    nn::time::PosixTime posixTime;
    if (common::GetNetworkTime(&posixTime))
    {
        nn::time::CalendarTime calTime;
        nn::time::CalendarAdditionalInfo calInfo;
        nn::time::ToCalendarTime(&calTime, &calInfo, posixTime);
        // (nn::time::DayOfWeek → Week への変換可否はソースコード冒頭のstatic assertでチェック済み)
        CheckDayEnabled(posixTime, calTime, static_cast<Week>(calInfo.dayOfWeek), notifyIfChanged);
    }
    else // 利用できない場合は無効状態とする
    {
        if (m_PlayTimerStatus.settings.isEnabled)
        {
            m_PlayTimerStatus.settings.isEnabled = false;
            if (notifyIfChanged)
            {
                m_SysEventSynchronization.Signal();
            }
        }
    }
}

void WatcherEventManager::CheckDayEnabled(nn::time::PosixTime timeNow, const nn::time::CalendarTime& calTimeNow, Week dayOfWeek, bool notifyIfChanged) NN_NOEXCEPT
{
    // 曜日が変わったことで有効・無効が変わる場合、それを示すために設定変更であることを通知する
    bool isEnabled = false;
    bool isSignalNecessary = false;
    {
        const PlayTimerDaySettings* pDaySettingForBedtime = GetDaySettingsForBedtime(calTimeNow, dayOfWeek);
        const PlayTimerDaySettings* pDaySettingForLimitTime = GetDaySettingsForLimitTime(dayOfWeek);
        isEnabled = (pDaySettingForLimitTime->isLimitTimeEnabled || pDaySettingForBedtime->isBedtimeEnabled);
    }
    if (m_PlayTimerStatus.settings.isEnabled != isEnabled)
    {
        m_PlayTimerStatus.settings.isEnabled = isEnabled;
        isSignalNecessary = true;
    }

    // 強制中断状態が解除されるかどうかのチェック
    // プレイタイマーが停止している場合もあるので
    // タイマー稼動中が前提の SetNextTimerEventForLimit ではなくここで判定する
    if (m_PlayTimerStatus.settings.playTimerMode == PlayTimerMode::PlayTimerMode_Suspend)
    {
        nn::TimeSpan spentTime = nn::TimeSpan();
        nn::time::PosixTime limitTime = nn::time::PosixTime();
        CalculateSpentTime(&spentTime, calTimeNow, timeNow);
        if (GetNextLimitTime(&limitTime, timeNow, calTimeNow, dayOfWeek, spentTime))
        {
            if (m_IsSuspended && limitTime != nn::time::PosixTime())
            {
                NN_DETAIL_PCTL_TRACE("[pctl] Suspension status changed (to not suspended) by day-change.\n");
                m_IsSuspended = false;
                m_SysEventSuspend.Clear();
                // 強制中断状態から非強制中断状態になったことを設定変更として通知する
                isSignalNecessary = true;
            }
        }
    }

    if (isSignalNecessary && notifyIfChanged)
    {
        m_SysEventSynchronization.Signal();
    }
}

void WatcherEventManager::OnExit() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        m_EventHolderExit.m_Event.Clear();

        if (m_PlayTimerStatus.isTimerRunning)
        {
            // 必ず stop 状態で終わる(OnTimerRunningStatusChanged の内部で最終確認時刻も更新する)
            OnTimerRunningStatusChanged(false);
            m_PlayTimerStatus.isTimerRunning = false;
        }
    }

    // 保存してから終了する
    // (内部で Mutex を扱うので lock の外で実行)
    ProcessSerializeData();
}

void WatcherEventManager::OnRefreshTimer() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    NN_DETAIL_PCTL_TRACE("[pctl] PlayTimer: Refresh timer event\n");
    m_EventHolderRefreshTimer.m_Event.Clear();
    // RefreshTimerEvents を呼び出してタイマーを発生させる
    RefreshTimerEvents(m_RefreshTimerPattern);
    m_RefreshTimerPattern = RefreshTimerPattern::Normal;
}

void WatcherEventManager::OnUpdatePlayTimerSettings() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    if (m_EventHolderUpdateTimerSettings.m_Event.TryWait())
    {
        NN_DETAIL_PCTL_TRACE("[pctl] PlayTimer: Update timer settings event\n");
        m_EventHolderUpdateTimerSettings.m_Event.Clear();
        m_EventHolderSaveTimerStatus.m_Event.Signal();
        // 設定更新が起きているので RefreshTimerEvents を呼び出してタイマーイベントを更新させる
        // (ネットワーク時計利用可否もリフレッシュされる)
        RefreshTimerEvents(RefreshTimerPattern::SettingChanged);
    }
}

void WatcherEventManager::OnNextCheck() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        m_TimerEventNextCheck.m_Event.Clear();

        // 地域名とデバイス設定の変更チェックのみ行う
        CheckDeviceSettingsChanged();

        if (m_IsLastNetworkTimeAvailable)
        {
            // ネットワーク時計が有効な状態では RefreshTimerEvents を定期的に行う必要がない
        }
        else
        {
            // イベント更新を行う
            // (ネットワーク時計が利用できるようになればイベントがセットされる)
            // ※ このタイミングではデバイス設定更新チェックを行わなず、
            // ネットワーク時計が利用できるようになった後のチェックタイミングで行うようにする
            RefreshTimerEvents(RefreshTimerPattern::Normal);
        }
    }
    // オンライン状態のチェックを行う(Lock不要)
    CheckOnlineStatus();
}

void WatcherEventManager::OnPostEvents() NN_NOEXCEPT
{
    PreparePostEvent();
    g_pWatcher->GetNetworkManager().RequestPostEventsBackground();
}

void WatcherEventManager::OnPostRemainingEvents() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        m_TimerEventPostRemainingEvents.m_Event.Clear();

        // Post要求前に日付変更チェックを行う
        CheckDayChangedNoLock();
    }

    // 蓄えたデータの送信をトリガーする
    // m_MutexMessage をロックし続ける必要が無いので外側で実行
    g_pWatcher->GetNetworkManager().RequestPostEventsBackground();
}

void WatcherEventManager::OnNotifyToOverlay() NN_NOEXCEPT
{
    bool needsAddLimitTimeReachedEvent;
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        NN_DETAIL_PCTL_TRACE("[pctl] PlayTimer: Notify event; %d sec.\n", static_cast<int>(m_TimeSecondsValueForNextNotification));
        m_TimerEventNotify.m_Event.Clear();
        overlay::NotifyPlayTimerRemainingTime(m_TimeSecondsValueForNextNotification, overlay::ConvertModeFlag(m_PlayTimerStatus.settings.playTimerMode));
        m_TimeSecondsValueForLastNotification = m_TimeSecondsValueForNextNotification;
        // 0 のときはイベント通知を行う
        // (※ 強制中断モードでもアラームモードでもイベントを発生させる)
        needsAddLimitTimeReachedEvent = (m_TimeSecondsValueForNextNotification == 0);
        // イベント更新を行う
        RefreshTimerEvents(RefreshTimerPattern::Normal);
    }
    // (Mutexロックの外側で行う)
    if (needsAddLimitTimeReachedEvent)
    {
        AddLimitTimeReachedEvent();
    }
}

void WatcherEventManager::OnNotifyExceededToOverlay() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    NN_DETAIL_PCTL_TRACE("[pctl] PlayTimer: Notify exceeded event; %lld sec.\n", m_TimeSecondsValueForExceededNotification);
    m_EventHolderNotifyExceeded.m_Event.Clear();
    overlay::NotifyTimeExceededOnStart(m_TimeSecondsValueForExceededNotification, overlay::ConvertModeFlag(m_PlayTimerStatus.settings.playTimerMode));
    // AddLimitTimeReachedEvent は呼び出さない仕様
}

void WatcherEventManager::OnPreSuspend() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    NN_DETAIL_PCTL_TRACE("[pctl] PlayTimer: Pre-suspend event\n");
    m_TimerEventPreSuspend.m_Event.Clear();
    // 強制中断のためのタイマーイベントをトリガー
    m_TimerEventSuspend.m_Event.StartOneShot(nn::TimeSpan::FromSeconds(TimeSpanForSuspension));
    m_IsSuspendEventTriggered = true;
}

void WatcherEventManager::OnSuspend() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        // タイマー無効化などで中断を取り止めている場合は実行しない
        if (!m_IsSuspendEventTriggered)
        {
            return;
        }
        NN_DETAIL_PCTL_TRACE("[pctl] PlayTimer: Suspend event\n");
        m_TimerEventSuspend.m_Event.Clear();
        m_IsSuspendEventTriggered = false;
        m_IsSuspended = true;
        // 強制中断をトリガー
        m_SysEventSuspend.Signal();
    }
    // m_MutexMessage をロックし続けたまま呼び出すことができないので外側で実行
    AddTimerSuspensionEvent();
}

void WatcherEventManager::OnDayChanged() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_TRACE("[pctl] PlayTimer: Day changed event\n");
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    m_TimerEventDayChanged.m_Event.Clear();

    // 消費時間をリセットする
    nn::time::PosixTime posixTime;
    if (!common::GetNetworkTime(&posixTime))
    {
        posixTime.value = 0;
    }

    nn::time::CalendarTime calTime;
    nn::time::CalendarAdditionalInfo calInfo;
    nn::time::ToCalendarTime(&calTime, &calInfo, posixTime);
    UpdateSpendTime(posixTime, posixTime, posixTime, calTime);

    // (nn::time::DayOfWeek → Week への変換可否はソースコード冒頭のstatic assertでチェック済み)
    RefreshEventsForDayChangedNoLock(posixTime, calTime, static_cast<Week>(calInfo.dayOfWeek));
}

void WatcherEventManager::OnBedtimeDayChanged() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_TRACE("[pctl] PlayTimer: Bedtime day changed event\n");
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    m_TimerEventBedtimeDayChanged.m_Event.Clear();

    nn::time::PosixTime posixTime;
    if (!common::GetNetworkTime(&posixTime))
    {
        posixTime.value = 0;
    }

    nn::time::CalendarTime calTime;
    nn::time::CalendarAdditionalInfo calInfo;
    nn::time::ToCalendarTime(&calTime, &calInfo, posixTime);

    // 曜日チェック
    // (通知は常に行うのでこのチェック内では不要)
    CheckDayEnabled(posixTime, calTime, static_cast<Week>(calInfo.dayOfWeek), false);

    // イベントをシグナルして再度 RefreshTimerEvents を呼び出させるようにする
    // (強制中断タイミングのリセットも一部必要なので日付変更パターンで更新する)
    SignalRefreshTimerEventNoLock(RefreshTimerPattern::DayChanged);

    // 日付変更により使う設定が変わるので設定変更を通知
    m_SysEventSynchronization.Signal();
}

void WatcherEventManager::OnLocationNameChanged() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_TRACE("[pctl] PlayTimer: Location name changed event\n");
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        m_EventLocationNameChanged.m_Event.Clear();

        nn::time::PosixTime now;
        if (common::GetNetworkTime(&now))
        {
            // 消費時間をリセットする
            UpdateSpendTime(now);
        }

        // イベントをシグナルして再度 RefreshTimerEvents を呼び出させるようにする
        SignalRefreshTimerEventNoLock(RefreshTimerPattern::SettingChanged);

        // 保存イベントを発行(遅延保存)
        m_EventHolderSaveTimerStatus.m_Event.Signal();

        // 状態変更をハンドルできるようにシステムイベントもSignalする
        m_SysEventSynchronization.Signal();
    }

    // デバイス状態の更新を発行する
    if (IsWatcherAvailable())
    {
        auto& manager = g_pWatcher->GetNetworkManager();
        if (manager.IsPairingActive())
        {
            manager.RefreshDeviceStatus();
            manager.RequestUpdateDeviceBackground();
        }
    }

    // (mutexロックのない箇所で呼び出す)
    AddLocationNameChangedEvent();
}

void WatcherEventManager::OnSaveDeviceEvents() NN_NOEXCEPT
{
    bool isSignaled = false;

    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        if (m_EventHolderSaveDeviceEvents.m_Event.TryWait())
        {
            isSignaled = true;
            m_EventHolderSaveDeviceEvents.m_Event.Clear();
        }
    }
    if (isSignaled && IsWatcherAvailable())
    {
        NN_DETAIL_PCTL_TRACE("[pctl] WatcherEventManager::OnSaveDeviceEvents(): write data\n");
        g_pWatcher->GetWatcherEventStorage().WriteDataToFile();
    }
}

void WatcherEventManager::OnSaveTimerStatus() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexMessage);
    if (m_EventHolderSaveTimerStatus.m_Event.TryWait())
    {
        m_EventHolderSaveTimerStatus.m_Event.Clear();
        NN_DETAIL_PCTL_TRACE("[pctl] WatcherEventManager::OnSaveTimerStatus(): write data\n");
        // m_MutexMessage を握ったまま実行する
        SaveSettingsNoLock();
    }
}

void WatcherEventManager::OnPowerAwake() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        m_EventHolderPowerAwake.m_Event.Clear();
    }

    if (IsWatcherAvailable())
    {
        g_pWatcher->GetNetworkManager().OnAwakeFromSleep();
    }
    // (タイマー開始はメニューがトリガーする)
    AddWakeupEvent();
}

void WatcherEventManager::OnRetrieveSettingsEvents(bool fromNpns) NN_NOEXCEPT
{
    int tryCount;
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        m_TimerEventRetrieveSettings.m_Event.Stop();
        m_TimerEventRetrieveSettings.m_Event.Clear();
        m_EventRetrieveSettingsFromNpns.m_Event.Clear();
        tryCount = m_TryCountForRetrieveSettings;
    }
    if (tryCount <= 0)
    {
        // (本来は 1 以上が指定されるはず)
        NN_DETAIL_PCTL_WARN("[pctl] Unexpected try count: %d; use 1 instead\n", tryCount);
        tryCount = 1;
    }

    // 待機中に連携が解除されていた場合はこれ以上何もしない
    if (!g_pWatcher->GetNetworkManager().IsPairingActive())
    {
        NN_DETAIL_PCTL_TRACE("[pctl] OnRetrieveSettingsEvents: pairing is not active\n");
        return;
    }

    NN_DETAIL_PCTL_TRACE("[pctl] OnRetrieveSettingsEvents (tryCount = %d)\n", tryCount);
    auto& manager = g_pWatcher->GetNetworkManager();
    if (fromNpns && tryCount == 1 && IntermittentOperation::IsInHalfAwake())
    {
        // 半起床中でNPNSの通知をトリガーとした設定同期を行う場合は、
        // 単純な設定同期の代わりに間欠起動モードの処理を行う
        NN_DETAIL_PCTL_TRACE("[pctl] Half-awake mode; process intermittent operation instead of retrieve settings.\n");
        NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid status here");

        watcher::IntermittentOperation::SetSchedulingNecessary(true);
        auto result = RequestStartIntermittentTask();
        if (result.IsSuccess())
        {
            NN_DETAIL_PCTL_TRACE("[pctl] Intermittent operation started.\n");
            // (成功時はネットワークのBGタスクとして処理が継続される)
        }
        else
        {
            NN_DETAIL_PCTL_TRACE("[pctl] Failed to start intermittent operation: result = 0x%08lX\n", result.GetInnerValueForDebug());
        }
    }
    else
    {
        // このメソッド自体は ResultCanceled が返るが特にハンドルしない
        manager.RequestSynchronizeSettingsBackground(manager.GetSavedDeviceId(),
            false,
            tryCount,
            fromNpns);
    }
}

void WatcherEventManager::OnStartIntermittentTask() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        m_EventHolderStartIntermittentTask.m_Event.Clear();
    }

    auto result = g_pWatcher->GetNetworkManager().StartIntermittentOperation();
    if (result.IsSuccess())
    {
        NN_DETAIL_PCTL_TRACE("[pctl] Intermittent operation triggered.\n");
        // (成功時はネットワークのBGタスクとして別スレッドで処理が継続される)
    }
    else
    {
        if (nn::pctl::ResultCanceled::Includes(result))
        {
            // (ログが紛らわしくなるのでキャンセル時のログは特殊対応する)
            NN_DETAIL_PCTL_TRACE("[pctl] Intermittent operation is not necessary now. (ResultCanceled)\n");
        }
        else
        {
            NN_DETAIL_PCTL_TRACE("[pctl] Failed to trigger intermittent operation: result = 0x%08lX\n", result.GetInnerValueForDebug());
        }
        // 失敗の戻り値である場合はここで Finish を宣言する
        // 成功時は IntermittentOperation が完了またはキャンセルしたときに呼び出される
        watcher::IntermittentOperation::NotifyTaskFinished();
    }
}

void WatcherEventManager::OnSavePlayState() NN_NOEXCEPT
{
    bool isSignaled = false;

    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        if (m_EventHolderSavePlayState.m_Event.TryWait())
        {
            isSignaled = true;
            m_EventHolderSavePlayState.m_Event.Clear();
        }
    }

    if (isSignaled)
    {
        NN_DETAIL_PCTL_TRACE("[pctl] WatcherEventManager::OnSavePlayState(): write data\n");
        SavePlayState();
    }
}

void WatcherEventManager::OnSaveDeviceUserData() NN_NOEXCEPT
{
    bool isSignaled = false;

    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        if (m_EventHolderSaveDeviceUserData.m_Event.TryWait())
        {
            isSignaled = true;
            m_EventHolderSaveDeviceUserData.m_Event.Clear();
        }
    }

    if (isSignaled)
    {
        NN_DETAIL_PCTL_TRACE("[pctl] WatcherEventManager::OnSaveDeviceUserData(): write data\n");
        SaveDeviceUserData();
    }
}

void WatcherEventManager::SaveSettingsNoLock() NN_NOEXCEPT
{
    // 無効化中はセーブデータに保存しない
    if (g_pMain->GetSettingsManager().IsAllFeaturesDisabled())
    {
        return;
    }

    SaveSettings(m_PlayTimerStatus);
}

void WatcherEventManager::SavePlayState() NN_NOEXCEPT
{
    bool isIgnorableTime;
    nn::time::PosixTime lastTimestamp;
    int64_t lastElapsedTimeSeconds;
    nn::time::PosixTime playTimerStoppedTime;

    {
        auto holder = PlayStateHolder::Acquire();

        isIgnorableTime = holder->IsIgnorableTime();
        lastTimestamp = holder->lastTimestamp;
        lastElapsedTimeSeconds = holder->lastElapsedTimeSeconds;
        playTimerStoppedTime = holder->playTimerStoppedTime;

        {
            common::FileStream stream;
            NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::OpenWrite(&stream, PlayStateFileName, sizeof(*holder.Get())));
            NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Write(0, holder.Get(), sizeof(*holder.Get())));
            NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Flush());
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    }
    // プレイ状態のデータをプレイ時間の処理に適用する
    // (holder をリリースしてから行う)
    ApplyPlayTimerStatus(isIgnorableTime, lastTimestamp, lastElapsedTimeSeconds, playTimerStoppedTime);
}

void WatcherEventManager::SaveDeviceUserData() NN_NOEXCEPT
{
    UpdateDeviceUserExecutor::SaveDeviceUserData();
}


void WatcherEventManager::AddDeviceLaunchEvent(const nn::time::PosixTime& timestamp) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddDeviceLaunchEvent(timestamp);
}

void WatcherEventManager::AddTimerSuspensionEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddTimerSuspensionEvent();
}

void WatcherEventManager::AddSleepEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddSleepEvent();
}

void WatcherEventManager::AddWakeupEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddWakeupEvent();
}

void WatcherEventManager::AddOnlineEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddOnlineEvent();
}

void WatcherEventManager::AddOfflineEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddOfflineEvent();
}

void WatcherEventManager::AddIdleEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddIdleEvent();
}

void WatcherEventManager::AddApplicationPlayedEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");

    nn::ncm::ApplicationId applicationId;
    if (ApplicationStateManager::GetCurrentApplicationInfo(&applicationId))
    {
        g_pWatcher->GetWatcherEventStorage().AddApplicationPlayedEvent(applicationId);
    }
}

void WatcherEventManager::AddAlarmDisabledEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddAlarmDisabledEvent();
}

void WatcherEventManager::AddAlarmEnabledEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddAlarmEnabledEvent();
}

void WatcherEventManager::AddLimitTimeReachedEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddLimitTimeReachedEvent();
}

void WatcherEventManager::AddPlayTimerStartedEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddPlayTimerStartedEvent();
}

void WatcherEventManager::AddPlayTimerStoppedEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddPlayTimerStoppedEvent();
}

void WatcherEventManager::AddLocationNameChangedEvent() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddLocationNameChangedEvent();
}

void WatcherEventManager::AddUnexpectedShutdownOccurEvent(const nn::time::PosixTime& timeAt) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_MutexMessage.IsLockedByCurrentThread(), "Invalid add-event call");
    g_pWatcher->GetWatcherEventStorage().AddUnexpectedShutdownOccurEvent(timeAt);
}


void WatcherEventManager::ThreadProc() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_TRACE("[pctl] PlayTimer: Thread started.\n");

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::MultiWaitHolderType* pSignaled;

        if (m_IsSleeping)
        {
            // スリープ時は限られたイベントのみハンドルする
            auto r = nn::os::WaitAny(
                m_EventHolderExit.m_Event.GetBase(),
                m_EventRetrieveSettingsFromNpns.m_Event.GetBase(), // NPNSトリガーの設定取得はスリープ中も見る
                m_EventHolderStartIntermittentTask.m_Event.GetBase(),
                m_EventHolderPowerAwake.m_Event.GetBase() // スリープ復帰時に元に戻れるようにこのイベントも待機する
                );
            switch (r)
            {
                case 0:
                    pSignaled = &m_EventHolderExit.m_WaitHolder;
                    break;
                case 1:
                    pSignaled = &m_EventRetrieveSettingsFromNpns.m_WaitHolder;
                    break;
                case 2:
                    pSignaled = &m_EventHolderStartIntermittentTask.m_WaitHolder;
                    break;
                case 3:
                    pSignaled = &m_EventHolderPowerAwake.m_WaitHolder;
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
            }
        }
        else
        {
            pSignaled = nn::os::WaitAny(&m_MultiWait);
        }
        NN_SDK_ASSERT_NOT_NULL(pSignaled);

        if (pSignaled == &m_EventHolderExit.m_WaitHolder)
        {
            OnExit();
            return;
        }
        else if (pSignaled == &m_EventHolderRefreshTimer.m_WaitHolder)
        {
            OnRefreshTimer();
        }
        else if (pSignaled == &m_EventHolderUpdateTimerSettings.m_WaitHolder)
        {
            OnUpdatePlayTimerSettings();
        }
        else if (pSignaled == &m_TimerEventNextCheck.m_WaitHolder)
        {
            OnNextCheck();
        }
        else if (pSignaled == &m_TimerEventPostEvents.m_WaitHolder)
        {
            OnPostEvents();
        }
        else if (pSignaled == &m_TimerEventPostRemainingEvents.m_WaitHolder)
        {
            OnPostRemainingEvents();
        }
        else if (pSignaled == &m_TimerEventNotify.m_WaitHolder)
        {
            OnNotifyToOverlay();
        }
        else if (pSignaled == &m_EventHolderNotifyExceeded.m_WaitHolder)
        {
            OnNotifyExceededToOverlay();
        }
        else if (pSignaled == &m_TimerEventPreSuspend.m_WaitHolder)
        {
            OnPreSuspend();
        }
        else if (pSignaled == &m_TimerEventSuspend.m_WaitHolder)
        {
            OnSuspend();
        }
        else if (pSignaled == &m_TimerEventDayChanged.m_WaitHolder)
        {
            OnDayChanged();
        }
        else if (pSignaled == &m_TimerEventBedtimeDayChanged.m_WaitHolder)
        {
            OnBedtimeDayChanged();
        }
        else if (pSignaled == &m_EventLocationNameChanged.m_WaitHolder)
        {
            OnLocationNameChanged();
        }
        else if (pSignaled == &m_EventHolderSaveDeviceEvents.m_WaitHolder)
        {
            OnSaveDeviceEvents();
        }
        else if (pSignaled == &m_EventHolderSaveTimerStatus.m_WaitHolder)
        {
            OnSaveTimerStatus();
        }
        else if (pSignaled == &m_EventHolderPowerAwake.m_WaitHolder)
        {
            OnPowerAwake();
        }
        else if (pSignaled == &m_TimerEventRetrieveSettings.m_WaitHolder ||
            pSignaled == &m_EventRetrieveSettingsFromNpns.m_WaitHolder)
        {
            OnRetrieveSettingsEvents(pSignaled == &m_EventRetrieveSettingsFromNpns.m_WaitHolder);
        }
        else if (pSignaled == &m_EventHolderStartIntermittentTask.m_WaitHolder)
        {
            OnStartIntermittentTask();
        }
        else if (pSignaled == &m_EventHolderSavePlayState.m_WaitHolder)
        {
            OnSavePlayState();
        }
        else if (pSignaled == &m_EventHolderSaveDeviceUserData.m_WaitHolder)
        {
            OnSaveDeviceUserData();
        }
    }
    NN_DETAIL_PCTL_TRACE("[pctl] PlayTimer: Thread shutdown.\n");
} // NOLINT(impl/function_size)

}}}}}
