﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>

#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/json/pctl_JsonDataHandler.h>
#include <nn/pctl/detail/service/watcher/pctl_ServerResource.h>
#include <nn/util/util_IntUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

namespace
{
    // type値が解釈できない場合もステータスコードをResult値に変換する
    // (common::ConvertStatusCodeToResult で扱われない独自ステータスコードもあるのでここで変換)
    static nn::Result HandleDefaultErrorFromStatusCode(int statusCode, nn::Result defaultResult) NN_NOEXCEPT
    {
        switch (statusCode)
        {
            case 400:
                return nn::pctl::ResultUnexpectedServerError400();
            case 401:
                return nn::pctl::ResultUnexpectedServerError401();
            case 403:
                return nn::pctl::ResultUnexpectedServerError403();
            case 404:
                return nn::pctl::ResultUnexpectedServerError404();
            case 405:
                return nn::pctl::ResultUnexpectedServerError405();
            case 406:
                return nn::pctl::ResultUnexpectedServerError406();
            case 409:
                return nn::pctl::ResultUnexpectedServerError409();
            case 424:
                return nn::pctl::ResultUnexpectedServerError424();
            case 500:
                return nn::pctl::ResultUnexpectedServerError500();
            case 531:
                return nn::pctl::ResultUnexpectedServerError531();
            case 532:
                return nn::pctl::ResultUnexpectedServerError532();
            case 202:
                // 「202 No Content」の場合は解析結果が ResultJsonParsingTerminated になるので
                // ResultSuccess に変換する
                if (nn::pctl::ResultJsonParsingTerminated::Includes(defaultResult))
                {
                    return nn::ResultSuccess();
                }
                else
                {
                    return defaultResult;
                }
            default:
                return defaultResult;
        }
    }

    ////////////////////////////////////////////////////////////////////////////

    static bool HandleProblemDetailsObject(void* param, int /*index*/) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WatcherErrorHandler*>(param);
        p->SetProblemDetailsAvailable();
        return true;
    }

    static bool HandleTypeFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WatcherErrorHandler*>(param);
        p->SetType(value, valueLength);
        return true;
    }

    static bool HandleTitleFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WatcherErrorHandler*>(param);
        p->SetTitle(value, valueLength);
        return true;
    }

    static bool HandleErrorCodeFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WatcherErrorHandler*>(param);
        p->SetErrorCode(value, valueLength);
        return true;
    }

    static bool HandleStatusFunction(void* param, int /*index*/, uint64_t value) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WatcherErrorHandler*>(param);
        if (!nn::util::IsIntValueRepresentable<int>(value))
        {
            NN_DETAIL_PCTL_INFO("Invalid 'status' value: %llu\n", value);
            return false;
        }
        p->SetStatus(static_cast<int>(value));
        return true;
    }

    static bool HandleDetailFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<WatcherErrorHandler*>(param);
        p->SetDetail(value, valueLength);
        return true;
    }

    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(static const, ExpectJsonFormatForProblemDetails)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN_CALLBACK(false, HandleProblemDetailsObject)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("type")          NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleTypeFunction)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("title")         NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleTitleFunction)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("errorCode")     NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleErrorCodeFunction)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("status")        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(HandleStatusFunction)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("detail")        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleDetailFunction)
            //NN_DETAIL_PCTL_JSON_EXPECT_KEY("instance")      NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(nullptr)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()
}

WatcherErrorHandler::WatcherErrorHandler() NN_NOEXCEPT :
    json::JsonWebErrorDataHandler(this, HandleError, ExpectJsonFormatForProblemDetails),
    m_ErrorCallback(nullptr),
    m_ParamCallback(nullptr),
    m_IsProblemDetailsAvailable(false)
{
    m_ProblemDetails.resultByType = nn::ResultSuccess();
}

WatcherErrorHandler::WatcherErrorHandler(json::HandleWebErrorFunction errorCallback, void* paramCallback) NN_NOEXCEPT :
    json::JsonWebErrorDataHandler(this, HandleError, ExpectJsonFormatForProblemDetails),
    m_ErrorCallback(errorCallback),
    m_ParamCallback(paramCallback),
    m_IsProblemDetailsAvailable(false)
{
    m_ProblemDetails.resultByType = nn::ResultSuccess();
}

void WatcherErrorHandler::SetType(const char* value, size_t length) NN_NOEXCEPT
{
    if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/400/invalid_params"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerInvalidParameters();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/401/invalid_token"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerInvalidToken();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/403/insufficient_scope"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerInsufficientScope();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/403/forbidden"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerForbidden();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/404/resource_is_not_found"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerResourceIsNotFound();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/405/method_not_allowed"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerMethodNotAllowed();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/406/not_acceptable_language"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerNotAcceptableLanguage();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/409/resource_already_exists"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerResourceAlreadyExists();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/409/over_the_time_limit"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerOverTheTimeLimit();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/424/failed_dependency"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerFailedDependency();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/500/internal_server_error"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultInternalServerError();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/531/under_maintenance"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerInMaintenance();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/532/service_terminated"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServiceTerminated();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/400/firmware_update_required"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerSystemUpdateRequired();
    }
    else if (IsEqualLiteralStringWithLength(value, length, "https://moon.nintendo.com/errors/v1/404/resource_not_available"))
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultServerResourceNotAvailable();
    }
    else
    {
        m_ProblemDetails.resultByType = nn::pctl::ResultUnknownError();
    }
} // NOLINT(impl/function_size)

nn::Result WatcherErrorHandler::HandleError(void* param, int statusCode, nn::Result defaultResult) NN_NOEXCEPT
{
    auto pThis = reinterpret_cast<WatcherErrorHandler*>(param);
    auto result = defaultResult;
    if (result.IsSuccess())
    {
        return result;
    }
    if (pThis->m_ProblemDetails.resultByType.IsFailure())
    {
        if (nn::pctl::ResultUnknownError::Includes(pThis->m_ProblemDetails.resultByType))
        {
            result = HandleDefaultErrorFromStatusCode(statusCode, result);
        }
        else
        {
            result = pThis->m_ProblemDetails.resultByType;
        }
    }
    else
    {
        result = HandleDefaultErrorFromStatusCode(statusCode, result);
    }

    NN_DETAIL_PCTL_INFO("Server return: status = %d\n", statusCode);
    if (pThis->m_IsProblemDetailsAvailable)
    {
        NN_DETAIL_PCTL_INFO("Details:\n");
        NN_DETAIL_PCTL_INFO("  status = %d\n", pThis->m_ProblemDetails.status);
        NN_DETAIL_PCTL_INFO("  title = %s\n", pThis->m_ProblemDetails.title);
        NN_DETAIL_PCTL_INFO("  errorCode = %s\n", pThis->m_ProblemDetails.errorCode);
        NN_DETAIL_PCTL_INFO("  detail = %s\n", pThis->m_ProblemDetails.detail);
    }

    if (pThis->m_ErrorCallback != nullptr)
    {
        return pThis->m_ErrorCallback(pThis->m_ParamCallback, statusCode, result);
    }
    else
    {
        return result;
    }
}

}}}}}
