﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/dispatcher/pctl_RetrieveSettingsDispatcher.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/json/pctl_JsonWebApi.h>
#include <nn/pctl/detail/service/json/pctl_JsonErrorHandler.h>
#include <nn/pctl/detail/service/json/pctl_JsonHttpInputStream.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher { namespace dispatcher {

namespace
{
    struct RetrieveSettingsParam
    {
        EtagInfo* etag;
        system::Settings* pSettings;
        system::FreeCommunicationApplicationSettings* pFreeCommunicationSettings;
        system::ExemptApplicationSettings* pExemptionSettings;
        PlayTimerSettings* pPlayTimerSettings;
        nn::ncm::ApplicationId currentApplicationId;
        Week weekCurrent; // Week_TotalCount を「毎日の設定」を示す値として代用
        bool notModified;
        bool isCustomRatingRestricted;
    };


    static bool HandleEtagFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        if (data->etag != nullptr)
        {
            size_t count = valueLength;
            if (count >= MaxEtagLength)
            {
                NN_DETAIL_PCTL_WARN("Etag is too long (length = %lu)\n", static_cast<uint32_t>(count));
                count = MaxEtagLength - 1;
            }
            std::memcpy(data->etag->etag, value, sizeof(char) * count);
            data->etag->etag[count] = 0;
        }
        return true;
    }

    static bool HandlePinCodeFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        // MEMO: 解除コード無し設定がある場合はここを修正
        if (valueLength < 4 || valueLength > system::MaxPinCodeLength || !system::IsValidPinCode(value, valueLength))
        {
            NN_DETAIL_PCTL_INFO("Invalid pin code: length = %lu, code = %s\n",
                static_cast<uint32_t>(valueLength), value);
            return false;
        }
        std::memcpy(data->pSettings->current.pinCode, value, sizeof(char) * valueLength);
        if (valueLength < system::MaxPinCodeLength)
        {
            data->pSettings->current.pinCode[valueLength] = 0;
        }
        return true;
    }

    static bool HandleSafetyLevelFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        auto val = GetIndexFromString(value, valueLength, SafetyLevelValues);
        if (val < 0)
        {
            NN_DETAIL_PCTL_INFO("Invalid safety-level value: length = %lu, value = %s\n",
                static_cast<uint32_t>(valueLength), value);
            return false;
        }
        data->pSettings->current.safetyLevel = static_cast<system::SizedSafetyLevel>(val);
        return true;
    }

    static bool HandleCustomRatingRestrictionFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        auto val = GetIndexFromString(value, valueLength, RestrictionValues);
        if (val < 0)
        {
            NN_DETAIL_PCTL_INFO("Invalid custom rating-restriction value: length = %lu, value = %s\n",
                static_cast<uint32_t>(valueLength), value);
            return false;
        }
        // val == 0 (UNRESTRICTED) のときは ratingAge を 0 にするが、
        // HandleCustomRatingAgeFunction の処理もあるので
        // 一旦別変数に退避させる
        data->isCustomRatingRestricted = (val != 0);
        return true;
    }

    static bool HandleCustomRatingAgeFunction(void* param, int /*index*/, uint64_t value) NN_NOEXCEPT
    {
        // 大きすぎる値は非対応
        if (value > 0xFFull)
        {
            NN_DETAIL_PCTL_INFO("Invalid custom rating-age value: %llu\n",
                value);
            return false;
        }

        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        data->pSettings->custom.ratingAge = static_cast<uint8_t>(value);
        return true;
    }

    static bool HandleCustomFreeCommunicationRestrictionFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        auto val = GetIndexFromString(value, valueLength, RestrictionValues);
        if (val < 0)
        {
            NN_DETAIL_PCTL_INFO("Invalid custom free-communication-restriction value: length = %lu, value = %s\n",
                static_cast<uint32_t>(valueLength), value);
            return false;
        }
        data->pSettings->custom.isFreeCommunicationRestrictedByDefault = (val != 0);
        return true;
    }

    static bool HandleApplicationListKeyFunction(bool* outIgnoreValue, void* param, int /*index*/, const char* value, size_t length) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        uint64_t idValue;
        if (!HandleIdString(&idValue, value, length))
        {
            NN_DETAIL_PCTL_INFO("Invalid application list key: length = %lu, value = %s\n",
                static_cast<uint32_t>(length), value);
            return false;
        }
        data->currentApplicationId.value = idValue;
        *outIgnoreValue = false;
        return true;
    }

    static bool HandleApplicationListRestrictionFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        auto val = GetIndexFromString(value, valueLength, RestrictionValues);
        if (val < 0)
        {
            NN_DETAIL_PCTL_INFO("Invalid application list restriction value: length = %lu, value = %s\n",
                static_cast<uint32_t>(valueLength), value);
            return false;
        }
        // 重複する場合は false が返る
        // (一旦 0 初期化しているので重複するのは同じデータが複数あるときなのでエラー扱いにする)
        return data->pFreeCommunicationSettings->AddRestrictedValue(data->currentApplicationId, val != 0);
    }

    static bool HandleApplicationListSafeLaunchFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        auto val = GetIndexFromString(value, valueLength, SafeLaunchValues);
        if (val < 0)
        {
            NN_DETAIL_PCTL_INFO("Invalid application list safelaunch value: length = %lu, value = %s\n",
                static_cast<uint32_t>(valueLength), value);
            return false;
        }
        // 重複する場合は false が返る
        // (一旦 0 初期化しているので重複するのは同じデータが複数あるときなのでエラー扱いにする)
        return data->pExemptionSettings->AddExemptedValue(data->currentApplicationId, val != 0);
    }

    static bool HandleCustomSnsPostRestrictionFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        auto val = GetIndexFromString(value, valueLength, RestrictionValues);
        if (val < 0)
        {
            NN_DETAIL_PCTL_INFO("Invalid custom sns-post-restriction value: length = %lu, value = %s\n",
                static_cast<uint32_t>(valueLength), value);
            return false;
        }
        data->pSettings->custom.isSnsPostRestricted = (val != 0);
        return true;
    }

    static bool HandleCustomRatingOrganizationFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        auto val = GetIndexFromString(value, valueLength, RatingOrganizationValues);
        if (val < 0)
        {
            NN_DETAIL_PCTL_INFO("Invalid rating organization value: length = %lu, value = %s\n",
                static_cast<uint32_t>(valueLength), value);
            return false;
        }
        // custom内のデータだが通常データとして扱う
        data->pSettings->current.isDefaultRatingOrganizationSet = 1;
        data->pSettings->current.ratingOrganization = static_cast<nn::ns::RatingOrganization>(val);
        return true;
    }

    static bool HandlePlayTimerDailyModeFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        auto val = GetIndexFromString(value, valueLength, PlayTimerDayWeekMode);
        if (val < 0)
        {
            NN_DETAIL_PCTL_INFO("Invalid play-timer mode value: length = %lu, value = %s\n",
                static_cast<uint32_t>(valueLength), value);
            return false;
        }
        data->pPlayTimerSettings->isWeekSettingsUsed = (val != 0);
        return true;
    }

    static bool HandlePlayTimerRestrictionModeFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        auto val = GetIndexFromString(value, valueLength, PlayTimerRestrictionMode);
        if (val < 0)
        {
            NN_DETAIL_PCTL_INFO("Invalid play-timer restriction mode value: length = %lu, value = %s\n",
                static_cast<uint32_t>(valueLength), value);
            return false;
        }
        data->pPlayTimerSettings->playTimerMode = static_cast<PlayTimerMode>(val);
        return true;
    }

    static bool HandlePlayTimerDailySettingsFunction(void* param, int /*index*/) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        data->weekCurrent = Week::Week_TotalCount;
        return true;
    }

    static bool HandlePlayTimerWeekKeyFunction(bool* outIgnoreValue, void* param, int /*index*/, const char* value, size_t length) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        if (IsEqualLiteralStringWithLength(value, length, "sunday"))
        {
            data->weekCurrent = Week::Week_Sunday;
        }
        else if (IsEqualLiteralStringWithLength(value, length, "monday"))
        {
            data->weekCurrent = Week::Week_Monday;
        }
        else if (IsEqualLiteralStringWithLength(value, length, "tuesday"))
        {
            data->weekCurrent = Week::Week_Tuesday;
        }
        else if (IsEqualLiteralStringWithLength(value, length, "wednesday"))
        {
            data->weekCurrent = Week::Week_Wednesday;
        }
        else if (IsEqualLiteralStringWithLength(value, length, "thursday"))
        {
            data->weekCurrent = Week::Week_Thursday;
        }
        else if (IsEqualLiteralStringWithLength(value, length, "friday"))
        {
            data->weekCurrent = Week::Week_Friday;
        }
        else if (IsEqualLiteralStringWithLength(value, length, "saturday"))
        {
            data->weekCurrent = Week::Week_Saturday;
        }
        else
        {
            NN_DETAIL_PCTL_INFO("Invalid play-timer week key name: %s\n", value);
            return false;
        }
        *outIgnoreValue = false;
        return true;
    }

    static bool HandlePlayTimerDailySettingsLimitTimeEnabledFunction(void* param, int /*index*/, bool value) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        if (data->weekCurrent == Week::Week_TotalCount)
        {
            data->pPlayTimerSettings->dailySettings.isLimitTimeEnabled = value;
        }
        else
        {
            data->pPlayTimerSettings->weekSettings[data->weekCurrent].isLimitTimeEnabled = value;
        }
        return true;
    }

    static bool HandlePlayTimerDailySettingsLimitTimeValueFunction(void* param, int /*index*/, nn::util::optional<uint64_t> value) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        if (value == nn::util::nullopt)
        {
            // null の場合は disabled になるはずだがひとまず 0 を指定する
            if (data->weekCurrent == Week::Week_TotalCount)
            {
                data->pPlayTimerSettings->dailySettings.limitTime = 0;
            }
            else
            {
                data->pPlayTimerSettings->weekSettings[data->weekCurrent].limitTime = 0;
            }
        }
        else
        {
            // 仕様上最大値は 360
            if (*value > 360)
            {
                NN_DETAIL_PCTL_INFO("Too large play-timer limit time value: %llu\n",
                    *value);
                return false;
            }
            if (data->weekCurrent == Week::Week_TotalCount)
            {
                data->pPlayTimerSettings->dailySettings.limitTime = static_cast<uint16_t>(*value);
            }
            else
            {
                data->pPlayTimerSettings->weekSettings[data->weekCurrent].limitTime = static_cast<uint16_t>(*value);
            }
        }
        return true;
    }

    static bool HandlePlayTimerDailySettingsBedtimeFunction(void* param, int /*index*/) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        nn::pctl::PlayTimerDaySettings* pSettings;
        if (data->weekCurrent == Week::Week_TotalCount)
        {
            pSettings = &data->pPlayTimerSettings->dailySettings;
        }
        else
        {
            pSettings = &data->pPlayTimerSettings->weekSettings[data->weekCurrent];
        }
        // 値のリセットをしておく
        pSettings->isBedtimeEnabled = false;
        pSettings->bedtimeHour = 0;
        pSettings->bedtimeMinute = 0;
        return true;
    }

    static bool HandlePlayTimerDailySettingsBedtimeEnabledFunction(void* param, int /*index*/, bool value) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        if (data->weekCurrent == Week::Week_TotalCount)
        {
            data->pPlayTimerSettings->dailySettings.isBedtimeEnabled = value;
        }
        else
        {
            data->pPlayTimerSettings->weekSettings[data->weekCurrent].isBedtimeEnabled = value;
        }
        return true;
    }

    static bool HandlePlayTimerDailySettingsBedtimeHourValueFunction(void* param, int /*index*/, uint64_t value) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);

        if (value > 23)
        {
            NN_DETAIL_PCTL_INFO("[pctl] playTimerRegulations: Too large hour value: %llu\n", value);
            return false;
        }

        if (data->weekCurrent == Week::Week_TotalCount)
        {
            data->pPlayTimerSettings->dailySettings.bedtimeHour = static_cast<uint8_t>(value);
        }
        else
        {
            data->pPlayTimerSettings->weekSettings[data->weekCurrent].bedtimeHour = static_cast<uint8_t>(value);
        }
        return true;
    }

    static bool HandlePlayTimerDailySettingsBedtimeMinuteValueFunction(void* param, int /*index*/, uint64_t value) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);

        if (value > 59)
        {
            NN_DETAIL_PCTL_INFO("[pctl] playTimerRegulations: Too large minute value: %llu\n", value);
            return false;
        }

        if (data->weekCurrent == Week::Week_TotalCount)
        {
            data->pPlayTimerSettings->dailySettings.bedtimeMinute = static_cast<uint8_t>(value);
        }
        else
        {
            data->pPlayTimerSettings->weekSettings[data->weekCurrent].bedtimeMinute = static_cast<uint8_t>(value);
        }
        return true;
    }

    static bool HandleUpdatedAtValueFunction(void* param, int /*index*/, uint64_t value) NN_NOEXCEPT
    {
        auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
        if (data->etag != nullptr)
        {
            data->etag->lastUpdatedAt = value;
        }
        return true;
    }

    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(static const, ExpectJsonFormatForParentalControlSetting)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("etag")                        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleEtagFunction)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("unlockCode")                  NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandlePinCodeFunction)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("functionalRestrictionLevel")  NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleSafetyLevelFunction)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("customSettings")              NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("ageRestriction")      NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                    NN_DETAIL_PCTL_JSON_EXPECT_KEY("restriction") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleCustomRatingRestrictionFunction)
                    NN_DETAIL_PCTL_JSON_EXPECT_KEY("ratingAge")   NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(HandleCustomRatingAgeFunction)
                NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("ugcRestriction")      NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                    NN_DETAIL_PCTL_JSON_EXPECT_KEY("restriction")         NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleCustomFreeCommunicationRestrictionFunction)
                    NN_DETAIL_PCTL_JSON_EXPECT_KEY("applications")        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                        NN_DETAIL_PCTL_JSON_EXPECT_ANY_KEY(HandleApplicationListKeyFunction) NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                            NN_DETAIL_PCTL_JSON_EXPECT_KEY("restriction") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleApplicationListRestrictionFunction)
                        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                    NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("snsPostRestriction")  NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleCustomSnsPostRestrictionFunction)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("ratingOrganization")  NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleCustomRatingOrganizationFunction)
            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("playTimerRegulations")        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("timerMode")           NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandlePlayTimerDailyModeFunction)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("restrictionMode")     NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandlePlayTimerRestrictionModeFunction)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("dailyRegulations")    NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN_CALLBACK(false, HandlePlayTimerDailySettingsFunction)
                    NN_DETAIL_PCTL_JSON_EXPECT_KEY("timeToPlayInOneDay")  NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                        NN_DETAIL_PCTL_JSON_EXPECT_KEY("enabled")     NN_DETAIL_PCTL_JSON_EXPECT_VALUE_BOOLEAN(HandlePlayTimerDailySettingsLimitTimeEnabledFunction)
                        NN_DETAIL_PCTL_JSON_EXPECT_KEY("limitTime")   NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64_NULLABLE(HandlePlayTimerDailySettingsLimitTimeValueFunction)
                    NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                    NN_DETAIL_PCTL_JSON_EXPECT_KEY("bedtime")             NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN_CALLBACK(false, HandlePlayTimerDailySettingsBedtimeFunction)
                        NN_DETAIL_PCTL_JSON_EXPECT_KEY("enabled")     NN_DETAIL_PCTL_JSON_EXPECT_VALUE_BOOLEAN(HandlePlayTimerDailySettingsBedtimeEnabledFunction)
                        NN_DETAIL_PCTL_JSON_EXPECT_KEY("endingTime")  NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(true)
                            NN_DETAIL_PCTL_JSON_EXPECT_KEY("hour")        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(HandlePlayTimerDailySettingsBedtimeHourValueFunction)
                            NN_DETAIL_PCTL_JSON_EXPECT_KEY("minute")      NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(HandlePlayTimerDailySettingsBedtimeMinuteValueFunction)
                        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                    NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("eachDayOfTheWeekRegulations") NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                    // sunday,monday,tuesday,wednesday,thursday,friday,saturday を1つでハンドルする
                    NN_DETAIL_PCTL_JSON_EXPECT_ANY_KEY(HandlePlayTimerWeekKeyFunction) NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                        NN_DETAIL_PCTL_JSON_EXPECT_KEY("timeToPlayInOneDay")  NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                            NN_DETAIL_PCTL_JSON_EXPECT_KEY("enabled")     NN_DETAIL_PCTL_JSON_EXPECT_VALUE_BOOLEAN(HandlePlayTimerDailySettingsLimitTimeEnabledFunction)
                            NN_DETAIL_PCTL_JSON_EXPECT_KEY("limitTime")   NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64_NULLABLE(HandlePlayTimerDailySettingsLimitTimeValueFunction)
                        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                        NN_DETAIL_PCTL_JSON_EXPECT_KEY("bedtime")             NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN_CALLBACK(false, HandlePlayTimerDailySettingsBedtimeFunction)
                            NN_DETAIL_PCTL_JSON_EXPECT_KEY("enabled")     NN_DETAIL_PCTL_JSON_EXPECT_VALUE_BOOLEAN(HandlePlayTimerDailySettingsBedtimeEnabledFunction)
                            NN_DETAIL_PCTL_JSON_EXPECT_KEY("endingTime")  NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(true)
                                NN_DETAIL_PCTL_JSON_EXPECT_KEY("hour")        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(HandlePlayTimerDailySettingsBedtimeHourValueFunction)
                                NN_DETAIL_PCTL_JSON_EXPECT_KEY("minute")      NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(HandlePlayTimerDailySettingsBedtimeMinuteValueFunction)
                            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                    NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("whitelistedApplications")      NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                NN_DETAIL_PCTL_JSON_EXPECT_ANY_KEY(HandleApplicationListKeyFunction) NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                    NN_DETAIL_PCTL_JSON_EXPECT_KEY("safeLaunch") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleApplicationListSafeLaunchFunction)
                NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("updatedAt")                   NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(HandleUpdatedAtValueFunction)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    static nn::Result HandleErrorResponse(void* param, int statusCode, nn::Result defaultResult) NN_NOEXCEPT
    {
        // 304 Not Modified をハンドルする
        if (statusCode == 304)
        {
            auto data = reinterpret_cast<RetrieveSettingsParam*>(param);
            if (data->etag != nullptr && data->etag->lastUpdatedAt != 0)
            {
                data->notModified = true;
                NN_RESULT_SUCCESS;
            }
        }
        return defaultResult;
    }

} // namespace `anonymous'

////////////////////////////////////////////////////////////////////////////////

nn::Result RetrieveSettingsDispatcher::Execute(EtagInfo* etag,
    bool* outIsNotModified,
    system::Settings* pSettings,
    system::FreeCommunicationApplicationSettings* pFreeCommuncationSettings,
    system::ExemptApplicationSettings* pExemptionSettings,
    PlayTimerSettings* pPlayTimerSettings,
    common::NetworkBuffer& bufferInfo,
    common::Cancelable* pCancelable,
    const char* token,
    ServerDeviceId deviceId) NN_NOEXCEPT
{
    RetrieveSettingsParam param;

    WatcherErrorHandler* pErrorHandler = new WatcherErrorHandler(HandleErrorResponse, &param);
    NN_RESULT_THROW_UNLESS(pErrorHandler != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        delete pErrorHandler;
    };

    // tokenヘッダーの作成
    // 22 == strlen("Authorization: Bearer ")
    size_t tokenHeaderLength = std::strlen(token) + 23;
    char* tokenHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * tokenHeaderLength));
    NN_RESULT_THROW_UNLESS(tokenHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(tokenHeader);
    };
    nn::util::SNPrintf(tokenHeader, tokenHeaderLength, "Authorization: Bearer %s", token);
    // Etag用のヘッダーの作成
    size_t ifNoneMatchHeaderLength = 0;
    char* ifNoneMatchHeader = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        if (ifNoneMatchHeader != nullptr)
        {
            FreeMemoryBlock(ifNoneMatchHeader);
        }
    };
    if (etag != nullptr && etag->lastUpdatedAt != 0)
    {
        // 15 == strlen("If-None-Match: ")
        ifNoneMatchHeaderLength = std::strlen(etag->etag) + 16;
        ifNoneMatchHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * ifNoneMatchHeaderLength));
        NN_RESULT_THROW_UNLESS(ifNoneMatchHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
        nn::util::SNPrintf(ifNoneMatchHeader, ifNoneMatchHeaderLength, "If-None-Match: %s",
            etag->etag);
    }

    // URLの作成
    char url[UrlBufferLength_RetrieveSettings];

    nn::util::SNPrintf(url, std::extent<decltype(url)>::value, UrlFormat_RetrieveSettings,
        ServerEndpoint, deviceId);

    param.etag = etag;

    // 以下の4つは呼び出し元で事前に初期化されているものとする
    param.pSettings = pSettings;
    param.pFreeCommunicationSettings = pFreeCommuncationSettings;
    param.pExemptionSettings = pExemptionSettings;
    param.pPlayTimerSettings = pPlayTimerSettings;

    param.notModified = false;
    param.isCustomRatingRestricted = false;

    nn::pctl::detail::service::json::JsonHttpInputStream stream;
    json::JsonDataHandler handler(&param, ExpectJsonFormatForParentalControlSetting);

    NN_RESULT_DO(stream.GetRequest().Open(url));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader(tokenHeader));
    if (ifNoneMatchHeader != nullptr)
    {
        NN_RESULT_DO(stream.GetRequest().AddRequestHeader(ifNoneMatchHeader));
    }

    NN_RESULT_DO(
        json::ParseWebStream(&handler, pErrorHandler, &stream,
            bufferInfo.GetBufferForJsonValue(), common::NetworkBuffer::JsonValueBufferMemorySize,
            bufferInfo.GetBufferForStream(), common::NetworkBuffer::StreamBufferMemorySize,
            pCancelable)
        );

    *outIsNotModified = param.notModified;
    // 解析後の値調整
    if (!param.notModified)
    {
        // MEMO: 解除コード無し設定がある場合はここを修正
        NN_RESULT_THROW_UNLESS(pSettings->current.pinCode[0] != 0, nn::pctl::ResultResponseFormatError());

        // プレイタイマーの有効・無効はサーバー上には無いので一旦 false にする
        pPlayTimerSettings->isEnabled = false;

        // 一時的に退避していた値を戻す
        if (!param.isCustomRatingRestricted)
        {
            pSettings->custom.ratingAge = 0;
        }

        // Etagが空であれば無効化する
        if (etag->etag[0] == 0)
        {
            etag->lastUpdatedAt = 0;
        }
    }

    NN_RESULT_SUCCESS;
}

////////////////////////////////////////////////////////////////////////////////

}}}}}}
