﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/dispatcher/pctl_RetrieveOwnersDispatcher.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/json/pctl_JsonWebApi.h>
#include <nn/pctl/detail/service/json/pctl_JsonErrorHandler.h>
#include <nn/pctl/detail/service/json/pctl_JsonHttpInputStream.h>
#include <nn/pctl/detail/service/json/pctl_JsonStructuredWriter.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher { namespace dispatcher {

namespace
{
    ////////////////////////////////////////////////////////////////////////////////

    struct ServerDeviceOwnerArrayParam
    {
        ServerDeviceOwner* arrayOwners;
        size_t maxCount;
        size_t actualCount;
        size_t currentIndex;
    };

    static bool HandleTotalCountForArrayOfDeviceOwnerFunction(void* param, int /*index*/, uint64_t value) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<ServerDeviceOwnerArrayParam*>(param);
        // 大きすぎる値は非対応
        if (value > 0xFFFFFFFFull)
        {
            return false;
        }
        p->actualCount = static_cast<size_t>(value);
        return true;
    }

    static bool HandleItemForArrayOfDeviceOwnerFunction(void* param, int index) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<ServerDeviceOwnerArrayParam*>(param);
        if (index < 0 || static_cast<size_t>(index) >= p->maxCount)
        {
            // 無視するだけ
            return true;
        }
        // index の更新はここで行う
        p->currentIndex = index;
        // (miiUri も 0 初期化される)
        std::memset(&p->arrayOwners[index], 0, sizeof(p->arrayOwners[index]));
        return true;
    }

    static bool HandleDeviceIdForArrayOfDeviceOwnerFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<ServerDeviceOwnerArrayParam*>(param);
        if (p->currentIndex >= p->maxCount)
        {
            // 無視するだけ
            return true;
        }
        return HandleIdString(&p->arrayOwners[p->currentIndex].owningDeviceId, value, valueLength);
    }

    static bool HandleAccountIdForArrayOfDeviceOwnerFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<ServerDeviceOwnerArrayParam*>(param);
        if (p->currentIndex >= p->maxCount)
        {
            // 無視するだけ
            return true;
        }
        if (value == nullptr || valueLength == 0)
        {
            p->arrayOwners[p->currentIndex].user.accountId = 0;
            return true;
        }
        return HandleIdString(&p->arrayOwners[p->currentIndex].user.accountId, value, valueLength);
    }

    static bool HandleNicknameForArrayOfDeviceOwnerFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<ServerDeviceOwnerArrayParam*>(param);
        if (p->currentIndex >= p->maxCount)
        {
            // 無視するだけ
            return true;
        }
        if (value == nullptr || valueLength == 0)
        {
            p->arrayOwners[p->currentIndex].user.nickname[0] = 0;
            return true;
        }
        if (valueLength >= nn::account::NicknameBytesMax)
        {
            valueLength = nn::account::NicknameBytesMax - 1;
        }
        nn::util::Strlcpy(p->arrayOwners[p->currentIndex].user.nickname, value, static_cast<int>(valueLength) + 1);
        return true;
    }

    static bool HandleMiiUriForArrayOfDeviceOwnerFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<ServerDeviceOwnerArrayParam*>(param);
        if (p->currentIndex >= p->maxCount)
        {
            // 無視するだけ
            return true;
        }
        if (value == nullptr)
        {
            // 'miiUri' オブジェクトが null の場合はここに来ないが
            // HandleItemForArrayOfDeviceOwnerFunction で0初期化されているので問題なし
            p->arrayOwners[p->currentIndex].user.miiUri[0] = 0;
            return true;
        }
        if (valueLength >= MaxMiiUriLength)
        {
            valueLength = MaxMiiUriLength - 1;
        }
        nn::util::Strlcpy(p->arrayOwners[p->currentIndex].user.miiUri, value, static_cast<int>(valueLength) + 1);
        return true;
    }

    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(static const, ExpectJsonFormatForArrayOfDeviceOwner)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("count")  NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(HandleTotalCountForArrayOfDeviceOwnerFunction)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("items")  NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_BEGIN(false)
                NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN_CALLBACK(false, HandleItemForArrayOfDeviceOwnerFunction)
                    NN_DETAIL_PCTL_JSON_EXPECT_KEY("deviceId")  NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleDeviceIdForArrayOfDeviceOwnerFunction)
                    NN_DETAIL_PCTL_JSON_EXPECT_KEY("user")      NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                        NN_DETAIL_PCTL_JSON_EXPECT_KEY("nintendoAccountId")  NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleAccountIdForArrayOfDeviceOwnerFunction)
                        NN_DETAIL_PCTL_JSON_EXPECT_KEY("nickname")           NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleNicknameForArrayOfDeviceOwnerFunction)
                        NN_DETAIL_PCTL_JSON_EXPECT_KEY("miiUri")             NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(true) // null可
                            // extraSmall: 96px, small: 128px, medium: 140px, large: 270px, extraLarge: 512px
                            // ここでは small のみ取り出す
                            NN_DETAIL_PCTL_JSON_EXPECT_KEY("small")          NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleMiiUriForArrayOfDeviceOwnerFunction)
                        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                    NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
                NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
            NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_END()
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

} // namespace `anonymous'

////////////////////////////////////////////////////////////////////////////////

nn::Result RetrieveDeviceOwnersDispatcher::Execute(size_t* outActualCount, ServerDeviceOwner* pOwners, size_t maxOwnerCount, common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable, const char* token, ServerDeviceId deviceId) NN_NOEXCEPT
{
    WatcherErrorHandler* pErrorHandler = new WatcherErrorHandler();
    NN_RESULT_THROW_UNLESS(pErrorHandler != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        delete pErrorHandler;
    };

    // tokenヘッダーの作成
    // 22 == strlen("Authorization: Bearer ")
    size_t tokenHeaderLength = std::strlen(token) + 23;
    char* tokenHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * tokenHeaderLength));
    NN_RESULT_THROW_UNLESS(tokenHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(tokenHeader);
    };
    nn::util::SNPrintf(tokenHeader, tokenHeaderLength, "Authorization: Bearer %s", token);

    // URLの作成
    char url[UrlBufferLength_RetrieveOwners];

    nn::util::SNPrintf(url, std::extent<decltype(url)>::value, UrlFormat_RetrieveOwners,
        ServerEndpoint, deviceId);

    ServerDeviceOwnerArrayParam param;
    param.arrayOwners = pOwners;
    param.maxCount = maxOwnerCount;
    param.actualCount = 0;
    param.currentIndex = 0;

    nn::pctl::detail::service::json::JsonHttpInputStream stream;
    json::JsonDataHandler handler(&param, ExpectJsonFormatForArrayOfDeviceOwner);

    NN_RESULT_DO(stream.GetRequest().Open(url));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader(tokenHeader));

    NN_RESULT_DO(
        json::ParseWebStream(&handler, pErrorHandler, &stream,
            bufferInfo.GetBufferForJsonValue(), common::NetworkBuffer::JsonValueBufferMemorySize,
            bufferInfo.GetBufferForStream(), common::NetworkBuffer::StreamBufferMemorySize,
            pCancelable)
        );

    *outActualCount = param.actualCount;

    NN_RESULT_SUCCESS;
}

}}}}}}
