﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/dispatcher/pctl_PairingDispatcher.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/json/pctl_JsonWebApi.h>
#include <nn/pctl/detail/service/json/pctl_JsonErrorHandler.h>
#include <nn/pctl/detail/service/json/pctl_JsonHttpInputStream.h>
#include <nn/pctl/detail/service/json/pctl_JsonStructuredWriter.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher { namespace dispatcher {

namespace
{
    ////////////////////////////////////////////////////////////////////////////////

    struct ServerDeviceSimpleArrayParam
    {
        ServerDeviceSimple* arrayDevices;
        size_t maxCount;
        size_t currentIndex;
        size_t actualCount;
    };

    static bool HandleTotalCountForArrayOfDeviceSimpleFunction(void* param, int /*index*/, uint64_t value) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<ServerDeviceSimpleArrayParam*>(param);
        p->actualCount = static_cast<size_t>(value);
        return true;
    }

    static bool HandleItemForArrayOfDeviceSimpleFunction(void* param, int index) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<ServerDeviceSimpleArrayParam*>(param);
        if (index < 0 || static_cast<size_t>(index) >= p->maxCount)
        {
            // 無視するだけ
            return true;
        }
        // index の更新はここで行う
        p->currentIndex = index;
        return true;
    }

    static bool HandleIdForArrayOfDeviceSimpleFunction(void* param, int /*index*/, const char* value, size_t valueLength) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<ServerDeviceSimpleArrayParam*>(param);
        if (p->currentIndex >= p->maxCount)
        {
            // 無視するだけ
            return true;
        }
        return HandleIdString(&p->arrayDevices[p->currentIndex].id, value, valueLength);
    }

    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(static const, ExpectJsonFormatForArrayOfDeviceSimple)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("count")  NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(HandleTotalCountForArrayOfDeviceSimpleFunction)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("items")  NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_BEGIN(false)
                NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN_CALLBACK(false, HandleItemForArrayOfDeviceSimpleFunction)
                    NN_DETAIL_PCTL_JSON_EXPECT_KEY("id")  NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(HandleIdForArrayOfDeviceSimpleFunction)
                NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
            NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_END()
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

} // namespace `anonymous'

////////////////////////////////////////////////////////////////////////////////

nn::Result FindDeviceWithPairingCodeDispatcher::Execute(ServerDeviceSimple& outDevice, common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable,
    const char* token, const char (& pairingCode)[MaxPairingCodeLength + 1]) NN_NOEXCEPT
{
    WatcherErrorHandler* pErrorHandler = new WatcherErrorHandler();
    NN_RESULT_THROW_UNLESS(pErrorHandler != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        delete pErrorHandler;
    };

    // tokenヘッダーの作成
    // 22 == strlen("Authorization: Bearer ")
    size_t tokenHeaderLength = std::strlen(token) + 23;
    char* tokenHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * tokenHeaderLength));
    NN_RESULT_THROW_UNLESS(tokenHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(tokenHeader);
    };
    nn::util::SNPrintf(tokenHeader, tokenHeaderLength, "Authorization: Bearer %s", token);

    // URLの作成
    char url[UrlBufferLength_FindDeviceWithPairingCode];

    nn::util::SNPrintf(url, std::extent<decltype(url)>::value, UrlFormat_FindDeviceWithPairingCode,
        ServerEndpoint, pairingCode);

    ServerDeviceSimpleArrayParam param;
    param.arrayDevices = &outDevice;
    param.maxCount = 1;
    param.currentIndex = 0;
    param.actualCount = 0;

    nn::pctl::detail::service::json::JsonHttpInputStream stream;
    json::JsonDataHandler handler(&param, ExpectJsonFormatForArrayOfDeviceSimple);

    NN_RESULT_DO(stream.GetRequest().Open(url));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader(tokenHeader));

    NN_RESULT_DO(
        json::ParseWebStream(&handler, pErrorHandler, &stream,
            bufferInfo.GetBufferForJsonValue(), common::NetworkBuffer::JsonValueBufferMemorySize,
            bufferInfo.GetBufferForStream(), common::NetworkBuffer::StreamBufferMemorySize,
            pCancelable)
        );

    // 見つからない場合/多い場合は不明な連携コード扱い
    NN_RESULT_THROW_UNLESS(param.actualCount == 1, nn::pctl::ResultUnknownPairingCode());

    NN_RESULT_SUCCESS;
}

}}}}}}
