﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/system/pctl_SettingsManager.h>

#include <nn/nn_Abort.h>
#include <nn/nn_StaticAssert.h>
#include <nn/crypto/crypto_HmacSha256Generator.h>
#include <nn/fs/fs_Result.h>
#include <nn/os/os_Random.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/pctl/detail/service/common/pctl_FileSystem.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace system {

namespace
{
    // マスターキー計算に使うデータ(乱数)を保存するファイル
    static const char MasterKeySettingsFileName[] = "mks.dat";

    // 使用するマスターキー計算用鍵(32バイト)とインデックス(バージョン)値
    // NOTE: 更新がある場合はこれらの値を変更し、MakeParentalControlMasterKey ツールの
    //   MasterKey.cs 内の記述も修正する
    static const int CurrentEncryptionKeyIndex = 1;
    static const uint8_t EncryptionKey[] = {
        0x56,0x34,0xA7,0x96,0xF7,0xB5,0x8B,0x04,0x13,0x4E,0x1F,0xD6,0x27,0x68,0xA1,0xFA,0xC1,0xAA,0xF8,0x79,0xF6,0xEF,0xC4,0x93,0x8C,0x66,0xFE,0xEB,0xB6,0xD1,0x14,0xA6
    };

    // HMAC SHA256 で利用する鍵のサイズ
    static const size_t EncryptionKeySize = 32;
    NN_STATIC_ASSERT(static_cast<size_t>(std::extent<decltype(EncryptionKey)>::value) == EncryptionKeySize);
    // お問い合わせコードの長さ(NULL文字含まず)
    static const size_t InquiryCodeLength = 10;
    // マスターキーの長さ(NULL文字含まず)
    static const size_t MasterKeyCodeLength = 8;
    // お問い合わせコードに含めるマスターキー計算用鍵のバージョン値
    static const int MasterKeyVersion = (10 + CurrentEncryptionKeyIndex);
    NN_STATIC_ASSERT(MasterKeyVersion >= 10); // 10以上になるようにする
    NN_STATIC_ASSERT(MasterKeyVersion < 100); // 100未満になるようにする

    // マスターキー計算に使うデータ
    struct MasterKeyCalculationData
    {
        uint64_t randomDataForInitial; //!< 初期化時に未設定であれば生成する乱数
    };

    // 保存してある計算用データ(MasterKeyCalculationData)をセーブデータファイルから読み込みます。
    bool ReadMasterKeyCalculationDataFromFile(MasterKeyCalculationData& data) NN_NOEXCEPT
    {
        size_t size = 0;
        // データが保存済みであればそれを読み込む
        {
            common::FileStream stream;
            auto result = common::FileSystem::OpenRead(&stream, MasterKeySettingsFileName);
            NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result),
                "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
            if (result.IsSuccess())
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Read(&size, 0,
                    &data, sizeof(data)));
            }
        }
        // size が不一致の場合(size が 0 も含む)はデータを読み込めてない
        return size == sizeof(data);
    }

    // 計算用データ(MasterKeyCalculationData)をセーブデータファイルに書き込みます。
    void WriteMasterKeyCalculationDataToFile(const MasterKeyCalculationData& data) NN_NOEXCEPT
    {
        {
            common::FileStream stream;
            NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::OpenWrite(&stream, MasterKeySettingsFileName, sizeof(data)));
            NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Write(0, &data, sizeof(data)));
            NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Flush());
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    }

    // 乱数を生成します。
    void GenerateRandomBytes(void* data, size_t size) NN_NOEXCEPT
    {
        nn::os::GenerateRandomBytes(data, size);
    }

    // 64ビット整数として乱数を生成します。ただし 0 にならないように生成を繰り返します。
    void GenerateRandomIntegerWithoutZero(uint64_t* outValue) NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(true))
        {
            GenerateRandomBytes(outValue, sizeof(uint64_t));
            // 乱数が 0 になる場合のみ繰り返す
            if (*outValue == 0)
            {
                continue;
            }
            break;
        }
    }

    // outCode に「お問い合わせコード」の文字列をセットします。
    // (outCode には必ずNULL文字が追加される)
    template <size_t N>
    inline void FormatInquiryCode(char (& outCode)[N], const MasterKeyCalculationData& data, uint64_t dataForInquiry) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(N >= InquiryCodeLength + 1);
        uint64_t tempValue = (data.randomDataForInitial ^ dataForInquiry); // XORを取る
        tempValue %= 100000000llu; // 下8桁を取得
        nn::util::SNPrintf(outCode, N, "%02d%08llu", static_cast<int>(MasterKeyVersion), tempValue);
        outCode[InquiryCodeLength] = 0;
    }
    NN_STATIC_ASSERT(InquiryCodeLength == 10);
} // namespace `anonymous-namespace'

// プロセス起動時処理
void SettingsManager::InitializeForMasterKey() NN_NOEXCEPT
{
    // NOTE: 全機能無効化中(m_IsDisabledAll == true)も初期化は行う
    m_DataForMasterKeyInquiry = 0;

    MasterKeyCalculationData data;
    // データが保存済みであればそれを読み込み、
    // 読み込めてない場合は初期化する
    if (!ReadMasterKeyCalculationDataFromFile(data))
    {
        GenerateRandomIntegerWithoutZero(&data.randomDataForInitial);
        WriteMasterKeyCalculationDataToFile(data);
    }
    // データ自体はここでは利用しない
}

// お問い合わせコード発行処理
void SettingsManager::GenerateInquiryCode(InquiryCode* outCode) NN_NOEXCEPT
{
    // お問い合わせコード用の乱数生成込みでデータを読み込む
    MasterKeyCalculationData data;
    // データが保存済みであればそれを読み込み、
    // 読み込めてない場合は MasterKeyCalculationData 全体を初期化する
    if (!ReadMasterKeyCalculationDataFromFile(data))
    {
        GenerateRandomIntegerWithoutZero(&data.randomDataForInitial);
    }
    // m_DataForMasterKeyInquiry は都度生成する
    GenerateRandomIntegerWithoutZero(&m_DataForMasterKeyInquiry);
    WriteMasterKeyCalculationDataToFile(data);

    // 予約領域は使用しないので0でクリアしておく
    std::memset(outCode->_reserved, 0, sizeof(outCode->_reserved));

    // 合計10桁の数を出力
    NN_STATIC_ASSERT(std::extent<decltype(outCode->codeString)>::value >= 11); // NULL文字込みで11文字格納できることの確認
    FormatInquiryCode(outCode->codeString, data, m_DataForMasterKeyInquiry);
}

// マスターキー照合処理
bool SettingsManager::CheckMasterKey(const InquiryCode& codeData, const char* masterKey, size_t masterKeyLength) NN_NOEXCEPT
{
    // 総当たりアタック対策のため1秒ウェイトを入れる
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // 長さが異なる場合は不一致とする
    // ※ masterKeyLength はNULL文字込みの長さ
    if (MasterKeyCodeLength != static_cast<size_t>(nn::util::Strnlen(masterKey, static_cast<int>(masterKeyLength))))
    {
        return false;
    }

    MasterKeyCalculationData data;
    // データが保存済みであればそれを読み込み、
    // 読み込めてない場合はお問い合わせコードが発行できていないので失敗とする
    if (!ReadMasterKeyCalculationDataFromFile(data))
    {
        return false;
    }

    // 何らかの理由でお問い合わせコード用の乱数が生成されていない場合は失敗とする
    if (m_DataForMasterKeyInquiry == 0)
    {
        return false;
    }

    uint8_t macData[nn::crypto::HmacSha256Generator::MacSize];

    {
        // お問い合わせコードのチェック
        char inquiryCode[InquiryCodeLength + 1];
        FormatInquiryCode(inquiryCode, data, m_DataForMasterKeyInquiry);
        if (std::strcmp(inquiryCode, codeData.codeString) != 0)
        {
            return false;
        }

        // MACの生成
        nn::crypto::GenerateHmacSha256Mac(macData, std::extent<decltype(macData)>::value,
            inquiryCode, InquiryCodeLength,
            EncryptionKey, EncryptionKeySize);
    }

    // 最初の6バイトをリトルエンディアンの整数として抜き出し、
    // そこから10進数換算で下位8桁を取り出す
    uint64_t masterKeyValue =
        (static_cast<uint64_t>(macData[0])) |
        (static_cast<uint64_t>(macData[1]) << 8) |
        (static_cast<uint64_t>(macData[2]) << 16) |
        (static_cast<uint64_t>(macData[3]) << 24) |
        (static_cast<uint64_t>(macData[4]) << 32) |
        (static_cast<uint64_t>(macData[5]) << 40);
    masterKeyValue %= 100000000ull;

    // 抜き出した値を文字列化して比較する
    {
        char targetKey[MasterKeyCodeLength + 1];
        nn::util::SNPrintf(targetKey, MasterKeyCodeLength + 1, "%08llu", masterKeyValue);
        targetKey[MasterKeyCodeLength] = 0;

        if (std::strncmp(targetKey, masterKey, MasterKeyCodeLength) != 0)
        {
            return false;
        }
    }

    return true;
}

// 解除コード変更があった場合の処理
void SettingsManager::OnChangedPinCode() NN_NOEXCEPT
{
    // 解除コード変更時は randomDataForInitial を再初期化して保存する

    MasterKeyCalculationData data;
    // データが保存済みであればそれを読み込む
    // 読み込めてない場合は初期化する(がここでは初期化するデータが無いので何もしない)
    ReadMasterKeyCalculationDataFromFile(data);

    GenerateRandomIntegerWithoutZero(&data.randomDataForInitial);
    WriteMasterKeyCalculationDataToFile(data);
}

}}}}}
