﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/system/pctl_Settings.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace system {

size_t FreeCommunicationApplicationSettings::GetAvailableCount() const NN_NOEXCEPT
{
    size_t count = 0;
    for (const auto& d : data)
    {
        for (const nn::ncm::ApplicationId& a : d.applicationIds)
        {
            if (a != nn::ncm::ApplicationId::GetInvalidId())
            {
                ++count;
            }
        }
    }
    return count;
}

bool FreeCommunicationApplicationSettings::IsAnyRestrictionEnabled() const NN_NOEXCEPT
{
    for (const auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            if (d.applicationIds[i] != nn::ncm::ApplicationId::GetInvalidId())
            {
                if (d.IsRestricted(i))
                {
                    return true;
                }
            }
        }
    }
    return false;
}

bool FreeCommunicationApplicationSettings::GetRestrictedValueByApplicationId(bool* outValue, nn::ncm::ApplicationId id) const NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            const nn::ncm::ApplicationId& a = d.applicationIds[i];
            if (a != nn::ncm::ApplicationId::GetInvalidId())
            {
                if (a == id)
                {
                    *outValue = d.IsRestricted(i);
                    return true;
                }
            }
        }
    }
    return false;
}

size_t FreeCommunicationApplicationSettings::GetRestrictedValues(nn::pctl::FreeCommunicationApplicationInfo* outArray, size_t offset, size_t count) const NN_NOEXCEPT
{
    if (count == 0)
    {
        return 0;
    }

    size_t actualCount = 0;
    for (const auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            const nn::ncm::ApplicationId& a = d.applicationIds[i];
            if (a != nn::ncm::ApplicationId::GetInvalidId())
            {
                if (offset > 0)
                {
                    --offset;
                }
                else
                {
                    outArray->applicationId = a;
                    outArray->isRestricted = d.IsRestricted(i);
                    ++outArray;
                    --count;
                    ++actualCount;
                    if (count == 0)
                    {
                        return actualCount;
                    }
                }
            }
        }
    }
    return actualCount;
}

bool FreeCommunicationApplicationSettings::SetRestrictedValueByApplicationId(const nn::pctl::FreeCommunicationApplicationInfo& info) NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            const nn::ncm::ApplicationId& a = d.applicationIds[i];
            if (a != nn::ncm::ApplicationId::GetInvalidId())
            {
                if (a == info.applicationId)
                {
                    d.SetRestricted(i, info.isRestricted);
                    return true;
                }
            }
        }
    }
    return false;
}

bool FreeCommunicationApplicationSettings::AddRestrictedValue(nn::ncm::ApplicationId id, bool defaultValue) NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            nn::ncm::ApplicationId* pId = &d.applicationIds[i];
            if (*pId == nn::ncm::ApplicationId::GetInvalidId())
            {
                *pId = id;
                d.SetRestricted(i, defaultValue);
                return true;
            }
            if (*pId == id)
            {
                return false;
            }
        }
    }
    return false;
}

bool FreeCommunicationApplicationSettings::DeleteRestrictedValueByApplicationId(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (auto& a : d.applicationIds)
        {
            if (a != nn::ncm::ApplicationId::GetInvalidId() && a == id)
            {
                a = nn::ncm::ApplicationId::GetInvalidId();
                return true;
            }
        }
    }
    return false;
}

void FreeCommunicationApplicationSettings::ResetValues() NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            if (d.applicationIds[i] != nn::ncm::ApplicationId::GetInvalidId())
            {
                d.SetRestricted(i, true);
            }
        }
    }
}

void FreeCommunicationApplicationSettings::Clear() NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (auto& a : d.applicationIds)
        {
            a = nn::ncm::ApplicationId::GetInvalidId();
        }
    }
}

bool FreeCommunicationApplicationSettings::UpdateDataFromList(const FreeCommunicationApplicationSettings& list,
    void (* unknownTitleCallback)(nn::ncm::ApplicationId id),
    void (* notTitleInListCallback)(nn::ncm::ApplicationId id)) NN_NOEXCEPT
{
    bool resultValue = false;
    for (auto& partialDataLhs : this->data)
    {
        partialDataLhs.SetAllItemNotCompared();
    }
    for (auto& partialDataRhs : list.data)
    {
        for (int indexRhs = 0; indexRhs < Partial::IdCountPerStructure; ++indexRhs)
        {
            auto& appIdRhs = partialDataRhs.applicationIds[indexRhs];
            if (appIdRhs == nn::ncm::ApplicationId::GetInvalidId())
            {
                break;
            }
            bool found = false;
            for (auto& partialDataLhs : this->data)
            {
                for (int indexLhs = 0; indexLhs < Partial::IdCountPerStructure; ++indexLhs)
                {
                    auto& appIdLhs = partialDataLhs.applicationIds[indexLhs];
                    if (appIdLhs == nn::ncm::ApplicationId::GetInvalidId())
                    {
                        break;
                    }
                    if (appIdLhs == appIdRhs)
                    {
                        // 1つでも異なっていればリスト不一致
                        bool rhsValue = partialDataRhs.IsRestricted(indexRhs);
                        if (partialDataLhs.IsRestricted(indexLhs) != rhsValue)
                        {
                            resultValue = true;
                            partialDataLhs.SetRestricted(indexLhs, rhsValue);
                        }
                        partialDataLhs.SetCompared(indexLhs, true);
                        found = true;
                        break;
                    }
                }
                if (found)
                {
                    break;
                }
            }
            // appIdRhs が this の中に見つからない(「list にあるが自データにないタイトル」)
            // → unknownTitleCallback を呼び出す
            if (!found)
            {
                if (unknownTitleCallback != nullptr)
                {
                    unknownTitleCallback(appIdRhs);
                }
            }
        }
    }
    // this の中に IsCompared が false なデータがある(「自データにはあるが list にないタイトル」)
    // → notTitleInListCallback を呼び出す
    if (notTitleInListCallback != nullptr)
    {
        for (auto& partialDataLhs : this->data)
        {
            for (int i = 0; i < Partial::IdCountPerStructure; ++i)
            {
                const auto& appId = partialDataLhs.applicationIds[i];
                if (appId == nn::ncm::ApplicationId::GetInvalidId())
                {
                    break;
                }
                if (!partialDataLhs.IsCompared(i))
                {
                    notTitleInListCallback(appId);
                }
            }
        }
    }
    return resultValue;
}

size_t ExemptApplicationSettings::GetAvailableCount() const NN_NOEXCEPT
{
    size_t count = 0;
    for (const auto& d : data)
    {
        for (const nn::ncm::ApplicationId& a : d.applicationIds)
        {
            if (a != nn::ncm::ApplicationId::GetInvalidId())
            {
                ++count;
            }
        }
    }
    return count;
}

bool ExemptApplicationSettings::IsAnyExemptionEnabled() const NN_NOEXCEPT
{
    for (const auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            if (d.applicationIds[i] != nn::ncm::ApplicationId::GetInvalidId())
            {
                if (d.IsExempted(i))
                {
                    return true;
                }
            }
        }
    }
    return false;
}

bool ExemptApplicationSettings::GetExemptedValueByApplicationId(bool* outValue, nn::ncm::ApplicationId id) const NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            const nn::ncm::ApplicationId& a = d.applicationIds[i];
            if (a != nn::ncm::ApplicationId::GetInvalidId())
            {
                if (a == id)
                {
                    *outValue = d.IsExempted(i);
                    return true;
                }
            }
        }
    }
    return false;
}

size_t ExemptApplicationSettings::GetExemptedValues(nn::pctl::ExemptApplicationInfo* outArray, size_t offset, size_t count) const NN_NOEXCEPT
{
    if (count == 0)
    {
        return 0;
    }

    size_t actualCount = 0;
    for (const auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            const nn::ncm::ApplicationId& a = d.applicationIds[i];
            if (a != nn::ncm::ApplicationId::GetInvalidId())
            {
                if (offset > 0)
                {
                    --offset;
                }
                else
                {
                    outArray->applicationId = a;
                    outArray->isExempted = d.IsExempted(i);
                    ++outArray;
                    --count;
                    ++actualCount;
                    if (count == 0)
                    {
                        return actualCount;
                    }
                }
            }
        }
    }
    return actualCount;
}

bool ExemptApplicationSettings::SetExemptedValueByApplicationId(const nn::pctl::ExemptApplicationInfo& info) NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            const nn::ncm::ApplicationId& a = d.applicationIds[i];
            if (a != nn::ncm::ApplicationId::GetInvalidId())
            {
                if (a == info.applicationId)
                {
                    d.SetExempted(i, info.isExempted);
                    return true;
                }
            }
        }
    }
    return false;
}

bool ExemptApplicationSettings::AddExemptedValue(nn::ncm::ApplicationId id, bool defaultValue) NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            nn::ncm::ApplicationId* pId = &d.applicationIds[i];
            if (*pId == nn::ncm::ApplicationId::GetInvalidId())
            {
                *pId = id;
                d.SetExempted(i, defaultValue);
                return true;
            }
            if (*pId == id)
            {
                return false;
            }
        }
    }
    return false;
}

bool ExemptApplicationSettings::DeleteExemptedValueByApplicationId(nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (auto& a : d.applicationIds)
        {
            if (a != nn::ncm::ApplicationId::GetInvalidId() && a == id)
            {
                a = nn::ncm::ApplicationId::GetInvalidId();
                return true;
            }
        }
    }
    return false;
}

void ExemptApplicationSettings::ResetValues() NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (int i = 0; i < Partial::IdCountPerStructure; ++i)
        {
            if (d.applicationIds[i] != nn::ncm::ApplicationId::GetInvalidId())
            {
                d.SetExempted(i, false);
            }
        }
    }
}

void ExemptApplicationSettings::Clear() NN_NOEXCEPT
{
    for (auto& d : data)
    {
        for (auto& a : d.applicationIds)
        {
            a = nn::ncm::ApplicationId::GetInvalidId();
        }
    }
}

bool ExemptApplicationSettings::UpdateDataFromList(const ExemptApplicationSettings& list,
    void(*unknownTitleCallback)(nn::ncm::ApplicationId id),
    void(*notTitleInListCallback)(nn::ncm::ApplicationId id)) NN_NOEXCEPT
{
    bool resultValue = false;
    for (auto& partialDataLhs : this->data)
    {
        partialDataLhs.SetAllItemNotCompared();
    }
    for (auto& partialDataRhs : list.data)
    {
        for (int indexRhs = 0; indexRhs < Partial::IdCountPerStructure; ++indexRhs)
        {
            auto& appIdRhs = partialDataRhs.applicationIds[indexRhs];
            if (appIdRhs == nn::ncm::ApplicationId::GetInvalidId())
            {
                break;
            }
            bool found = false;
            for (auto& partialDataLhs : this->data)
            {
                for (int indexLhs = 0; indexLhs < Partial::IdCountPerStructure; ++indexLhs)
                {
                    auto& appIdLhs = partialDataLhs.applicationIds[indexLhs];
                    if (appIdLhs == nn::ncm::ApplicationId::GetInvalidId())
                    {
                        break;
                    }
                    if (appIdLhs == appIdRhs)
                    {
                        // 1つでも異なっていればリスト不一致
                        bool rhsValue = partialDataRhs.IsExempted(indexRhs);
                        if (partialDataLhs.IsExempted(indexLhs) != rhsValue)
                        {
                            resultValue = true;
                            partialDataLhs.SetExempted(indexLhs, rhsValue);
                        }
                        partialDataLhs.SetCompared(indexLhs, true);
                        found = true;
                        break;
                    }
                }
                if (found)
                {
                    break;
                }
            }
            // appIdRhs が this の中に見つからない(「list にあるが自データにないタイトル」)
            // → unknownTitleCallback を呼び出す
            if (!found)
            {
                if (unknownTitleCallback != nullptr)
                {
                    unknownTitleCallback(appIdRhs);
                }
            }
        }
    }
    // this の中に IsCompared が false なデータがある(「自データにはあるが list にないタイトル」)
    // → notTitleInListCallback を呼び出す
    if (notTitleInListCallback != nullptr)
    {
        for (auto& partialDataLhs : this->data)
        {
            for (int i = 0; i < Partial::IdCountPerStructure; ++i)
            {
                const auto& appId = partialDataLhs.applicationIds[i];
                if (appId == nn::ncm::ApplicationId::GetInvalidId())
                {
                    break;
                }
                if (!partialDataLhs.IsCompared(i))
                {
                    notTitleInListCallback(appId);
                }
            }
        }
    }
    return resultValue;
}

bool ExemptApplicationSettings::GetExemptedValueByIndex(nn::pctl::ExemptApplicationInfo* pOutInfo, size_t index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(index, GetAvailableCount());

    auto d = data[index / Partial::IdCountPerStructure];
    int i = index % Partial::IdCountPerStructure;

    if (d.applicationIds[i] != nn::ncm::ApplicationId::GetInvalidId())
    {
        pOutInfo->applicationId = d.applicationIds[i];
        pOutInfo->isExempted = d.IsExempted(i);
        return true;
    }

    return false;
}

}}}}}
