﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/os/os_Thread.h>
#include <nn/pdm/pdm_QueryApi.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceFactory.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/socket.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/time/time_Api.h>
#include <curl/curl.h>

namespace nn { namespace pctl { namespace detail { namespace service {

namespace
{
    std::aligned_storage<sizeof(ServiceWatcher)>::type g_ServiceWatcherStorage;
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_NetworkManagerStack[watcher::NetworkManager::ThreadStackSize];
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_WatcherEventManagerStack[watcher::WatcherEventManager::ThreadStackSize];
}

namespace
{
    // ソケットの利用数・同時利用数。
    // ネットワークを扱うスレッドは1つに絞っているが、
    // closedなソケットが一定期間残る可能性があるため2を指定
    static const int SocketCount = 2;
    static const int SocketConcurrencyCount = SocketCount;
}

ServiceWatcher* g_pWatcher = nullptr;

void InitializeWatcher() NN_NOEXCEPT
{
    // Socket が要求するメモリと設定情報を保持する型インスタンス
    // (内部で大きなバッファーも持つのでサイズが大きくなる)
    // SystemConfigLightDefaultWithMemory は内部に NN_ALIGNAS なメンバ変数を持つので
    // インスタンスを直接宣言する
    typedef nn::socket::SystemConfigLightDefaultWithMemory<SocketCount, 0> PctlSocketConfig;
    NN_FUNCTION_LOCAL_STATIC(PctlSocketConfig, g_SocketConfig, (SocketConcurrencyCount));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

    nn::nifm::InitializeSystem();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::InitializeForSystem());
    nn::pdm::InitializeForQuery();
#endif

    nn::account::InitializeForSystemService();

    NN_ABORT_UNLESS_EQUAL(CURLE_OK, curl_global_init(CURL_GLOBAL_ALL));

    g_pWatcher = new(&g_ServiceWatcherStorage) ServiceWatcher;
    g_pWatcher->InitializeAndStartThread();
}

void FinalizeWatcher() NN_NOEXCEPT
{
    if (g_pWatcher == nullptr)
    {
        return;
    }

    g_pWatcher->~ServiceWatcher();
    g_pWatcher = nullptr;

    curl_global_cleanup();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::pdm::FinalizeForQuery();
    nn::npns::FinalizeForSystem();
#endif
    nn::time::Finalize();
    //nn::nifm::FinalizeSystem();
    nn::socket::Finalize();
}

bool IsWatcherAvailable() NN_NOEXCEPT
{
    return g_pWatcher != nullptr;
}

ServiceWatcher::ServiceWatcher() NN_NOEXCEPT :
    m_NetworkManager(&g_NetworkManagerStack, watcher::NetworkManager::ThreadStackSize),
    m_WatcherEventManager(&g_WatcherEventManagerStack, watcher::WatcherEventManager::ThreadStackSize),
    m_WatcherEventStorage()
{
    m_PairingInfoData.deviceId = 0;
    m_PairingInfoData.countOwners = 0;
    m_PairingInfoData.arrayOwners = nullptr;
    m_PairingInfoData.lastTickToUse = 0;
}

ServiceWatcher::~ServiceWatcher() NN_NOEXCEPT
{
    if (m_PairingInfoData.arrayOwners != nullptr)
    {
        FreeMemoryBlock(m_PairingInfoData.arrayOwners);
    }
}

void ServiceWatcher::InitializeAndStartThread() NN_NOEXCEPT
{
    m_WatcherEventStorage.Initialize();
    m_NetworkManager.LoadConfig();
    m_WatcherEventManager.Initialize();

    m_NetworkManager.StartThread();
    m_WatcherEventManager.StartThread();
}

}}}}
