﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/json/pctl_JsonDataHandler.h>
#include <nn/util/util_IntUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace json {

bool JsonDataHandler::StartObject() NN_NOEXCEPT
{
    if (m_CurrentDepth >= MaxDepth - 1)
    {
        return false;
    }
    if (TryToIgnoreObjectOrArray(true))
    {
        return true;
    }
    auto& info = m_DepthInfo[m_CurrentDepth];
    int pos = info.currentPosition;
    if (pos >= info.subArrayLength)
    {
        return false;
    }

    {
        auto& data = info.subArray[pos];
        if ((data.type & ExpectType::ExpectType_TypeMask) != ExpectType::ExpectType_ObjectBegin)
        {
            return false;
        }
        if (data.data != nullptr)
        {
            if (!reinterpret_cast<HandleObjectValueFunction>(data.data)(m_Param, info.index))
            {
                return false;
            }
        }
    }
    // 対応する ObjectEnd を探す
    int nestCount = 0;
    ++pos;
    while (pos < info.subArrayLength)
    {
        uint8_t type = (info.subArray[pos].type & ExpectType::ExpectType_TypeMask);
        if (type == ExpectType::ExpectType_ObjectEnd || type == ExpectType::ExpectType_ArrayEnd)
        {
            if (nestCount == 0)
            {
                NN_SDK_ASSERT(type == ExpectType::ExpectType_ObjectEnd, "Invalid data definition: ObjectBegin found but no ObjectEnd found (confused for ArrayEnd?)");
                auto& infoChild = m_DepthInfo[++m_CurrentDepth];
                infoChild.subArray = &info.subArray[info.currentPosition + 1];
                infoChild.subArrayLength = pos - (info.currentPosition + 1);
                infoChild.currentPosition = 0;
                infoChild.index = 0;
                infoChild.isObject = true;
                infoChild.isRepeat = false;
                return true;
            }
            // 厳密な対応はここでは見ない
            --nestCount;
        }
        else if (type == ExpectType::ExpectType_ObjectBegin || type == ExpectType::ExpectType_ArrayBegin)
        {
            if (nestCount == MaxDepth)
            {
                // 入れ子が深すぎる
                NN_SDK_ASSERT(false, "Invalid data definition: Too much recursion");
                return false;
            }
            ++nestCount;
        }
        ++pos;
    }
    NN_SDK_ASSERT(false, "Invalid data definition: ObjectBegin found but no ObjectEnd found");
    return true;
}

bool JsonDataHandler::EndObject(RAPIDJSON_NAMESPACE::SizeType numObjects) NN_NOEXCEPT
{
    NN_UNUSED(numObjects);

    auto& info = m_DepthInfo[m_CurrentDepth];
    if (!TryToIgnoreKeyValue())
    {
        int pos = info.currentPosition;
        if (pos >= info.subArrayLength)
        {
            if (m_CurrentDepth == 0)
            {
                return false;
            }
        }

        for (; pos < info.subArrayLength - 1; )
        {
            auto& data = info.subArray[pos];
            auto& dataValue = info.subArray[pos + 1];
            auto dataValueType = (dataValue.type & ExpectType::ExpectType_TypeMask);
            NN_SDK_ASSERT(((data.type & ExpectType::ExpectType_TypeMask) == ExpectType::ExpectType_Key) ||
                ((data.type & ExpectType::ExpectType_TypeMask) == ExpectType::ExpectType_AnyKey));
            NN_SDK_ASSERT(dataValueType != ExpectType::ExpectType_Key);
            NN_UNUSED(data);
            NN_UNUSED(dataValue);
            NN_UNUSED(dataValueType);
            pos = CalculateNextPosition(pos + 1);
            if (pos < 0)
            {
                NN_SDK_ASSERT(false, "Invalid data definition: unknown data after key data");
                return false;
            }
        }
    }
    // 階層を1つ上にセットする
    --m_CurrentDepth;
    // 直前の階層で消費した要素数を加える
    UpdateCurrentPositionForCurrentDepth(m_DepthInfo[m_CurrentDepth].currentPosition + info.subArrayLength + 1);

    return true;
}

bool JsonDataHandler::StartArray() NN_NOEXCEPT
{
    if (m_CurrentDepth >= MaxDepth - 1)
    {
        return false;
    }
    if (TryToIgnoreObjectOrArray(false))
    {
        return true;
    }
    auto& info = m_DepthInfo[m_CurrentDepth];
    int pos = info.currentPosition;
    if (pos >= info.subArrayLength)
    {
        return false;
    }

    bool isRepeat;
    {
        auto& data = info.subArray[pos];
        if ((data.type & ExpectType::ExpectType_TypeMask) != ExpectType::ExpectType_ArrayBegin)
        {
            return false;
        }
        isRepeat = ((data.type & ExpectType::ExpectType_ArrayRepeatValues) != 0);
        if (data.data != nullptr)
        {
            if (!reinterpret_cast<HandleArrayValueFunction>(data.data)(m_Param, info.index))
            {
                return false;
            }
        }
    }
    // 対応する ArrayEnd を探す
    int nestCount = 0;
    ++pos;
    while (pos < info.subArrayLength)
    {
        uint8_t type = (info.subArray[pos].type & ExpectType::ExpectType_TypeMask);
        if (type == ExpectType::ExpectType_ObjectEnd || type == ExpectType::ExpectType_ArrayEnd)
        {
            if (nestCount == 0)
            {
                NN_SDK_ASSERT(type == ExpectType::ExpectType_ArrayEnd, "Invalid data definition: ArrayBegin found but no ArrayEnd found (confused for ObjectEnd?)");
                auto& infoChild = m_DepthInfo[++m_CurrentDepth];
                infoChild.subArray = &info.subArray[info.currentPosition + 1];
                infoChild.subArrayLength = pos - (info.currentPosition + 1);
                infoChild.currentPosition = 0;
                infoChild.index = 0;
                infoChild.isObject = false;
                infoChild.isRepeat = isRepeat;
                return true;
            }
            // 厳密な対応はここでは見ない
            --nestCount;
        }
        else if (type == ExpectType::ExpectType_ObjectBegin || type == ExpectType::ExpectType_ArrayBegin)
        {
            if (nestCount == MaxDepth)
            {
                // 入れ子が深すぎる
                NN_SDK_ASSERT(false, "Invalid data definition: Too much recursion");
                return false;
            }
            ++nestCount;
        }
        ++pos;
    }
    NN_SDK_ASSERT(false, "Invalid data definition: ArrayBegin found but no ArrayEnd found");
    return true;
}

bool JsonDataHandler::EndArray(RAPIDJSON_NAMESPACE::SizeType numObjects) NN_NOEXCEPT
{
    NN_UNUSED(numObjects);

    auto& info = m_DepthInfo[m_CurrentDepth];
    if (!TryToIgnoreKeyValue())
    {
        if (info.currentPosition >= info.subArrayLength)
        {
            if (m_CurrentDepth == 0)
            {
                return false;
            }
        }
    }
    // 階層を1つ上にセットする
    --m_CurrentDepth;
    // 直前の階層で消費した要素数を加える
    UpdateCurrentPositionForCurrentDepth(m_DepthInfo[m_CurrentDepth].currentPosition + info.subArrayLength + 1);

    return true;
}

bool JsonDataHandler::Key(const char* key, RAPIDJSON_NAMESPACE::SizeType length, bool) NN_NOEXCEPT
{
    if (TryToIgnoreKeyValue())
    {
        return true;
    }
    auto& info = m_DepthInfo[m_CurrentDepth];
    int pos = info.currentPosition;
    if (pos >= info.subArrayLength)
    {
        return false;
    }

    if (!info.isObject)
    {
        return false;
    }
    bool allowUnknownKey = false;
    {
        // 必ず Object の階層下にあるはず
        NN_SDK_ASSERT(m_CurrentDepth > 0 && info.isObject);
        auto& parentDepthInfo = m_DepthInfo[m_CurrentDepth - 1];
        auto& data = parentDepthInfo.subArray[parentDepthInfo.currentPosition];
        NN_SDK_ASSERT((data.type & ExpectType::ExpectType_TypeMask) == ExpectType::ExpectType_ObjectBegin);
        allowUnknownKey = ((data.type & ExpectType::ExpectType_AllowAnyKey) != 0);
    }

    int posAnyKey = -1;
    for (; pos < info.subArrayLength; )
    {
        auto& data = info.subArray[pos];
        auto& dataValue = info.subArray[pos + 1];
        auto dataValueType = (dataValue.type & ExpectType::ExpectType_TypeMask);
        if ((data.type & ExpectType::ExpectType_TypeMask) == ExpectType::ExpectType_AnyKey)
        {
            NN_SDK_ASSERT(posAnyKey == -1, "Unexpected: multiple ANY_KEY definition is not allowed");
            posAnyKey = pos;
        }
        else
        {
            NN_SDK_ASSERT((data.type & ExpectType::ExpectType_TypeMask) == ExpectType::ExpectType_Key);
            NN_SDK_ASSERT(dataValueType != ExpectType::ExpectType_Key);
            NN_UNUSED(dataValueType);
            {
                const char* k = reinterpret_cast<const char*>(data.data);
                if (std::strncmp(k, key, length) == 0 && k[length] == 0)
                {
                    info.currentPosition = pos + 1;
                    return true;
                }
            }
        }
        pos = CalculateNextPosition(pos + 1);
        if (pos < 0)
        {
            NN_SDK_ASSERT(false, "Invalid data definition: unknown data after key data");
            break;
        }
    }
    if (posAnyKey != -1)
    {
        auto& data = info.subArray[posAnyKey];
        bool ignore = false;
        if (!reinterpret_cast<HandleAnyKeyFunction>(data.data)(&ignore, m_Param, info.index, key, static_cast<size_t>(length)))
        {
            return false;
        }
        if (!ignore)
        {
            info.currentPosition = posAnyKey + 1;
            return true;
        }
    }
    if (allowUnknownKey)
    {
        SetIgnoreNextData();
        return true;
    }
    return false;
}

bool JsonDataHandler::Null() NN_NOEXCEPT
{
    if (TryToIgnoreKeyValue())
    {
        return true;
    }
    auto& info = m_DepthInfo[m_CurrentDepth];
    int pos = info.currentPosition;
    if (pos >= info.subArrayLength)
    {
        return false;
    }

    auto& data = info.subArray[pos];
    auto dataType = (data.type & ExpectType::ExpectType_TypeMask);
    if (dataType == ExpectType::ExpectType_ObjectBegin || dataType == ExpectType::ExpectType_ArrayBegin)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) == 0)
        {
            return false;
        }
    }
    else if (dataType == ExpectType::ExpectType_String)
    {
        // pass null to the callback
        if (!reinterpret_cast<HandleStringValueFunction>(data.data)(m_Param, info.index, nullptr, 0))
        {
            return false;
        }
    }
    else if (dataType == ExpectType::ExpectType_Int64)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) == 0)
        {
            return false;
        }
        // pass nn::util::nullopt to the callback
        if (!reinterpret_cast<HandleInt64OrNullValueFunction>(data.data)(m_Param, info.index, nn::util::nullopt))
        {
            return false;
        }
    }
    else if (dataType == ExpectType::ExpectType_Uint64)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) == 0)
        {
            return false;
        }
        // pass nn::util::nullopt to the callback
        if (!reinterpret_cast<HandleUint64OrNullValueFunction>(data.data)(m_Param, info.index, nn::util::nullopt))
        {
            return false;
        }
    }
    else if (dataType == ExpectType::ExpectType_Double)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) == 0)
        {
            return false;
        }
        // pass nn::util::nullopt to the callback
        if (!reinterpret_cast<HandleDoubleOrNullValueFunction>(data.data)(m_Param, info.index, nn::util::nullopt))
        {
            return false;
        }
    }
    else if (dataType == ExpectType::ExpectType_Boolean)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) == 0)
        {
            return false;
        }
        // pass nn::util::nullopt to the callback
        if (!reinterpret_cast<HandleBooleanOrNullValueFunction>(data.data)(m_Param, info.index, nn::util::nullopt))
        {
            return false;
        }
    }
    else
    {
        // null is not allowed
        return false;
    }

    pos = CalculateNextPosition(pos);
    if (pos < 0)
    {
        NN_SDK_ASSERT(false, "Invalid data definition: unexpected termination for expect data");
        return false;
    }
    // 引数に End の位置または現在の位置を渡す
    UpdateCurrentPositionForCurrentDepth(pos - 1);
    return true;
}

bool JsonDataHandler::Bool(bool value) NN_NOEXCEPT
{
    if (TryToIgnoreKeyValue())
    {
        return true;
    }
    auto& info = m_DepthInfo[m_CurrentDepth];
    int pos = info.currentPosition;
    if (pos >= info.subArrayLength)
    {
        return false;
    }

    auto& data = info.subArray[pos];
    auto dataType = (data.type & ExpectType::ExpectType_TypeMask);
    if (dataType == ExpectType::ExpectType_Boolean)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleBooleanOrNullValueFunction>(data.data)(m_Param, info.index, nn::util::optional<bool>(value)))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleBooleanValueFunction>(data.data)(m_Param, info.index, value))
            {
                return false;
            }
        }
    }
    else
    {
        // disallow value
        return false;
    }

    pos = CalculateNextPosition(pos);
    if (pos < 0)
    {
        NN_SDK_ASSERT(false, "Invalid data definition: unexpected termination for expect data");
        return false;
    }
    // 引数に End の位置または現在の位置を渡す
    UpdateCurrentPositionForCurrentDepth(pos - 1);
    return true;
}

bool JsonDataHandler::Int(int32_t value) NN_NOEXCEPT
{
    if (TryToIgnoreKeyValue())
    {
        return true;
    }
    auto& info = m_DepthInfo[m_CurrentDepth];
    int pos = info.currentPosition;
    if (pos >= info.subArrayLength)
    {
        return false;
    }

    auto& data = info.subArray[pos];
    auto dataType = (data.type & ExpectType::ExpectType_TypeMask);
    if (dataType == ExpectType::ExpectType_Int64)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleInt64OrNullValueFunction>(data.data)(m_Param, info.index, nn::util::optional<int64_t>(value)))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleInt64ValueFunction>(data.data)(m_Param, info.index, value))
            {
                return false;
            }
        }
    }
    else if (dataType == ExpectType::ExpectType_Uint64)
    {
        // disallow if the value is negative
        if (value < 0)
        {
            return false;
        }
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleUint64OrNullValueFunction>(data.data)(m_Param, info.index,
                nn::util::optional<uint64_t>(static_cast<uint64_t>(value))))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleUint64ValueFunction>(data.data)(m_Param, info.index, static_cast<uint64_t>(value)))
            {
                return false;
            }
        }
    }
    else if (dataType == ExpectType::ExpectType_Double)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleDoubleOrNullValueFunction>(data.data)(m_Param, info.index,
                nn::util::optional<double>(static_cast<double>(value))))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleDoubleValueFunction>(data.data)(m_Param, info.index, static_cast<double>(value)))
            {
                return false;
            }
        }
    }
    else
    {
        // disallow value
        return false;
    }

    pos = CalculateNextPosition(pos);
    if (pos < 0)
    {
        NN_SDK_ASSERT(false, "Invalid data definition: unexpected termination for expect data");
        return false;
    }
    // 引数に End の位置または現在の位置を渡す
    UpdateCurrentPositionForCurrentDepth(pos - 1);
    return true;
}

bool JsonDataHandler::Uint(uint32_t value) NN_NOEXCEPT
{
    if (TryToIgnoreKeyValue())
    {
        return true;
    }
    auto& info = m_DepthInfo[m_CurrentDepth];
    int pos = info.currentPosition;
    if (pos >= info.subArrayLength)
    {
        return false;
    }

    auto& data = info.subArray[pos];
    auto dataType = (data.type & ExpectType::ExpectType_TypeMask);
    if (dataType == ExpectType::ExpectType_Uint64)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleUint64OrNullValueFunction>(data.data)(m_Param, info.index, nn::util::optional<uint64_t>(value)))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleUint64ValueFunction>(data.data)(m_Param, info.index, value))
            {
                return false;
            }
        }
    }
    else if (dataType == ExpectType::ExpectType_Int64)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleInt64OrNullValueFunction>(data.data)(m_Param, info.index,
                nn::util::optional<int64_t>(static_cast<int64_t>(value))))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleInt64ValueFunction>(data.data)(m_Param, info.index, static_cast<int64_t>(value)))
            {
                return false;
            }
        }
    }
    else if (dataType == ExpectType::ExpectType_Double)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleDoubleOrNullValueFunction>(data.data)(m_Param, info.index,
                nn::util::optional<double>(static_cast<double>(value))))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleDoubleValueFunction>(data.data)(m_Param, info.index, static_cast<double>(value)))
            {
                return false;
            }
        }
    }
    else
    {
        // disallow value
        return false;
    }

    pos = CalculateNextPosition(pos);
    if (pos < 0)
    {
        NN_SDK_ASSERT(false, "Invalid data definition: unexpected termination for expect data");
        return false;
    }
    // 引数に End の位置または現在の位置を渡す
    UpdateCurrentPositionForCurrentDepth(pos - 1);
    return true;
}

bool JsonDataHandler::Int64(int64_t value) NN_NOEXCEPT
{
    if (TryToIgnoreKeyValue())
    {
        return true;
    }
    auto& info = m_DepthInfo[m_CurrentDepth];
    int pos = info.currentPosition;
    if (pos >= info.subArrayLength)
    {
        return false;
    }

    auto& data = info.subArray[pos];
    auto dataType = (data.type & ExpectType::ExpectType_TypeMask);
    if (dataType == ExpectType::ExpectType_Int64)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleInt64OrNullValueFunction>(data.data)(m_Param, info.index, nn::util::optional<int64_t>(value)))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleInt64ValueFunction>(data.data)(m_Param, info.index, value))
            {
                return false;
            }
        }
    }
    else if (dataType == ExpectType::ExpectType_Uint64)
    {
        // disallow if the value is not representable
        if (!nn::util::IsIntValueRepresentable<uint64_t, int64_t>(value))
        {
            return false;
        }
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleUint64OrNullValueFunction>(data.data)(m_Param, info.index,
                nn::util::optional<uint64_t>(static_cast<uint64_t>(value))))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleUint64ValueFunction>(data.data)(m_Param, info.index, static_cast<uint64_t>(value)))
            {
                return false;
            }
        }
    }
    else if (dataType == ExpectType::ExpectType_Double)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleDoubleOrNullValueFunction>(data.data)(m_Param, info.index,
                nn::util::optional<double>(static_cast<double>(value))))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleDoubleValueFunction>(data.data)(m_Param, info.index, static_cast<double>(value)))
            {
                return false;
            }
        }
    }
    else
    {
        // disallow value
        return false;
    }

    pos = CalculateNextPosition(pos);
    if (pos < 0)
    {
        NN_SDK_ASSERT(false, "Invalid data definition: unexpected termination for expect data");
        return false;
    }
    // 引数に End の位置または現在の位置を渡す
    UpdateCurrentPositionForCurrentDepth(pos - 1);
    return true;
}

bool JsonDataHandler::Uint64(uint64_t value) NN_NOEXCEPT
{
    if (TryToIgnoreKeyValue())
    {
        return true;
    }
    auto& info = m_DepthInfo[m_CurrentDepth];
    int pos = info.currentPosition;
    if (pos >= info.subArrayLength)
    {
        return false;
    }

    auto& data = info.subArray[pos];
    auto dataType = (data.type & ExpectType::ExpectType_TypeMask);
    if (dataType == ExpectType::ExpectType_Uint64)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleUint64OrNullValueFunction>(data.data)(m_Param, info.index, nn::util::optional<uint64_t>(value)))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleUint64ValueFunction>(data.data)(m_Param, info.index, value))
            {
                return false;
            }
        }
    }
    else if (dataType == ExpectType::ExpectType_Int64)
    {
        // disallow if the value is not representable
        if (!nn::util::IsIntValueRepresentable<int64_t, uint64_t>(value))
        {
            return false;
        }
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleInt64OrNullValueFunction>(data.data)(m_Param, info.index,
                nn::util::optional<int64_t>(static_cast<int64_t>(value))))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleInt64ValueFunction>(data.data)(m_Param, info.index, static_cast<int64_t>(value)))
            {
                return false;
            }
        }
    }
    else if (dataType == ExpectType::ExpectType_Double)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleDoubleOrNullValueFunction>(data.data)(m_Param, info.index,
                nn::util::optional<double>(static_cast<double>(value))))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleDoubleValueFunction>(data.data)(m_Param, info.index, static_cast<double>(value)))
            {
                return false;
            }
        }
    }
    else
    {
        // disallow value
        return false;
    }

    pos = CalculateNextPosition(pos);
    if (pos < 0)
    {
        NN_SDK_ASSERT(false, "Invalid data definition: unexpected termination for expect data");
        return false;
    }
    // 引数に End の位置または現在の位置を渡す
    UpdateCurrentPositionForCurrentDepth(pos - 1);
    return true;
}

bool JsonDataHandler::Double(double value) NN_NOEXCEPT
{
    if (TryToIgnoreKeyValue())
    {
        return true;
    }
    auto& info = m_DepthInfo[m_CurrentDepth];
    int pos = info.currentPosition;
    if (pos >= info.subArrayLength)
    {
        return false;
    }

    auto& data = info.subArray[pos];
    auto dataType = (data.type & ExpectType::ExpectType_TypeMask);
    if (dataType == ExpectType::ExpectType_Double)
    {
        if ((data.type & ExpectType::ExpectType_AllowNull) != 0)
        {
            // pass to the callback
            if (!reinterpret_cast<HandleDoubleOrNullValueFunction>(data.data)(m_Param, info.index,
                nn::util::optional<double>(value)))
            {
                return false;
            }
        }
        else
        {
            // pass to the callback
            if (!reinterpret_cast<HandleDoubleValueFunction>(data.data)(m_Param, info.index, value))
            {
                return false;
            }
        }
    }
    else
    {
        // disallow value (even if it allows Int64/Uint64)
        return false;
    }

    pos = CalculateNextPosition(pos);
    if (pos < 0)
    {
        NN_SDK_ASSERT(false, "Invalid data definition: unexpected termination for expect data");
        return false;
    }
    // 引数に End の位置または現在の位置を渡す
    UpdateCurrentPositionForCurrentDepth(pos - 1);
    return true;
}

bool JsonDataHandler::String(const char* value, RAPIDJSON_NAMESPACE::SizeType length, bool) NN_NOEXCEPT
{
    if (TryToIgnoreKeyValue())
    {
        return true;
    }
    auto& info = m_DepthInfo[m_CurrentDepth];
    int pos = info.currentPosition;
    if (pos >= info.subArrayLength)
    {
        return false;
    }

    auto& data = info.subArray[pos];
    auto dataType = (data.type & ExpectType::ExpectType_TypeMask);
    if (dataType == ExpectType::ExpectType_String)
    {
        // pass to the callback
        if (!reinterpret_cast<HandleStringValueFunction>(data.data)(m_Param, info.index, value, static_cast<size_t>(length)))
        {
            return false;
        }
    }
    else
    {
        // disallow value
        return false;
    }

    pos = CalculateNextPosition(pos);
    if (pos < 0)
    {
        NN_SDK_ASSERT(false, "Invalid data definition: unexpected termination for expect data");
        return false;
    }
    // 引数に End の位置または現在の位置を渡す
    UpdateCurrentPositionForCurrentDepth(pos - 1);
    return true;
}


void JsonDataHandler::UpdateCurrentPositionForCurrentDepth(int lastPosition) NN_NOEXCEPT
{
    auto& info = m_DepthInfo[m_CurrentDepth];
    // 配列の場合は次の位置、オブジェクトの場合は先頭位置にセットする
    // (オブジェクトは先頭位置からキーを探索するため)
    if (info.isObject)
    {
        info.currentPosition = 0;
    }
    else
    {
        // 繰り返しの場合は最初に戻す
        if (info.isRepeat)
        {
            info.currentPosition = 0;
        }
        else
        {
            info.currentPosition = lastPosition + 1;
        }
    }
    ++info.index;
}

int JsonDataHandler::CalculateNextPosition(int startPosition) const NN_NOEXCEPT
{
    auto& info = m_DepthInfo[m_CurrentDepth];
    auto& data = info.subArray[startPosition];
    auto dataType = (data.type & ExpectType::ExpectType_TypeMask);
    if (dataType != ExpectType::ExpectType_ObjectBegin && dataType != ExpectType::ExpectType_ArrayBegin)
    {
        return startPosition + 1;
    }
    ++startPosition;
    // 対応する ObjectEnd/ArrayEnd を探す
    int nestCount = 0;
    while (startPosition < info.subArrayLength)
    {
        uint8_t type = (info.subArray[startPosition].type & ExpectType::ExpectType_TypeMask);
        if (type == ExpectType::ExpectType_ObjectEnd || type == ExpectType::ExpectType_ArrayEnd)
        {
            if (nestCount == 0)
            {
                // Begin と End の種類が一致していない場合は -1 を返す
                // (「!=」の lhs と rhs がそれぞれ Object である場合は true、Array である場合は false になる)
                if ((dataType == ExpectType::ExpectType_ObjectBegin) != (type == ExpectType::ExpectType_ObjectEnd))
                {
                    return -1;
                }
                // End の次の位置を返す
                return startPosition + 1;
            }
            // 厳密な対応はここでは見ない
            --nestCount;
        }
        else if (type == ExpectType::ExpectType_ObjectBegin || type == ExpectType::ExpectType_ArrayBegin)
        {
            if (nestCount == MaxDepth)
            {
                // 入れ子が深すぎる
                NN_SDK_ASSERT(false, "Invalid data definition: Too much recursion");
                return -1;
            }
            ++nestCount;
        }
        ++startPosition;
    }
    return info.subArrayLength;
}

bool JsonDataHandler::TryToIgnoreObjectOrArray(bool isObject) NN_NOEXCEPT
{
    for (int depth = 0; depth <= m_CurrentDepth; ++depth)
    {
        auto& info = m_DepthInfo[depth];
        if (info.isObject)
        {
            // Objectであり、現在の読み取り位置が末尾にあれば無視状態になっているとみなして
            // 階層のみ1つ下に進める
            if (info.currentPosition == info.subArrayLength)
            {
                NN_SDK_ASSERT(m_CurrentDepth < MaxDepth - 1);
                auto& infoChild = m_DepthInfo[++m_CurrentDepth];
                infoChild.subArray = nullptr;
                infoChild.subArrayLength = 0;
                infoChild.currentPosition = 0;
                infoChild.index = 0;
                infoChild.isObject = isObject;
                infoChild.isRepeat = false;
                return true;
            }
        }
    }
    return false;
}

bool JsonDataHandler::TryToIgnoreKeyValue() NN_NOEXCEPT
{
    for (int depth = 0; depth <= m_CurrentDepth; ++depth)
    {
        auto& info = m_DepthInfo[depth];
        if (info.isObject)
        {
            // Objectであり、現在の読み取り位置が末尾にあれば無視状態になっているとみなす
            // 逆に末尾でなければ無視されない
            if (info.currentPosition != info.subArrayLength)
            {
                continue;
            }
            // 階層が現在の位置であれば無視状態を解除する
            if (depth == m_CurrentDepth)
            {
                info.currentPosition = 0;
                ++info.index;
            }
            return true;
        }
    }
    return false;
}

void JsonDataHandler::SetIgnoreNextData() NN_NOEXCEPT
{
    auto& info = m_DepthInfo[m_CurrentDepth];
    info.currentPosition = info.subArrayLength;
}

}}}}}
