﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/common/pctl_Constants.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfoTypes.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/system/settings_Region.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
// 暫定
#include <nn/time/time_StandardUserSystemClock.h>
#else
#include <nn/time/time_StandardNetworkSystemClock.h>
#endif
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace common {

// MaxSystemVersionStringLength の長さをここでチェック
NN_STATIC_ASSERT(MaxSystemVersionStringLength == std::extent<decltype(nn::settings::system::FirmwareVersion::displayVersion)>::value);

namespace
{
    // 値をキャッシュしておく(スタック領域節約も兼ねる)
    bool g_IsVersionRetrieved = false;
    nn::settings::system::FirmwareVersion g_FirmwareVersion;

    RegionForDefaultRatingOrganization ConvertRegionValue(nn::settings::system::RegionCode regionCode) NN_NOEXCEPT
    {
        switch (regionCode)
        {
            case nn::settings::system::RegionCode::RegionCode_Japan:
                return RegionForDefaultRatingOrganization::Japan;
            case nn::settings::system::RegionCode::RegionCode_Usa:
                return RegionForDefaultRatingOrganization::NorthAmerica;
            case nn::settings::system::RegionCode::RegionCode_Europe:
                return RegionForDefaultRatingOrganization::Europe;
            case nn::settings::system::RegionCode::RegionCode_Australia:
                return RegionForDefaultRatingOrganization::Australia;
            default:
                return RegionForDefaultRatingOrganization::Others;
        }
    }

    LanguageForDefaultRatingOrganization ConvertLanguageValue(nn::settings::Language language) NN_NOEXCEPT
    {
        switch (language)
        {
            case nn::settings::Language::Language_AmericanEnglish:
            case nn::settings::Language::Language_BritishEnglish:
                return LanguageForDefaultRatingOrganization::English;
            case nn::settings::Language::Language_CanadianFrench:
            case nn::settings::Language::Language_French:
                return LanguageForDefaultRatingOrganization::French;
            case nn::settings::Language::Language_LatinAmericanSpanish:
            case nn::settings::Language::Language_Spanish:
                return LanguageForDefaultRatingOrganization::Spanish;
            case nn::settings::Language::Language_Italian:
                return LanguageForDefaultRatingOrganization::Italian;
            case nn::settings::Language::Language_Dutch:
                return LanguageForDefaultRatingOrganization::Dutch;
            case nn::settings::Language::Language_Japanese:
                return LanguageForDefaultRatingOrganization::Japanese;
            case nn::settings::Language::Language_German:
                return LanguageForDefaultRatingOrganization::German;
            case nn::settings::Language::Language_Portuguese:
                return LanguageForDefaultRatingOrganization::Portuguese;
            case nn::settings::Language::Language_Russian:
                return LanguageForDefaultRatingOrganization::Russian;
            case nn::settings::Language::Language_Taiwanese: // (TraditionalChinese として扱う)
            case nn::settings::Language::Language_TraditionalChinese:
                return LanguageForDefaultRatingOrganization::TraditionalChinese;
            case nn::settings::Language::Language_Chinese: // (SimplifiedChinese として扱う)
            case nn::settings::Language::Language_SimplifiedChinese:
                return LanguageForDefaultRatingOrganization::SimplifiedChinese;
            case nn::settings::Language::Language_Korean:
                return LanguageForDefaultRatingOrganization::Korean;
            default:
                return LanguageForDefaultRatingOrganization::Others;
        }
    }

    void GetFirmwareVersionImpl() NN_NOEXCEPT
    {
        if (g_IsVersionRetrieved)
        {
            return;
        }
        nn::settings::system::GetFirmwareVersion(&g_FirmwareVersion);
        g_IsVersionRetrieved = true;
    }

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    bool g_IsUserTimeEnabled = false;
    nn::time::PosixTime g_UserTime;
#endif
}

bool GetSystemLanguage(nn::settings::Language* outLanguage) NN_NOEXCEPT
{
    nn::settings::LanguageCode languageCode;
    nn::settings::LanguageCode* pAvailableLanguageCodes;
    int countLanguageCodes;

    nn::settings::GetLanguageCode(&languageCode);

    // LanguageCode -> Language の変換は自力で行う必要がある
    countLanguageCodes = nn::settings::GetAvailableLanguageCodeCount();
    pAvailableLanguageCodes = static_cast<nn::settings::LanguageCode*>(
        AllocateMemoryBlock(sizeof(nn::settings::LanguageCode) * countLanguageCodes)
        );
    // LanguageCode のサイズは小さいので割り当て失敗はよほどメモリが足りないとき
    NN_ABORT_UNLESS(pAvailableLanguageCodes != nullptr, "Cannot allocate language-code buffer");
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(pAvailableLanguageCodes);
    };
    nn::settings::GetAvailableLanguageCodes(pAvailableLanguageCodes, countLanguageCodes);

    for (int i = 0; i < countLanguageCodes; ++i)
    {
        if (languageCode == pAvailableLanguageCodes[i])
        {
            *outLanguage = static_cast<nn::settings::Language>(i);
            return true;
        }
    }
    return false;
}

nn::ns::RatingOrganization GetDefaultRatingOrganizationFromSystemSettings() NN_NOEXCEPT
{
    RegionForDefaultRatingOrganization region;
    LanguageForDefaultRatingOrganization language;

    {
        nn::settings::system::RegionCode r;
        nn::settings::system::GetRegionCode(&r);
        region = ConvertRegionValue(r);
    }

    {
        nn::settings::Language sysLanguage;
        if (GetSystemLanguage(&sysLanguage))
        {
            language = ConvertLanguageValue(sysLanguage);
        }
        else
        {
            language = LanguageForDefaultRatingOrganization::Others;
        }
    }

    return DefaultRatingOrganizations[static_cast<int>(region) * static_cast<int>(LanguageForDefaultRatingOrganization::Max) + static_cast<int>(language)];
}

bool GetNetworkTime(nn::time::PosixTime* outTime) NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // デバッグ/テスト用に時刻を捻じ曲げる操作があった場合はそれを返す
    if (g_IsUserTimeEnabled)
    {
        *outTime = g_UserTime;
        return true;
    }
#endif

    nn::time::PosixTime time = {};
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // 暫定
    if (nn::time::StandardUserSystemClock::GetCurrentTime(&time).IsFailure())
#else
    if (nn::time::StandardNetworkSystemClock::GetCurrentTime(&time).IsFailure())
#endif
    {
        *outTime = nn::time::PosixTime();
        return false;
    }
    *outTime = time;
    return true;
}

// デバッグ/テスト用に時刻を捻じ曲げるための関数
void SetNetworkTimeForDebug(bool isEnabled, nn::time::PosixTime timeSeconds) NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    g_IsUserTimeEnabled = isEnabled;
    g_UserTime = timeSeconds;
#else
    NN_UNUSED(isEnabled);
    NN_UNUSED(timeSeconds);
#endif
}

uint32_t GetSystemVersion() NN_NOEXCEPT
{
    GetFirmwareVersionImpl();
    return static_cast<uint32_t>(g_FirmwareVersion.GetComparableVersion());
}

uint32_t GetSystemVersion(char (& systemVersionString)[MaxSystemVersionStringLength]) NN_NOEXCEPT
{
    GetFirmwareVersionImpl();
    nn::util::Strlcpy(systemVersionString, g_FirmwareVersion.displayVersion, MaxSystemVersionStringLength);
    return static_cast<uint32_t>(g_FirmwareVersion.GetComparableVersion());
}

}}}}}
