﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/common/pctl_HttpRequest.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/common/pctl_ClientCertificate.h>
#include <nn/pctl/detail/service/common/pctl_HttpErrorHandler.h>

#include <curl/curl.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/nn_SdkLog.h>

#include <cstdarg>
#include <cstdio>

#define CURL_EASY_DO(exp) \
    do                                                              \
    {                                                               \
        CURLcode code_ = (exp);                                     \
                                                                    \
        if (code_ != CURLE_OK)                                      \
        {                                                           \
            return detail::service::common::HandleHttpError(code_); \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(0))

namespace nn { namespace pctl { namespace detail { namespace service { namespace common {

namespace
{
    inline CURL* GetCurl(void* curl)
    {
        return reinterpret_cast<CURL*>(curl);
    }
    inline CURLM* GetCurlMulti(void* multi)
    {
        return reinterpret_cast<CURLM*>(multi);
    }
    inline curl_slist* GetCurlSlist(void* data)
    {
        return reinterpret_cast<curl_slist*>(data);
    }

    struct ResponseHeaderEndCallbackData
    {
        int statusCode;
    };

    bool ResponseHeaderEndCallback(int statusCode, void* param) NN_NOEXCEPT
    {
        ResponseHeaderEndCallbackData* pData = static_cast<ResponseHeaderEndCallbackData*>(param);
        pData->statusCode = statusCode;
        return true;
    }

    static CURLcode SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(curl);
        NN_UNUSED(ssl);
        NN_UNUSED(param);

        nn::ssl::Context* context = reinterpret_cast<nn::ssl::Context*>(ssl);

        auto result = common::InitializeSslContext(context);
        if (result.IsFailure())
        {
            return CURLE_ABORTED_BY_CALLBACK;
        }

        return CURLE_OK;
    }

    int HttpTransferInfoFunction(void* clientp, curl_off_t dltotal, curl_off_t dlnow, curl_off_t ultotal, curl_off_t ulnow) NN_NOEXCEPT
    {
        NN_UNUSED(dltotal);
        NN_UNUSED(dlnow);
        NN_UNUSED(ultotal);
        NN_UNUSED(ulnow);

        HttpRequest* request = static_cast<HttpRequest*>(clientp);

        // キャンセルの判定を行う
        if (request->CheckCancelStatusForCallback())
        {
            return 1; // 非0で中止
        }

        return 0;
    }

    void Dump(const char *text, unsigned char *ptr, size_t size)
    {
        size_t i;
        size_t c;
        unsigned int width = 0x10;

        NN_SDK_LOG("%s, %10.10ld bytes (0x%8.8lx)\n", text, (long)size, (long)size);

        for (i = 0; i<size; i += width) {
            NN_SDK_LOG("%4.4lx: ", (long)i);

            /* show hex to the left */
            for (c = 0; c < width; c++) {
                if (i + c < size)
                    NN_SDK_LOG("%02x ", ptr[i + c]);
                else
                    NN_SDK_LOG("   ");
            }

            /* show data on the right */
            for (c = 0; (c < width) && (i + c < size); c++) {
                NN_SDK_LOG("%c", (ptr[i + c] >= 0x20 && ptr[i + c] < 0x80) ? ptr[i + c] : '.');
            }

            NN_SDK_LOG("\n"); /* newline */
        }
    }

    int TraceFunction(CURL *handle, curl_infotype type, char *data, size_t size, void *userp)
    {
        const char *text;
        NN_UNUSED(handle);
        NN_UNUSED(userp);

        switch (type) {
        case CURLINFO_TEXT:
            NN_SDK_LOG("== Info: %s", data);
        default: /* in case a new one is introduced to shock us */
            return 0;

        case CURLINFO_HEADER_OUT:
            text = "=> Send header";
            break;
        case CURLINFO_DATA_OUT:
            text = "=> Send data";
            break;
        case CURLINFO_SSL_DATA_OUT:
            text = "=> Send SSL data";
            break;
        case CURLINFO_HEADER_IN:
            text = "<= Recv header";
            break;
        case CURLINFO_DATA_IN:
            text = "<= Recv data";
            break;
        case CURLINFO_SSL_DATA_IN:
            text = "<= Recv SSL data";
            break;
        }

        Dump(text, (unsigned char *)data, size);
        return 0;
    }
}

nn::Result ConvertStatusCodeToResult(int statusCode) NN_NOEXCEPT
{
    if (statusCode >= 200 && statusCode < 300)
    {
        return nn::ResultSuccess();
    }
    switch (statusCode)
    {
        case 404:
            return nn::pctl::ResultUnexpectedServerError404();
        case 500:
            return nn::pctl::ResultUnexpectedServerError500();
        case 503:
            return nn::pctl::ResultUnexpectedServerError503();
        case 504:
            return nn::pctl::ResultUnexpectedServerError504();
        default:
            break;
    }
    if (statusCode >= 300 && statusCode < 400)
    {
        return nn::pctl::ResultUnexpectedServerError3xx();
    }
    if (statusCode >= 400 && statusCode < 500)
    {
        return nn::pctl::ResultUnexpectedServerError4xx();
    }
    if (statusCode >= 500 && statusCode < 600)
    {
        return nn::pctl::ResultUnexpectedServerError5xx();
    }
    NN_DETAIL_PCTL_INFO("Unknown status code: %d\n", statusCode);
    return nn::pctl::ResultUnexpectedServerErrorXxx();
}

nn::Result PerformRequest(HttpRequest* pRequest,
    void* streamWorkBuffer,
    size_t streamWorkBufferSize,
    const common::Cancelable* pCancelable) NN_NOEXCEPT
{
    ResponseHeaderEndCallbackData data = { 0 };

    pRequest->SetCancelable(pCancelable);
    pRequest->RegisterResponseHeaderEndCallback(ResponseHeaderEndCallback, &data);

    while (NN_STATIC_CONDITION(true))
    {
        size_t read = 0;
        auto result = pRequest->Receive(&read, streamWorkBuffer, streamWorkBufferSize);
        NN_RESULT_THROW_UNLESS(!pCancelable->IsCanceled(), ResultCanceled());
        NN_RESULT_DO(result);
        if (read == 0)
        {
            break;
        }
    }

    NN_RESULT_DO(common::ConvertStatusCodeToResult(data.statusCode));

    NN_RESULT_SUCCESS;
}

////////////////////////////////////////////////////////////////////////////////

HttpRequest::HttpRequest() NN_NOEXCEPT :
    m_Curl(nullptr),
    m_Multi(nullptr),
    m_Headers(nullptr),
    m_Param(),
    m_ResponseHeaderCallback(nullptr),
    m_ResponseHeaderParam(nullptr),
    m_ResponseHeaderEndCallback(nullptr),
    m_ResponseHeaderEndParam(nullptr),
    m_SendDataCallback(nullptr),
    m_SendDataParam(nullptr),
    m_SendDataSize(0),
    m_StatusCode(0),
    m_IsOpened(false),
    m_IsCanceledInCallback(false)
{
}

HttpRequest::~HttpRequest() NN_NOEXCEPT
{
    Close();
}

nn::Result HttpRequest::Open(const char* url) NN_NOEXCEPT
{
    return Open("GET", url);
}

nn::Result HttpRequest::Open(const char* request, const char* url) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(request);
    NN_SDK_REQUIRES_NOT_NULL(url);

    NN_SDK_ASSERT(!m_IsOpened);

    m_IsCanceledInCallback = false;

    CURL* curl = curl_easy_init();

    if (!curl)
    {
        return ResultHttpErrorInterfaceFailed();
    }

    NN_UTIL_SCOPE_EXIT
    {
        if (curl)
        {
            curl_easy_cleanup(curl);
        }
    };

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_CUSTOMREQUEST, request));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_URL, url));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, SslCtxFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, HttpHeaderFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_HEADERDATA, this));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, HttpWriteFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_WRITEDATA, &m_Param));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_XFERINFOFUNCTION, HttpTransferInfoFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_XFERINFODATA, this));
    if (m_SendDataCallback != nullptr)
    {
        CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_POST, 1));
        CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE_LARGE, m_SendDataSize));
        CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_READFUNCTION, HttpReadFunction));
        CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_READDATA, this));
    }

    // 開発中に HTTP 通信ログを出したい場合、有効にする。
    //CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_VERBOSE, 1));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_DEBUGFUNCTION, TraceFunction));

    // 最大リダイレクト回数を制限する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_MAXREDIRS, 1));
    // 1 B/s 以下の転送速度の通信が 30 秒以上続いたら切断する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_LIMIT, 1));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_TIME, 30));
    // 接続タイムアウト（CONNECT 完了までの時間）を 30 秒にする。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 30));
    // 証明書の有効期限確認のみ無効化する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_VERIFYDATE, 0));

    CURLM* multi = curl_multi_init();

    if (!multi)
    {
        return ResultHttpErrorOutOfMemory();
    }

    NN_UTIL_SCOPE_EXIT
    {
        if (multi)
        {
            curl_multi_cleanup(multi);
        }
    };

    CURLMcode code = curl_multi_add_handle(multi, curl);

    if (code != CURLM_OK)
    {
        curl_multi_cleanup(multi);

        if (code == CURLM_OUT_OF_MEMORY)
        {
            return ResultHttpErrorOutOfMemory();
        }
        else
        {
            return ResultHttpErrorFailedInit();
        }
    }

    m_Curl = curl;
    curl = nullptr;

    m_Multi = multi;
    multi = nullptr;

    m_Param.readNext = true;

    m_IsOpened = true;

    NN_RESULT_DO(AddRequestHeader(UserAgentHeaderForHttpRequest));
    NN_RESULT_DO(AddRequestHeader(AdditionalVersionHeaderForHttpRequest));

    return nn::ResultSuccess();
}

void HttpRequest::Close() NN_NOEXCEPT
{
    if (m_IsOpened)
    {
        curl_multi_remove_handle(GetCurlMulti(m_Multi), GetCurl(m_Curl));
        curl_multi_cleanup(GetCurlMulti(m_Multi));

        if (m_Headers)
        {
            curl_slist_free_all(GetCurlSlist(m_Headers));
        }

        curl_easy_cleanup(GetCurl(m_Curl));

        m_IsOpened = false;
    }
}

nn::Result HttpRequest::AddRequestHeader(const char* header) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(header);

    NN_SDK_ASSERT(m_IsOpened);

    curl_slist* headers = curl_slist_append(GetCurlSlist(m_Headers), header);

    if (!headers)
    {
        return ResultHttpErrorOutOfMemory();
    }

    m_Headers = headers;

    CURL_EASY_DO(curl_easy_setopt(GetCurl(m_Curl), CURLOPT_HTTPHEADER, GetCurlSlist(m_Headers)));

    return nn::ResultSuccess();
}

nn::Result HttpRequest::AddRequestHeaderFormat(const char* format, ...) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(format);

    NN_SDK_ASSERT(m_IsOpened);

    char line[RequestHeaderLineSize] = {};

    va_list args;
    va_start(args, format);

    int length = nn::util::VSNPrintf(line, sizeof (line), format, args);

    va_end(args);

    if (length >= static_cast<int>(RequestHeaderLineSize))
    {
        return ResultInvalidValue();
    }

    return AddRequestHeader(line);
}

nn::Result HttpRequest::SetPostField(const char* data, bool copy) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(data);

    NN_SDK_ASSERT(m_IsOpened);

    CURL_EASY_DO(curl_easy_setopt(GetCurl(m_Curl), copy ? CURLOPT_COPYPOSTFIELDS : CURLOPT_POSTFIELDS, data));

    return nn::ResultSuccess();
}

void HttpRequest::RegisterResponseHeaderCallback(ResponseHeaderCallback function, void* param) NN_NOEXCEPT
{
    m_ResponseHeaderCallback = function;
    m_ResponseHeaderParam = param;
}

void HttpRequest::RegisterResponseHeaderEndCallback(ResponseHeaderEndCallback function, void* param) NN_NOEXCEPT
{
    m_ResponseHeaderEndCallback = function;
    m_ResponseHeaderEndParam = param;
}

nn::Result HttpRequest::RegisterSendDataCallback(HttpSendDataCallback function, void* param, size_t sendDataSize) NN_NOEXCEPT
{
    m_SendDataCallback = function;
    m_SendDataParam = param;
    m_SendDataSize = sendDataSize;

    if (m_IsOpened)
    {
        CURL_EASY_DO(curl_easy_setopt(GetCurl(m_Curl), CURLOPT_POST, 1));
        CURL_EASY_DO(curl_easy_setopt(GetCurl(m_Curl), CURLOPT_READFUNCTION, HttpReadFunction));
        CURL_EASY_DO(curl_easy_setopt(GetCurl(m_Curl), CURLOPT_READDATA, this));
        CURL_EASY_DO(curl_easy_setopt(GetCurl(m_Curl), CURLOPT_POSTFIELDSIZE_LARGE, m_SendDataSize));
    }

    return nn::ResultSuccess();
}

void HttpRequest::SetCancelable(const common::Cancelable* cancelable) NN_NOEXCEPT
{
    m_Cancelable = cancelable;
}

nn::Result HttpRequest::Receive(size_t* outRead, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsOpened);

    if (IsCanceled())
    {
        return ResultCanceled();
    }

    m_Param.read = 0;

    if (m_Param.readNext)
    {
        m_Param.stream = buffer;
        m_Param.streamSize = bufferSize;
        m_Param.positionInHttpBuffer = 0;
        m_Param.readNext = false;
    }
    else
    {
        curl_easy_pause(GetCurl(m_Curl), CURLPAUSE_CONT);
    }

    int stillRunning = 0;

    while (m_Param.read == 0)
    {
        int rval = 0;
        CURLMcode code = curl_multi_wait(GetCurlMulti(m_Multi), nullptr, 0, 10, &rval);

        if (code != CURLM_OK)
        {
            if (code == CURLM_OUT_OF_MEMORY)
            {
                return ResultHttpErrorOutOfMemory();
            }
            else
            {
                return ResultHttpErrorBadFunctionArgument();
            }
        }

        if (rval == -1)
        {
            return ResultHttpErrorRecvError();
        }

        curl_multi_perform(GetCurlMulti(m_Multi), &stillRunning);

        if (stillRunning == 0)
        {
            break;
        }

        if (IsCanceled())
        {
            return ResultCanceled();
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    *outRead = m_Param.read;

    if (stillRunning == 0)
    {
        CURLMsg* msg = nullptr;

        do
        {
            int n = 0;
            msg = curl_multi_info_read(GetCurlMulti(m_Multi), &n);

            if (msg && (msg->msg == CURLMSG_DONE))
            {
                if (m_IsCanceledInCallback)
                {
                    return ResultCanceled();
                }
                return detail::service::common::HandleHttpError(msg->data.result);
            }
        }
        while (msg);
    }

    return nn::ResultSuccess();
}

bool HttpRequest::CheckCancelStatusForCallback() NN_NOEXCEPT
{
    if (IsCanceled())
    {
        m_IsCanceledInCallback = true;
        return true;
    }
    return false;
}

bool HttpRequest::IsCanceled() const NN_NOEXCEPT
{
    if (m_Cancelable)
    {
        return m_Cancelable->IsCanceled();
    }

    return false;
}

size_t HttpRequest::HttpHeaderFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    HttpRequest* stream = static_cast<HttpRequest*>(param);

    if (stream->IsCanceled())
    {
        stream->m_IsCanceledInCallback = true;
        return 0;
    }

    size_t bufferSize = size * count;

    if (nn::util::Strncmp(buffer, "HTTP/", 5) == 0)
    {
        int majorVersion = 0;
        int minorVersion = 0;
        int statusCode = 0;

        char space = '\0';

        int result = std::sscanf(buffer, "HTTP/%1d.%1d %03d%c", &majorVersion, &minorVersion, &statusCode, &space);

        if (result == 4 && space == ' ')
        {
            stream->m_StatusCode = statusCode;
        }

        return bufferSize;
    }

    // 1xx 系のステータスコードは内部処理に影響を与えるものであるため、アプリに影響を与えないようにする。
    if (stream->m_StatusCode / 100 == 1)
    {
        return bufferSize;
    }

    if (bufferSize == 2 && buffer[0] == '\r' && buffer[1] == '\n')
    {
        if (stream->m_ResponseHeaderEndCallback)
        {
            bool result = stream->m_ResponseHeaderEndCallback(stream->m_StatusCode, stream->m_ResponseHeaderEndParam);

            if (!result)
            {
                bufferSize = 0;
            }
        }
        return bufferSize;
    }

    char* label = buffer;
    char* colon = strchr(label, ':');

    if (!colon)
    {
        // unexpected
        return 0;
    }

    char* value = colon + 1;

    while (*value == ' ')
    {
        value++;
    }

    char* cr = strchr(value, '\r');

    if (!cr)
    {
        return bufferSize;
    }

    *colon = '\0';
    *cr = '\0';

    if (stream->m_ResponseHeaderCallback)
    {
        bool result = stream->m_ResponseHeaderCallback(label, value, stream->m_ResponseHeaderParam);

        if (!result)
        {
            bufferSize = 0;
        }
    }

    *colon = ':';
    *cr = '\r';

    return bufferSize;
}

size_t HttpRequest::HttpReadFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    HttpRequest* p = reinterpret_cast<HttpRequest*>(param);

    return p->m_SendDataCallback(buffer, size, count, p->m_SendDataParam);
}

size_t HttpRequest::HttpWriteFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    HttpWriteFunctionParam* p = reinterpret_cast<HttpWriteFunctionParam*>(param);

    size_t bufferSize = size * count;
    size_t streamSize = bufferSize - p->positionInHttpBuffer;

    size_t copySize = (streamSize < p->streamSize ? streamSize : p->streamSize);

    std::memcpy(p->stream, &buffer[p->positionInHttpBuffer], copySize);
    p->read = copySize;
    p->positionInHttpBuffer += copySize;

    NN_SDK_ASSERT(p->positionInHttpBuffer <= bufferSize);

    if (bufferSize == p->positionInHttpBuffer)
    {
        // WriteFunction のバッファをすべてパースしたら、次のデータを読み込む。
        p->readNext = true;

        return bufferSize;
    }
    else
    {
        return CURL_WRITEFUNC_PAUSE;
    }
}

}}}}}
