﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/pctl/detail/ipc/pctl_IpcSession.h>
#include <nn/pctl/detail/ipc/pctl_IParentalControlServiceFactory.sfdl.h>

#if NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_DFC
#include <nn/pctl/detail/service/pctl_ParentalControlServiceFactoryImpl.h>
#include <nn/pctl/detail/service/pctl_IpcServer.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>

#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/sm/sm_Result.h>

#if defined(NN_DETAIL_PCTL_CONFIG_USE_FINALIZE_SESSION)
#include <cstdlib>
#endif

namespace nn { namespace pctl { namespace detail { namespace ipc {

namespace
{
    bool g_IsSessionInitialized = false;

    std::aligned_storage<service::ParentalControlServiceFactoryImpl::SessionSize>::type g_SessionStorage;
    nn::sf::ExpHeapAllocator g_SessionAllocator;
    nn::lmem::HeapHandle g_SessionHeapHandle;
    std::aligned_storage<sizeof(service::ParentalControlServiceFactoryImpl)>::type g_FactoryStorage;
    service::ParentalControlServiceFactoryImpl* g_pFactory;
    bool g_IsWatcherSessionInitialized = false;
    IParentalControlService* g_pActiveService = nullptr;

#if defined(NN_DETAIL_PCTL_CONFIG_USE_FINALIZE_SESSION)
    void AtExitMain() NN_NOEXCEPT
    {
        FinalizeIpcSession();
    }
    void AtExitWatcher() NN_NOEXCEPT
    {
        // Watcher用に初期化している場合でも通常の終了処理を行う
        // (FinalizeIpcSession は多重に呼び出せる)
        // MEMO: ssl が atexit を使っているため、AtExitWatcher は
        //   ssl の atexit より先に呼び出されるが、
        //   AtExitMain はそれより後に呼び出されることになるので
        //   先に Finalize を行う必要がある
        FinalizeIpcSession();
    }
#endif

    nn::Result InitializeIpcSessionImpl() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!g_IsSessionInitialized);

        // 同一プロセスモードの場合はここでサーバー処理の初期化を実施
        service::InitializeAllocatorForServer();
        service::InitializeMain();

        g_SessionHeapHandle = nn::lmem::CreateExpHeap(&g_SessionStorage, sizeof(g_SessionStorage), nn::lmem::CreationOption_ThreadSafe);
        g_SessionAllocator.Attach(g_SessionHeapHandle);
        g_pFactory = new(&g_FactoryStorage) service::ParentalControlServiceFactoryImpl(&g_SessionAllocator);

        g_IsSessionInitialized = true;

        // サービスオブジェクトを生成しておく
        GetServiceImpl();

#if defined(NN_DETAIL_PCTL_CONFIG_USE_FINALIZE_SESSION)
        // (自動テスト用)
        std::atexit(AtExitMain);
        // at_quick_exit で終了処理を行う必要は現状無い
        //std::at_quick_exit(AtExitMain);
#endif

        NN_RESULT_SUCCESS;
    }

    nn::Result InitializeWatcherIpcSessionImpl() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!g_IsWatcherSessionInitialized);

        service::InitializeWatcher();
        g_IsWatcherSessionInitialized = true;

#if defined(NN_DETAIL_PCTL_CONFIG_USE_FINALIZE_SESSION)
        // service::InitializeWatcher の終了処理を呼び出せるようにハンドラを設定
        // (自動テスト用)
        std::atexit(AtExitWatcher);
        // at_quick_exit で終了処理を行う必要は現状無い
        //std::at_quick_exit(AtExitWatcher);
#endif

        NN_RESULT_SUCCESS;
    }
}

// Finalize を行う場合は以下の処理
#if defined(NN_DETAIL_PCTL_CONFIG_USE_FINALIZE_SESSION)
void FinalizeIpcSessionImpl() NN_NOEXCEPT
{
    if (!g_IsSessionInitialized)
    {
        return;
    }

    if (g_pActiveService != nullptr)
    {
        nn::sf::ReleaseSharedObject(g_pActiveService);
        g_pActiveService = nullptr;
    }

    g_pFactory->~ParentalControlServiceFactoryImpl();
    g_SessionAllocator.Detach();
    nn::lmem::DestroyExpHeap(g_SessionHeapHandle);
    if (g_IsWatcherSessionInitialized)
    {
        service::FinalizeWatcher();
    }
    service::FinalizeMain();

    g_IsSessionInitialized = false;
}
#endif

// NOTE: nninitInitializeSdkModule -> nn::pctl::Initialize から呼び出されるため、
// 扱う変数が static initializer を利用しないようにする
nn::Result InitializeIpcSessionIfNeeded() NN_NOEXCEPT
{
    if (g_IsSessionInitialized)
    {
        NN_RESULT_SUCCESS;
    }
    return InitializeIpcSessionImpl();
}

nn::Result InitializeWatcherIpcSessionIfNeeded() NN_NOEXCEPT
{
    if (g_IsWatcherSessionInitialized)
    {
        NN_RESULT_SUCCESS;
    }
    NN_SDK_ASSERT(g_IsSessionInitialized);
    return InitializeWatcherIpcSessionImpl();
}

IParentalControlService* GetServiceImpl() NN_NOEXCEPT
{
    if (!g_pActiveService)
    {
        nn::sf::SharedPointer<IParentalControlService> service = nullptr;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_pFactory->CreateService(&service, 0));
        g_pActiveService = service.Detach();
    }

    return g_pActiveService;
}

}}}}

#endif // NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_DFC
