﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/pcie/pcie.h>
#include <nn/pcie/pcie_LoggedState.h>
#include <nn/pcie/driver/pcie.h>
#include "../detail/pcie_Macros.h"
#include "../detail/pcie_ServiceName.h"
#include "pcie_LoggedStateManagerImpl.h"

namespace nn { namespace pcie { namespace server {

LoggedStateManagerImpl::LoggedStateManagerImpl() NN_NOEXCEPT
    : m_IsLoggedStateStatisticsReported(false),
    m_LoggedStateEvent(nn::os::EventClearMode_AutoClear, true)
{
    NN_PCIE_LOG_VERBOSE("LoggedStateManagerImpl::LoggedStateManagerImpl()\n");

    // Setup driver callback for logged state updates
    nn::pcie::driver::SetLoggedStateCallback(LoggedStateCallback, reinterpret_cast<uintptr_t>(this));
}

LoggedStateManagerImpl::~LoggedStateManagerImpl() NN_NOEXCEPT
{
    NN_PCIE_LOG_VERBOSE("LoggedStateManagerImpl::~LoggedStateManagerImpl()\n");

    // Clear driver callback for logged state updates
    nn::pcie::driver::SetLoggedStateCallback(LoggedStateCallback, reinterpret_cast<uintptr_t>(nullptr));
}

nn::Result LoggedStateManagerImpl::GetLoggedState(const nn::sf::OutBuffer& output, nn::sf::Out<bool> isLoggedErrorPending, bool isCleared) NN_NOEXCEPT
{
    nn::Result result;
    isLoggedErrorPending.Set(m_IsLoggedErrorPending);
    m_IsLoggedErrorPending = false;
    result = nn::pcie::driver::GetLoggedState(reinterpret_cast<LoggedState*>(output.GetPointerUnsafe()), isCleared);
    return result;
}

nn::Result LoggedStateManagerImpl::GetLoggedStateEvent(nn::sf::Out<nn::sf::NativeHandle> loggedStateEventHandle, bool isStatisticReported) NN_NOEXCEPT
{

    m_IsLoggedStateStatisticsReported = isStatisticReported;
    loggedStateEventHandle.Set(nn::sf::NativeHandle(m_LoggedStateEvent.GetReadableHandle(), false));
    return ResultSuccess();
}

void LoggedStateManagerImpl::DoLoggedStateCallback(bool isDueToError) NN_NOEXCEPT
{
    m_IsLoggedErrorPending |= isDueToError;
    if(isDueToError || m_IsLoggedStateStatisticsReported)
    {
        m_LoggedStateEvent.Signal();
    }
}

} // namespace server
} // namespace pci
} // namespace nn
