﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file    pci_Tegra21xRootComplex.cpp
 * @brief   TK2 Platform PCIe Root Complex
 */
#include "pcie_Tegra.h"

namespace nn { namespace pcie { namespace driver { namespace detail {

const RootComplex::Config Tegra21xRootComplexConfiguration =
{
    BusSpeed_EGen2,   // speedLimit
    2,                // numPorts
    {
        {             // port 0
#if defined NN_DETAIL_PCIE_WITHOUT_HB
            false,      // isEnabled
#else
            true,       // isEnabled
#endif
            false,      // isClockDisabled
            4           // laneCount
        },
        {             // port 1
            true,       // isEnabled
            false,      // isClockDisabled
            1           // laneCount
        }
    }
};

class Driver;

class Tegra21xRootComplex : public RootComplex
{
public:
    NN_IMPLICIT Tegra21xRootComplex(Driver *pDriver, Config *pCfg);
    virtual ~Tegra21xRootComplex();
    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;

protected:
    // Register virtual base addresses
    uintptr_t  m_PcieA1Base;
    uintptr_t  m_PcieA2Base;
    uintptr_t  m_PcieA3Base;
    uintptr_t  m_PcieCtrl0CfgBase;
    uintptr_t  m_PcieCtrl1CfgBase;
    uintptr_t  m_PciePca0Base;
    uintptr_t  m_PciePca1Base;
    uintptr_t  m_PciePadsBase;
    uintptr_t  m_PcieAfiBase;
    uintptr_t  m_XusbPadctlBase;
    uintptr_t  m_PinMuxAuxBase;

    void SynchronizedWrite(volatile uint32_t *address, uint32_t value);

protected:
    // Implement RootComplex's API
    virtual Result Initialize();
    virtual Result Finalize();
    virtual Result Resume();
    virtual Result Suspend();

private:
    enum PortOperation
    {
        PORT_OP_DISABLE_CLOCK,
        PORT_OP_ENABLE_CLOCK_OVERRIDE,
        PORT_OP_ENABLE_CLOCK_AUTOMATIC,
        PORT_OP_RESET,
        PORT_OP_HOLD_RESET,
        PORT_OP_GET_STATE
    };
    enum InterruptEvent
    {
        InterruptEvent_Msi,
        InterruptEvent_Int,
        InterruptEvent_Wake,
        InterruptEvent_Max,
    };
    enum Params
    {
        Params_ConfigSpaceSize   = 0x04000000,
        Params_IoSpaceSize       = 0x00010000,
        Params_PrefetchMemSize   = 0x02000000,
        Params_NoPrefetchMemSize = 0x02000000,
        Params_MsiBufferSize     = 0x10000,
    };
    enum LocalEventId
    {
        LocalEventId_Invalid,
        LocalEventId_PortMonitorTimer
    };
    struct LaneConfiguration
    {
        uint8_t  address;
        uint16_t data;
    };

    // Waiting for PLL
    static const uint32_t MaxPllWaitTimeInUs  = 1000000;
    static const uint32_t PllPollIntervalInUs = MaxPllWaitTimeInUs / 100;

    // Endpoint device reset
    static const uint32_t EndpointResetDurationInUs = 2000;
    static const uint32_t EndpointResetLtssmDetectTimeInUs = 20000;

    // Waiting for PCIe link
    static const uint32_t MaxLinkWaitTimeInMs = 350;
    static const uint32_t LinkPollIntervalInMs = MaxLinkWaitTimeInMs / 35;

    // Interrupt vectors
    static const os::InterruptName WakeInterruptName = 32 + 100;
    static const os::InterruptName MsiInterruptName = 32 + 99;
    static const os::InterruptName MainInterruptName = 32 + 98;

    // IO VA
    static const nn::dd::DeviceVirtualAddress IoVaHeapBase = 0x80000000;
    static const nn::dd::DeviceVirtualAddress IoVaHeapSize = 0x60000000;
    static const nn::dd::DeviceVirtualAddress IoVaInternalBase = 0xF0000000;

    nn::os::InterruptEventType  m_InterruptEvent[InterruptEvent_Max];

    // Configuration resolved at runtime
    nn::pcv::MicroVolt m_Ldo1Voltage;
    uint32_t     m_RpClkClampThreshold;
    uint32_t     m_PadsRefClkCfg0;

    // Configuration retrieved from nn::settings
    int32_t      m_PcieActiveLaneMap;
    int32_t      m_PcieInactiveLaneMap;
    int32_t      m_UsbLaneMap;
    char         m_SocName[32];

    // PADCTRL mux IDDQ pattern
    uint32_t     m_PadctrlMuxIddqMask;

    // MSI DMA buffer
    void*        m_pMsiBuffer;

    //Config Space(within A2)
    ResourceAddr m_ConfigSpaceStart;
    ResourceAddr m_ConfigSpaceEnd;
    uintptr_t    m_ConfigSpaceStartPhys;

    //I/O Space(within A2)
    ResourceAddr m_IoSpaceStart;
    uintptr_t    m_IoSpaceStartPhys;

    //Prefetchable Memory(within A3)
    ResourceAddr m_PrefetchMemStart;
    uintptr_t    m_PrefetchMemStartPhys;

    // Non-Prefetchable Memory(within A3)
    ResourceAddr m_NoPrefetchMemStart;
    uintptr_t    m_NoPrefetchMemStartPhys;

    // Logged state
    RootComplexLoggedState m_LoggedState;

    // Local event handling
    LocalEventPoolType  m_LocalEventPool;
    LocalEventType      m_LocalEventStorage[4];
    LocalEventType      m_PortMonitorTimers[MaxSwitchPortCount];

    // IRQ Managers
    static const int32_t msiIrqMgrPoolId = 0;
    static const int32_t intPort0IrqMgrBasePoolId = 0x10;
    static const int32_t intPort1IrqMgrBasePoolId = 0x10 + AFI_INT_TOTAL_VECTORS;
    typedef IrqManager<MaxFunctionsPerIrq, 1, EndpointFunction *, 0> PortIntxIrqMgrType;
    PortIntxIrqMgrType m_port0IntIrqMgrs[AFI_INT_TOTAL_VECTORS];
    PortIntxIrqMgrType m_port1IntIrqMgrs[AFI_INT_TOTAL_VECTORS];
    IrqManager<AFI_MSI_TOTAL_VECTORS, nn::pcie::MaxIrqVectorsPerDeviceFunction,
        EndpointFunction *, msiIrqMgrPoolId> m_msiIrqMgr;

    Result InitPhy();
    Result InitXusb();
    Result InitPinMux();

    Result DoPortOperation(DeviceNumber portNum, PortOperation op);
    Result SetPresentMapOverride(DeviceNumber portNum, bool isPresent);
    Result DoPmeTurnoff();
    Result SetAspmEnable(EndpointFunction *pEf, bool isEnabled);
    Result GetDmaBusAddressRange(BusAddress *pOutBase, BusAddress *pOutSize);
    Result PortControl(DeviceNumber devNum, PortCommand portCommand);
    Result GetLoggedState(RootComplexLoggedState* pOut, bool isCleared);
    Result InitAFIBars();
    Result InitAFIRegisters();
    Result InitPCIeRegisters(DeviceNumber portNum);
    Result InitPCIRegisters(DeviceNumber portNum);
    Result InitMSI();
    void   ResolveMemMap();
    Result SetInterruptVectorEnable(bool isEnabled);
    uint32_t GetEndpointFunctionEnableMask(int32_t baseResourceIndex);
    Result ComputeConfigSpaceAddress(volatile uint32_t** ppOut, BusNumber busNum, DeviceNumber devNum,
                                     FunctionNumber funcNum, int32_t where);

    // Implement RootComplex's API
    Result ReadConfigSpace(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                           int32_t where, uint32_t size, void *pVal);
    Result WriteConfigSpace(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                            int32_t where, uint32_t size, uint32_t val);
    void GetRegions(Region *pIo, Region *pPrefetchMem, Region *pNoPrefetchMem);
    Result AcquireIrq(EndpointFunction *pEp, IrqOptions *pOptions, IrqProfile *pRetProfile);
    Result ReleaseIrq(IrqHandle irqHandle);
    Result SetIrqEnable(IrqHandle irqHandle, int32_t index, bool irqEnabled);
    Result GetIrqProfile(IrqHandle irqHandle, IrqProfile *pRetProfile);
    void HandleInterrupt(IntVector intVector);
    void SetErrorReportingEnable(bool enable);

    bool HandleINTInterrupt();
    bool DispatchIntx(PortIntxIrqMgrType *pMgr);
    bool HandleSidebandMessageInterrupt(uint32_t sigRegVal);
    bool HandleMSIInterrupt();

    uint32_t L1SSTPowerOnScaledValue(uint32_t capValue);
    Result   ConfigureCommonClock(BridgeFunction *pBf, EndpointFunction *pEf);
    void     DumpPCIeSpace(uint32_t *cfgBase, uint32_t *virtBase, uint32_t *physBase, uint32_t size);

    void MonitorPortCounters(DeviceNumber portNum, bool isPortMonitorTimerArmed);
    void HandleLocalEvent(LocalEventType *pEvent);
    static void HandleLocalEventStatic(void *context, LocalEventType *pEvent)
    {
        Tegra21xRootComplex* pThis = reinterpret_cast<Tegra21xRootComplex*>(context);
        pThis->HandleLocalEvent(pEvent);
    }

};

} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn

