﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief Root complex interface class
 *
 * @details This class defines and partially implements a
 *          platform independent root complex interface.
 *          - The purpose of this class is to facilitate easy
 *            support for multiple hardware platforms.
 *          - Platform specific root complex implementations
 *            shall inherit this class, implementing all
 *            required API.
 *          - This class shall not be directly instantiated.
 */

namespace nn { namespace pcie { namespace driver { namespace detail {

class EndpointFunction;
class Driver;

class RootComplex
{

public:
    enum ControllerType
    {
        ControllerType_Invalid,
        ControllerType_Tegra124,
        ControllerType_Tegra21x,
    };
    enum PortCommand
    {
        PortCommand_HoldReset,
        PortCommand_CycleReset,
    };
    struct PortConfig
    {
        bool     isEnabled;
        bool     isClockDisabled;
        int32_t  laneCount;
    };
    struct Config
    {
        BusSpeed     speedLimit;
        int32_t      numPorts;
        PortConfig   ports[MaxSwitchPortCount];
    };
    struct IrqProfile
    {
        IrqType   irqType;
        uint32_t  enableMask;
        IrqHandle irqHandle;
        union
        {
            struct
            {
                BusAddress messageAddress;
                uint16_t   messageData;
            }msi;
        }type;
    };
    struct IrqOptions
    {
        IrqType irqType;
        union
        {
            struct
            {
                bool unused;
            }
            msi;
            struct
            {
                uint8_t intPin;
            }
            intx;
        }type;
    };
    struct Region
    {
        ResourceAddr     resourceAddr;
        BusAddress       busAddr;
        size_t           size;
    };
    typedef int32_t IntVector;

    NN_IMPLICIT RootComplex(Driver *pDriver, Config *pCfg);
    virtual ~RootComplex();


    virtual Result Initialize();
    virtual Result Finalize();
    virtual Result Resume();
    virtual Result Suspend();
    virtual bool IsInitialized();
    virtual bool IsResumed();
    virtual Result ReadConfigSpace(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                                   int32_t where, uint32_t size, void *pVal) = 0;
    virtual Result WriteConfigSpace(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                                    int32_t where, uint32_t size, uint32_t val) = 0;
    virtual void GetRegions(Region *pIo, Region *pPrefetchMem, Region *pNoPrefetchMem) = 0;
    virtual Result AcquireIrq(EndpointFunction *pEp, IrqOptions *pOptions, IrqProfile *pRetProfile) = 0;
    virtual Result ReleaseIrq(IrqHandle irqHandle) = 0;
    virtual Result SetIrqEnable(IrqHandle irqHandle, int32_t index, bool irqEnabled) = 0;
    virtual Result GetIrqProfile(IrqHandle irqHandle, IrqProfile *pRetProfile) = 0;
    virtual void HandleInterrupt(IntVector intVector) = 0;
    virtual void SetErrorReportingEnable(bool enable) = 0;
    virtual Result SetAspmEnable(EndpointFunction *pEp, bool isEnabled) = 0;
    virtual Result GetDmaBusAddressRange(BusAddress *pOutBase, BusAddress *pOutSize) = 0;
    virtual Result GetLoggedState(RootComplexLoggedState* pOut, bool isCleared) = 0;
    virtual Result PortControl(DeviceNumber devNum, PortCommand portCommand) = 0;
    Result SetIrqEnable(IrqHandle irqHandle, bool irqEnabled);
    Bus* GetBus();
    Result WriteConfigSpaceMasked32(BusNumber busNum, DeviceNumber devNum, FunctionNumber funcNum,
                                    int32_t where, uint32_t clearMask, uint32_t setMask);
    Result WaitPolledRegister32(volatile uint32_t *pReg, uint32_t mask,
                                bool polarity, int64_t retryIntervalInMicroseconds,
                                uint32_t maxWaitTimeInMicroseconds, const char *subject);
    BusSpeed GetBusSpeedLimitPolicy();
    Result CreateDmaMap(BusAddress *pReturnedBusAddressBase, int32_t procVaIndex, nn::dd::ProcessHandle procHandle,
                        uint64_t procVa, size_t size, nn::dd::MemoryPermission cpuPermission,
                        nn::dd::MemoryPermission devicePermission, int32_t tag);
    Result DestroyDmaMapByBusAddress(BusAddress busAddr);
    Result DestroyDmaMapByProcAddr(int32_t procVaIndex, uint64_t procVa);
    Result DestroyDmaMapsByTag(int32_t tag);
    Result GetDmaBusAddress(BusAddress* pReturnedBusAddress, int32_t procVaIndex, uint64_t procVa);

    void * operator new(size_t size) NN_NOEXCEPT;
    void operator delete(void *p, size_t size) NN_NOEXCEPT;


protected:
    bool            m_IsInitialized;
    bool            m_IsResumed;
    Driver          *m_pDriver;
    Config          m_Config;
    ControllerType  m_ControllerType;
    Bus             *m_pBus;          // downstream bus
    IoVaManager<nn::pcie::MinimumDmaAddressAlignment, nn::pcie::MaxClientCount + 1> m_IoVaManager;
    nn::dd::DeviceAddressSpaceType m_DeviceAddressSpace;

private:
    static void DestroyByTagCallout(void* calloutContext, IoVaAllocationSummary& summary);
};

} // namespace detail
} // namespace driver
} // namespace pcie
} // namespace nn


