﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <map>
#include <memory>

#include "pcie_PrivateIncludes.h"


namespace nn  { namespace pcie { namespace driver {  namespace detail {

class IoVaFreeSegment
{
public:
IoVaFreeSegment(uint32_t size, BusAddress ioVa) :
    m_Size(size), m_IoVa(ioVa){}
    uint32_t GetSize() {return m_Size;}
    BusAddress GetIoVa() {return m_IoVa;}
private:
    uint32_t   m_Size;
    BusAddress m_IoVa;
};

class IoVaAllocatedSegment
{
public:
IoVaAllocatedSegment(uint32_t size, uint32_t alignedOffset, BusAddress ioVa,
                     uint32_t requestedSize, int32_t procIndex, uint64_t procVa,
                     uint64_t context, int32_t tag) :
    m_Size(size), m_AlignedOffset(alignedOffset), m_IoVa(ioVa), m_RequestedSize(requestedSize),
        m_ProcVaIndex(procIndex), m_ProcVa(procVa), m_Context(context), m_Tag(tag) {}
    uint32_t GetBaseSize() {return m_Size + m_AlignedOffset;}
    uint32_t GetSize() {return m_Size;}
    uint32_t GetRequestedSize() {return m_RequestedSize;}
    BusAddress  GetBaseIoVa() {return m_IoVa;}
    BusAddress  GetIoVa() {return m_IoVa + m_AlignedOffset;}
    int32_t  GetProcVaIndex() {return m_ProcVaIndex;}
    uint64_t GetProcVa() {return m_ProcVa;}
    uint64_t GetContext(){return m_Context;}
    int32_t GetTag(){return m_Tag;}
private:
    uint32_t   m_Size;
    uint32_t   m_AlignedOffset;
    BusAddress m_IoVa;
    uint32_t   m_RequestedSize;
    int32_t    m_ProcVaIndex;
    uint64_t   m_ProcVa;
    uint64_t   m_Context;
    int32_t    m_Tag;
};

struct IoVaAllocationSummary
{
    uint32_t   requestedSize;
    int32_t    procVaIndex;
    uint64_t   procVa;
    BusAddress ioVa;
    uint64_t   context;
    int32_t    tag;
};

template <class T>
struct IoVaSegmentAllocator : public std::allocator<T>
{
    typedef std::allocator<T> Base;
    typedef typename Base::pointer pointer;
    typedef typename Base::reference reference;
    typedef typename Base::const_pointer const_pointer;
    typedef typename Base::const_reference const_reference;
    typedef typename Base::size_type size_type;
    IoVaSegmentAllocator() NN_NOEXCEPT {}
    IoVaSegmentAllocator(const IoVaSegmentAllocator&) NN_NOEXCEPT {}

    template<class U>
    IoVaSegmentAllocator(const IoVaSegmentAllocator<U>&) NN_NOEXCEPT {}

    template<class U>
    struct rebind
    {
      typedef IoVaSegmentAllocator<U> other;
    };

    pointer address(reference x) const { return &x; }
    const_pointer address(const_reference x) const { return &x; }
    pointer allocate(size_type count, const_pointer hint = 0) NN_NOEXCEPT
    {
        NN_UNUSED(hint);
        if(count == 0) {return nullptr;}
        pointer p = static_cast<pointer>(nn::pcie::detail::MemoryAlloc(
                                             NN_PCIE_ROUNDUP_SIZE(sizeof(T) * count,128),
                                             "IoVaSegment"));
        if(p == nullptr) NN_PCIE_ABORT("IoVaSegmentAllocator::allocate() failed.\n");
        return p;
    }
    void deallocate(pointer p, size_type count) NN_NOEXCEPT
    {
        NN_UNUSED(count);
        nn::pcie::detail::MemoryFree(p, "IoVaSegment");
    }
};

template<uint32_t MinimumSegmentSize, int32_t MaxProcVaLists>
class IoVaManager
{
public:
    IoVaManager() { }
    ~IoVaManager() { }
    void Initialize(BusAddress heapBase, uint32_t heapSize)
    {
        m_HeapBase = heapBase;
        m_HeapSize = heapSize;
        m_MapCount = 0;
        if(!m_FreeSizeList.empty()) NN_PCIE_ABORT("IoVaManager::Initialize() invalid state.\n");
        m_FreeSizeList.insert(std::make_pair(heapBase, IoVaFreeSegment(heapSize, heapBase)));
    }
    void Finalize()
    {
        m_FreeSizeList.clear();
        m_AllocatedIoVaList.clear();
        for(int32_t clientIndex=0; clientIndex < MaxProcVaLists; clientIndex++)
        {
            m_AllocatedProcVaList[clientIndex].clear();
        }
    }
    BusAddress Allocate(uint32_t requestedSize, uint32_t alignment, int32_t procVaIndex, uint64_t procVa,
                        uint64_t context, int32_t tag)
    {
        BusAddress ioVa = 0ULL;
        uint32_t alignmentR = NN_PCIE_ROUNDUP_SIZE(alignment, MinimumSegmentSize);
        uint32_t sizeR = NN_PCIE_ROUNDUP_SIZE(requestedSize, alignmentR);
        if(procVa & (static_cast<uint64_t>(alignment) - 1))
        {
            NN_PCIE_ABORT("IoVaManager::Allocate() misaligned procVa=%llx.\n",procVa);
            return 0ULL;
        }
        if((procVaIndex >= MaxProcVaLists) || (procVaIndex < 0))
        {
            NN_PCIE_ABORT("IoVaManager::Allocate() invalid procVaIndex=%d.\n",procVaIndex);
            return 0ULL;
        }
        for (FreeSizeListType::iterator itr = m_FreeSizeList.lower_bound(sizeR); itr != m_FreeSizeList.end(); itr++)
        {
            IoVaFreeSegment allocSeg = itr->second;
            BusAddress ioVaAligned = NN_PCIE_ROUNDUP_SIZE(allocSeg.GetIoVa(), alignmentR);
            uint32_t offset = static_cast<uint32_t>(ioVaAligned - allocSeg.GetIoVa());
            uint32_t remainingSize = allocSeg.GetSize() - offset;
            if((allocSeg.GetSize() > offset) && (remainingSize >= sizeR))
            {
                uint32_t allocSegSize = allocSeg.GetSize();

                // remove from free list, it's now allocated
                m_FreeSizeList.erase(itr);

                // do we need to break up this segment?
                if((remainingSize - sizeR) > MinimumSegmentSize)
                {
                    // make new free segment out of surplus
                    uint32_t surplusSize = remainingSize - sizeR;
                    BusAddress surplusIoVa =  allocSeg.GetIoVa() + offset + sizeR;
                    m_FreeSizeList.insert(std::make_pair(surplusIoVa,
                                                         IoVaFreeSegment(surplusSize, surplusIoVa)));
                    allocSegSize = allocSegSize - surplusSize;
                }

                // create ioVa record of allocation
                if(m_AllocatedIoVaList.insert(std::make_pair(allocSeg.GetIoVa(),
                                                             IoVaAllocatedSegment(allocSegSize, offset, allocSeg.GetIoVa(),
                                                                                  requestedSize, procVaIndex, procVa,
                                                                                  context, tag))).second == false)
                {
                    NN_PCIE_ABORT("IoVaManager::Allocate() failed to insert into m_AllocatedIoVaList.\n");
                    return 0ULL;
                }

                // pass back allocated ioVa
                ioVa = allocSeg.GetIoVa() + offset;

                // create procVa record of allocation
                if(procVa != 0ULL)
                {
                    if(m_AllocatedProcVaList[procVaIndex].insert(std::make_pair(procVa,
                                                                                IoVaAllocatedSegment(allocSegSize, offset, allocSeg.GetIoVa(),
                                                                                                     requestedSize, procVaIndex, procVa,
                                                                                                     context, tag))).second == false)
                    {
                        NN_PCIE_ABORT("IoVaManager::Allocate() failed to insert procVa=0x%llx into m_AllocatedProcVaList.\n",procVa);
                        return 0ULL;
                    }
                }

                break;
            }
        }
        m_MapCount++;
        return ioVa;
    }
    Result Free(BusAddress ioVa)
    {
        if(m_AllocatedIoVaList.empty())
        {
            NN_PCIE_LOG_INFO("IoVaManager::Free(ioVa=%llx) failed to locate ioVa, empty list.\n", ioVa);
            return ResultNonExistentDmaMapping();
        }
        auto locatedIoVaMap = m_AllocatedIoVaList.find(ioVa);
        if(locatedIoVaMap == m_AllocatedIoVaList.end())
        {
            AllocatedProcVaListType::iterator itr = m_AllocatedIoVaList.upper_bound(ioVa);
            locatedIoVaMap = (itr==m_AllocatedIoVaList.begin()) ? (itr) : (--itr);
            if(ioVa < locatedIoVaMap->second.GetIoVa())
            {
                NN_PCIE_LOG_VERBOSE("IoVaManager::Free(ioVa=%llx) did not locate ioVa, out of bounds 1.\n",
                                    ioVa);
                return ResultNonExistentDmaMapping();
            }
            uint64_t offset = ioVa - locatedIoVaMap->second.GetIoVa();
            if(offset >= locatedIoVaMap->second.GetSize())
            {
                NN_PCIE_LOG_VERBOSE("IoVaManager::Free(ioVa=%llx) did not locate procVa, out of bounds 2.\n",
                                    ioVa);
                return ResultNonExistentDmaMapping();
            }
        }

        // make a local copy, since we are about to remove it from the map
        IoVaAllocatedSegment freedSegment = locatedIoVaMap->second;

        // release procVa from allocated map if provided
        if(freedSegment.GetProcVa()!=0ULL)
        {
            m_AllocatedProcVaList[freedSegment.GetProcVaIndex()].erase(freedSegment.GetProcVa());
        }

        // release ioVa from allocated map
        m_AllocatedIoVaList.erase(locatedIoVaMap);

        // add iova to free map
        m_FreeSizeList.insert(std::make_pair(freedSegment.GetBaseIoVa(),
                                             IoVaFreeSegment(freedSegment.GetBaseSize(),
                                                             freedSegment.GetBaseIoVa())));

        // Until we have logic to coalesce freed segments, re-init free list when last map is freed
        if(--m_MapCount == 0)
        {
            m_FreeSizeList.clear();
            m_FreeSizeList.insert(std::make_pair(m_HeapBase, IoVaFreeSegment(m_HeapSize, m_HeapBase)));
        }

        return ResultSuccess();
    }
    Result FreeByTag(int32_t tag, void* calloutContext,
                     void (*callback)(void* calloutContext,IoVaAllocationSummary& summary))
    {
        Result result = ResultSuccess();
        for (AllocatedIoVaListType::iterator itr = m_AllocatedIoVaList.begin(); itr != m_AllocatedIoVaList.end(); )
        {
            IoVaAllocatedSegment freedSegment = itr->second;
            itr++;
            if(freedSegment.GetTag() == tag)
            {
                IoVaAllocationSummary summary;
                summary.requestedSize = freedSegment.GetRequestedSize();
                summary.procVaIndex   = freedSegment.GetProcVaIndex();
                summary.procVa        = freedSegment.GetProcVa();
                summary.ioVa          = freedSegment.GetIoVa();
                summary.context       = freedSegment.GetContext();
                summary.tag           = freedSegment.GetTag();
                if(callback) (*callback)(calloutContext, summary);
                NN_PCIE_ABORT_UPON_ERROR(Free(freedSegment.GetBaseIoVa()));
            }
        }
        return result;
    }
    Result GetProcVa(BusAddress ioVa, int32_t procVaIndex, uint64_t* pReturnedProcVa)
    {
        Result result = ResultSuccess();
        uint64_t offset = 0ULL;
        if((procVaIndex >= MaxProcVaLists) || (procVaIndex < 0))
        {
            NN_PCIE_LOG_INFO("IoVaManager::GetProcVa(ioVa=%llx, procVaIndex=%d) invalid procVaIndex\n",
                             ioVa, procVaIndex);
            return ResultNonExistentDmaMapping();
        }
        if(m_AllocatedIoVaList.empty())
        {
            NN_PCIE_LOG_VERBOSE("IoVaManager::GetProcVa(ioVa=%llx, procVaIndex=%d) did not locate procVa, empty list.\n",
                                ioVa, procVaIndex);
            return ResultNonExistentDmaMapping();
        }
        auto locatedIoVaMap = m_AllocatedIoVaList.find(ioVa);
        if(locatedIoVaMap == m_AllocatedIoVaList.end())
        {
            AllocatedIoVaListType::iterator itr = m_AllocatedIoVaList.upper_bound(ioVa);
            locatedIoVaMap = (itr==m_AllocatedIoVaList.begin()) ? (itr) : (--itr);
            if(ioVa < locatedIoVaMap->second.GetIoVa())
            {
                NN_PCIE_LOG_VERBOSE("IoVaManager::GetProcVa(ioVa=%llx, procVaIndex=%d) did not locate ioVa, out of bounds 1.\n",
                ioVa, procVaIndex);
                return ResultNonExistentDmaMapping();
            }
            offset = ioVa - locatedIoVaMap->second.GetIoVa();
            if(offset >= locatedIoVaMap->second.GetSize())
            {
                NN_PCIE_LOG_VERBOSE("IoVaManager::GetProcVa(ioVa=%llx, procVaIndex=%d) did not locate ioVa, out of bounds 2.\n",
                                    ioVa, procVaIndex);
                return ResultNonExistentDmaMapping();
            }
        }
        auto locatedProcVaMap = m_AllocatedProcVaList[procVaIndex].find(locatedIoVaMap->second.GetProcVa());
        if(locatedProcVaMap == m_AllocatedProcVaList[procVaIndex].end())
        {
            NN_PCIE_ABORT("IoVaManager::GetProcVa(ioVa=%llx, procVaIndex=%d) inconsistent procVa map.\n",
                          ioVa, procVaIndex);
            return ResultNonExistentDmaMapping();
        }
        *pReturnedProcVa = locatedProcVaMap->second.GetProcVa() + offset;
        return result;
    }
    Result GetIoVa(int32_t procVaIndex, uint64_t procVa, BusAddress* pReturnedIoVa)
    {
        Result result = ResultSuccess();
        uint64_t offset = 0ULL;
        if((procVaIndex >= MaxProcVaLists) || (procVaIndex < 0))
        {
            NN_PCIE_LOG_INFO("IoVaManager::GetIoVa(procVa=%llx, procVaIndex=%d) invalid procVaIndex\n",
                             procVa, procVaIndex);
            return ResultNonExistentDmaMapping();
        }
        if(m_AllocatedProcVaList[procVaIndex].empty())
        {
            NN_PCIE_LOG_VERBOSE("IoVaManager::GetIoVa(procVa=%llx, procVaIndex=%d) did not locate procVa, empty list.\n",
                                procVa, procVaIndex);
            return ResultNonExistentDmaMapping();
        }
        auto locatedProcVaMap = m_AllocatedProcVaList[procVaIndex].find(procVa);
        if(locatedProcVaMap == m_AllocatedProcVaList[procVaIndex].end())
        {
            AllocatedProcVaListType::iterator itr = m_AllocatedProcVaList[procVaIndex].upper_bound(procVa);
            locatedProcVaMap = (itr==m_AllocatedProcVaList[procVaIndex].begin()) ? (itr) : (--itr);
            if(procVa < locatedProcVaMap->second.GetProcVa())
            {
                NN_PCIE_LOG_VERBOSE("IoVaManager::GetIoVa(procVa=%llx, procVaIndex=%d) did not locate procVa, out of bounds 1.\n",
                                    procVa, procVaIndex);
                return ResultNonExistentDmaMapping();
            }
            offset = procVa - locatedProcVaMap->second.GetProcVa();
            if(offset >= locatedProcVaMap->second.GetSize())
            {
                NN_PCIE_LOG_VERBOSE("IoVaManager::GetIoVa(procVa=%llx, procVaIndex=%d) did not locate procVa, out of bounds 2.\n",
                                    procVa, procVaIndex);
                return ResultNonExistentDmaMapping();
            }
        }
        *pReturnedIoVa = locatedProcVaMap->second.GetIoVa() + offset;
        return result;
    }
    Result LookupByIoVa(BusAddress ioVa,  IoVaAllocationSummary* pReturnedSummary)
    {
        Result result = ResultSuccess();
        uint64_t offset = 0ULL;
        if(m_AllocatedIoVaList.empty())
        {
            return ResultNonExistentDmaMapping();
        }
        auto locatedIoVaMap = m_AllocatedIoVaList.find(ioVa);
        if(locatedIoVaMap == m_AllocatedIoVaList.end())
        {
            AllocatedIoVaListType::iterator itr = m_AllocatedIoVaList.upper_bound(ioVa);
            locatedIoVaMap = (itr==m_AllocatedIoVaList.begin()) ? (itr) : (--itr);
            if(ioVa < locatedIoVaMap->second.GetIoVa())
            {
                //NN_PCIE_LOG_VERBOSE("IoVaManager::LookupByIoVa(ioVa=%llx) did not locate ioVa, out of bounds 1.\n", ioVa);
                return ResultNonExistentDmaMapping();
            }
            offset = ioVa - locatedIoVaMap->second.GetIoVa();
            if(offset >= locatedIoVaMap->second.GetSize())
            {
                //NN_PCIE_LOG_VERBOSE("IoVaManager::LookupByIoVa(ioVa=%llx) did not locate ioVa, out of bounds 2.\n", ioVa);
                return ResultNonExistentDmaMapping();
            }
        }
        if(pReturnedSummary!=NULL)
        {
            pReturnedSummary->requestedSize = locatedIoVaMap->second.GetRequestedSize();
            pReturnedSummary->procVaIndex   = locatedIoVaMap->second.GetProcVaIndex();
            pReturnedSummary->procVa        = locatedIoVaMap->second.GetProcVa();
            pReturnedSummary->ioVa          = locatedIoVaMap->second.GetIoVa();
            pReturnedSummary->context       = locatedIoVaMap->second.GetContext();
        }
        return result;
    }
    Result LookupByProcVa(int32_t procVaIndex, uint64_t procVa, IoVaAllocationSummary* pReturnedSummary)
    {
        Result result = ResultSuccess();
        uint64_t offset = 0ULL;
        if((procVaIndex >= MaxProcVaLists) || (procVaIndex < 0))
        {
            NN_PCIE_LOG_INFO("IoVaManager::LookupByProcVa(procVa=%llx, procVaIndex=%d) invalid procVaIndex\n",
                             procVa, procVaIndex);
            return ResultNonExistentDmaMapping();
        }
        if(m_AllocatedProcVaList[procVaIndex].empty())
        {
            NN_PCIE_LOG_VERBOSE("LookupByProcVa::GetIoVa(procVa=%llx, procVaIndex=%d) did not locate procVa, empty list.\n",
                                procVa, procVaIndex);
            return ResultNonExistentDmaMapping();
        }
        auto locatedProcVaMap = m_AllocatedProcVaList[procVaIndex].find(procVa);
        if(locatedProcVaMap == m_AllocatedProcVaList[procVaIndex].end())
        {
            AllocatedProcVaListType::iterator itr = m_AllocatedProcVaList[procVaIndex].upper_bound(procVa);
            locatedProcVaMap = (itr==m_AllocatedProcVaList[procVaIndex].begin()) ? (itr) : (--itr);
            if(procVa < locatedProcVaMap->second.GetProcVa())
            {
                NN_PCIE_LOG_VERBOSE("LookupByProcVa::GetIoVa(procVa=%llx, procVaIndex=%d) did not locate procVa, out of bounds 1.\n",
                                    procVa, procVaIndex);
                return ResultNonExistentDmaMapping();
            }
            offset = procVa - locatedProcVaMap->second.GetProcVa();
            if(offset >= locatedProcVaMap->second.GetSize())
            {
                NN_PCIE_LOG_VERBOSE("LookupByProcVa::GetIoVa(procVa=%llx, procVaIndex=%d) did not locate procVa, out of bounds 2.\n",
                                    procVa, procVaIndex);
                return ResultNonExistentDmaMapping();
            }
        }
        if(pReturnedSummary!=NULL)
        {
            pReturnedSummary->requestedSize = locatedProcVaMap->second.GetRequestedSize();
            pReturnedSummary->procVaIndex   = locatedProcVaMap->second.GetProcVaIndex();
            pReturnedSummary->procVa        = locatedProcVaMap->second.GetProcVa();
            pReturnedSummary->ioVa          = locatedProcVaMap->second.GetIoVa();
            pReturnedSummary->context       = locatedProcVaMap->second.GetContext();
        }
        return result;
    }
private:
    typedef std::multimap<uint32_t, IoVaFreeSegment, std::less<uint32_t>,
        IoVaSegmentAllocator<std::pair<const uint32_t, IoVaFreeSegment>>> FreeSizeListType;
    typedef std::map<BusAddress, IoVaAllocatedSegment, std::less<BusAddress>,
        IoVaSegmentAllocator<std::pair<const BusAddress, IoVaAllocatedSegment>>> AllocatedIoVaListType;
    typedef std::map<uint64_t, IoVaAllocatedSegment, std::less<uint64_t>,
        IoVaSegmentAllocator<std::pair<const uint64_t, IoVaAllocatedSegment>>> AllocatedProcVaListType;
    FreeSizeListType        m_FreeSizeList;
    AllocatedIoVaListType   m_AllocatedIoVaList;
    AllocatedProcVaListType m_AllocatedProcVaList[MaxProcVaLists];
    int32_t                 m_MapCount;
    BusAddress              m_HeapBase;
    uint32_t                m_HeapSize;

};


} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn
