﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace olsc { namespace srv { namespace util {

Result CreateFile(const char* path, size_t fileSize) NN_NOEXCEPT
{
    NN_RESULT_TRY(fs::CreateFile(path, static_cast<int64_t>(fileSize)))
        NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
        {
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

Result DeleteFile(const char* path) NN_NOEXCEPT
{
    NN_RESULT_TRY(fs::DeleteFile(path))
        NN_RESULT_CATCH(fs::ResultPathNotFound) {}
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

Result ReadFile(const char* path, void* out, size_t maxReadSize, int64_t offset) NN_NOEXCEPT
{
    size_t readSize;
    NN_RESULT_DO(ReadFile(&readSize, path, out, maxReadSize, offset));
    NN_ABORT_UNLESS(readSize == maxReadSize, "invalid read size.");
    NN_RESULT_SUCCESS;
}

Result ReadFile(size_t* outReadSize, const char* path, void* out, size_t maxReadSize, int64_t offset) NN_NOEXCEPT
{
    fs::FileHandle file;

    NN_RESULT_TRY(fs::OpenFile(&file, path, fs::OpenMode_Read))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
    NN_RESULT_END_TRY;
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

    NN_RESULT_DO(fs::ReadFile(outReadSize, file, offset, out, maxReadSize));
    NN_RESULT_SUCCESS;
}

Result WriteFile(const char* path, const void* data, size_t dataSize) NN_NOEXCEPT
{
    NN_RESULT_TRY(CreateFile(path, dataSize))
        NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
        {
            // 既にファイルが存在する場合はエラーを無視して、ファイルオープン後にファイルサイズを調整する。
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
    NN_RESULT_END_TRY;

    fs::FileHandle file;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&file, path, fs::OpenMode_Write));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::SetFileSize(file, dataSize));

    auto option = fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush);
    NN_RESULT_DO(fs::WriteFile(file, 0, data, dataSize, option));
    NN_RESULT_SUCCESS;
}

Result ModifyFile(const char* path, const void* data, size_t dataSize, int64_t offset) NN_NOEXCEPT
{
    fs::FileHandle file;
    NN_RESULT_TRY(fs::OpenFile(&file, path, fs::OpenMode_Write))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            NN_RESULT_RETHROW;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
        }
    NN_RESULT_END_TRY;
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

    int64_t fileSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetFileSize(&fileSize, file));
    NN_ABORT_UNLESS(static_cast<int64_t>(offset + dataSize) <= fileSize, "[nn::olsc] File size will exceeds the limit.\n");
    NN_ABORT_UNLESS(offset < fileSize, "[nn::olsc] Specified offset of file exceeds current file size.\n");

    auto option = fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush);
    NN_RESULT_DO(fs::WriteFile(file, offset, data, dataSize, option));
    NN_RESULT_SUCCESS;
}


}}}} //namespace nn::olsc::srv::database
