﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/olsc/srv/transfer/olsc_TransferUtil.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveDownload.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveUpload.h>

#include <nn/util/util_TFormatString.h>
#include <nn/util/util_Base64.h>

#include "../adaptor/olsc_SaveDataArchiveInfoAdaptor.h"
#include "../adaptor/olsc_MixedInfoAdaptor.h"

namespace nn { namespace olsc { namespace srv { namespace transfer {

// TODO : launch_required_version, has_thumbnail を引数に追加
Result StartSaveDataArchiveUpload(SaveDataArchiveInfo* out, const account::Uid& uid, ApplicationId appId, const SeriesInfo& info, size_t dataSize, time::PosixTime updatedTime, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // uid 未使用
    NN_UNUSED(uid);

    // 適当な値を入れる
    uint32_t launchRequiredVersion = 0;
    bool hasThumbnail = false;

    // URL の作成
    char url[256];
    auto l = CreateUrlForStartSaveDataArchiveUpload(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    adaptor::SaveDataArchiveInfoMemoryOutputStream outputStream(out, 1); // 最大 1 つ
    adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);

    const size_t DigestSize = 32;
    char digest[DigestSize] = {};
    memcpy(digest, &info.commitId, sizeof(info.commitId));

    static const size_t Base64Size = ((DigestSize * 4 + (3 - 1) ) / 3 ) + 2;
    char b64Value[Base64Size];
    auto b64Error = nn::util::Base64::ToBase64String(b64Value, sizeof(b64Value), digest, DigestSize, nn::util::Base64::Mode_UrlSafe);
    NN_ABORT_UNLESS(b64Error == nn::util::Base64::Status_Success);

    char jsonString[512];
    l = nn::util::TSNPrintf(jsonString, sizeof(jsonString),
        "{"
             "\"application_id\": \"%016llx\","
             "\"data_size\": %llu,"
             "\"saved_at_as_unixtime\": %lld," // Posix time は signed Integer
             "\"launch_required_version\": %llu,"
             "\"series_id\": %llu,"
             "\"encoded_digest\": \"%s\","
             "\"has_thumbnail\": %s,"
             "\"auto_backup\": false"
        "}"
        , appId.value, dataSize, updatedTime.value,
        static_cast<Bit64>(launchRequiredVersion), info.seriesId,
        b64Value, hasThumbnail ? "true" : "false"
    );
    NN_UNUSED(l);
    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, jsonString, curlHandle, workBuffer, workBufferSize, pCancelable));

    NN_RESULT_SUCCESS;
}

// TODO : launch_required_version, has_thumbnail を引数に追加
Result StartSaveDataArchiveDifferentialUpload(SaveDataArchiveInfo* pOutSda, int* pOutCfCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId id, const SeriesInfo& info, size_t dataSize, time::PosixTime updatedTime, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // 適当な値を入れる
    uint32_t launchRequiredVersion = 0;
    bool hasThumbnail = false;

    // URL の作成
    char url[256];
    auto l = CreateUrlForStartSaveDataArchiveDiffUpload(url, sizeof(url), id);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    adaptor::SaveDataArchiveInfoMemoryOutputStream sdaOutputStream(pOutSda, 1); // 最大 1 つ
    adaptor::ComponentFileInfoMemoryOutputStream cfOutputStream(componentFileList, static_cast<int>(listCount));
    adaptor::MixedInfoAdaptor adaptor(&sdaOutputStream, &cfOutputStream);

    const size_t DigestSize = 32;
    char digest[DigestSize] = {};
    memcpy(digest, &info.commitId, sizeof(info.commitId));

    static const size_t Base64Size = ((DigestSize * 4 + (3 - 1) ) / 3 ) + 2;
    char b64Value[Base64Size];
    auto b64Error = nn::util::Base64::ToBase64String(b64Value, sizeof(b64Value), digest, DigestSize, nn::util::Base64::Mode_UrlSafe);
    NN_ABORT_UNLESS(b64Error == nn::util::Base64::Status_Success);

    char jsonString[512];
    l = nn::util::TSNPrintf(jsonString, sizeof(jsonString),
        "{"
             "\"data_size\": %llu,"
             "\"saved_at_as_unixtime\": %lld," // Posix time は signed Integer
             "\"launch_required_version\": %llu,"
             "\"series_id\": %llu,"
             "\"encoded_digest\": \"%s\","
             "\"has_thumbnail\": %s,"
             "\"auto_backup\": false"
        "}"
        , dataSize, updatedTime.value,
        static_cast<Bit64>(launchRequiredVersion), info.seriesId,
        b64Value, hasThumbnail ? "true" : "false"
    );
    NN_UNUSED(l);
    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, jsonString, curlHandle, workBuffer, workBufferSize, pCancelable));

    *pOutCfCount = cfOutputStream.GetCount();
    NN_RESULT_SUCCESS;
}

// TODO : KeySeed と mac は fs に実装後 fs のものを使う
Result FinishSaveDataArchiveUpload(SaveDataArchiveId sdaId, const srv::KeySeed& keySeed, const srv::InitialDataMac& mac, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForFinishSaveDataArchiveUpload(url, sizeof(url), sdaId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    SaveDataArchiveInfo sda;

    adaptor::SaveDataArchiveInfoMemoryOutputStream outputStream(&sda, 1); // 最大 1 つ
    adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);

    static const size_t Base64SizeForKeySeed = ((srv::KeySeed::Size * 4 + (3 - 1) ) / 3 ) + 2;
    char b64ValueForKeySeed[Base64SizeForKeySeed];
    auto b64Error = nn::util::Base64::ToBase64String(b64ValueForKeySeed, sizeof(b64ValueForKeySeed), keySeed.data, sizeof(keySeed.data), nn::util::Base64::Mode_UrlSafe);
    NN_ABORT_UNLESS(b64Error == nn::util::Base64::Status_Success);

    static const size_t Base64SizeForMac = ((srv::InitialDataMac::Size * 4 + (3 - 1) ) / 3 ) + 2;
    char b64ValueForMac[Base64SizeForMac];
    b64Error = nn::util::Base64::ToBase64String(b64ValueForMac, sizeof(b64ValueForMac), mac.data, sizeof(mac.data), nn::util::Base64::Mode_UrlSafe);
    NN_ABORT_UNLESS(b64Error == nn::util::Base64::Status_Success);

    char jsonString[512];
    l = nn::util::TSNPrintf(jsonString, sizeof(jsonString),
        "{"
             "\"encoded_key_seed\": \"%s\","
             "\"encoded_mac\": \"%s\""
        "}"
        , b64ValueForKeySeed, b64ValueForMac
    );
    NN_UNUSED(l);

    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, jsonString, curlHandle, workBuffer, workBufferSize, pCancelable));

    // finish_upload でも sda がサーバーから返ってくるがこの sda は使用予定なしなので一応 sdaId の一致確認だけして終わり
    NN_SDK_ASSERT(sda.id == sdaId);

    NN_RESULT_SUCCESS;
}

// TODO : mac は fs に実装後 fs のものを使う
Result FinishSaveDataArchiveDifferentialUpload(SaveDataArchiveId sdaId, const srv::InitialDataMac& mac, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForFinishSaveDataArchiveUpload(url, sizeof(url), sdaId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    SaveDataArchiveInfo sda;
    adaptor::SaveDataArchiveInfoMemoryOutputStream outputStream(&sda, 1); // 最大 1 つ
    adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);

    static const size_t Base64SizeForMac = ((srv::InitialDataMac::Size * 4 + (3 - 1) ) / 3 ) + 2;
    char b64ValueForMac[Base64SizeForMac];
    auto b64Error = nn::util::Base64::ToBase64String(b64ValueForMac, sizeof(b64ValueForMac), mac.data, sizeof(mac.data), nn::util::Base64::Mode_UrlSafe);
    NN_ABORT_UNLESS(b64Error == nn::util::Base64::Status_Success);

    char jsonString[512];
    l = nn::util::TSNPrintf(jsonString, sizeof(jsonString),
        "{"
             "\"encoded_mac\": \"%s\""
        "}"
        , b64ValueForMac
    );
    NN_UNUSED(l);

    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, jsonString, curlHandle, workBuffer, workBufferSize, pCancelable));

    // finish_upload でも sda がサーバーから返ってくるがこの sda は使用予定なしなので一応 sdaId の一致確認だけして終わり
    NN_SDK_ASSERT(sda.id == sdaId);

    NN_RESULT_SUCCESS;
}

Result DeleteSaveDataArchive(SaveDataArchiveId sdaId, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForDeleteSaveDataArchive(url, sizeof(url), sdaId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    SaveDataArchiveInfo sda;

    adaptor::SaveDataArchiveInfoMemoryOutputStream outputStream(&sda, 1); // 最大 1 つ
    adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);
    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, "", curlHandle, workBuffer, workBufferSize, pCancelable));

    NN_RESULT_SUCCESS;
}

Result ExtendSaveDataArchiveUploadTimeout(SaveDataArchiveInfo* pOutSda, int* pOutCfCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId sdaId, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForExtendSaveDataArchiveUploadTimeout(url, sizeof(url), sdaId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    adaptor::SaveDataArchiveInfoMemoryOutputStream sdaOutputStream(pOutSda, 1); // 最大 1 つ
    adaptor::ComponentFileInfoMemoryOutputStream cfOutputStream(componentFileList, static_cast<int>(listCount));
    adaptor::MixedInfoAdaptor adaptor(&sdaOutputStream, &cfOutputStream);

    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, "", curlHandle, workBuffer, workBufferSize, pCancelable));

    *pOutCfCount = cfOutputStream.GetCount();

    NN_RESULT_SUCCESS;
}

}}}} //namespace nn::olsc::srv::transfer

