﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/olsc/srv/transfer/olsc_TransferUtil.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveDownload.h>
#include <nn/olsc/srv/transfer/olsc_ComponentFileUpload.h>

#include <nn/util/util_TFormatString.h>
#include <nn/util/util_Base64.h>

#include "../adaptor/olsc_ComponentFileInfoAdaptor.h"

#include "olsc_Uploader.h"

namespace nn { namespace olsc { namespace srv { namespace transfer {

Result GetComponentFileSignedUrlForUpload(ComponentFileInfo* pOut, ComponentFileId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForGetComponentFileInfo(url, sizeof(url), id);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    adaptor::ComponentFileInfoMemoryOutputStream outputStream(pOut, 1); // 最大 1 つ
    adaptor::ComponentFileInfoAdaptor adaptor(&outputStream);

    char jsonString[128];

    l = nn::util::TSNPrintf(jsonString, sizeof(jsonString),
        "{"
             "\"permissions\": [\"write\"]"
        "}");

    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, jsonString, curlHandle, workBuffer, workBufferSize, pCancelable));

    NN_SDK_ASSERT(outputStream.GetCount() == 1);
    NN_RESULT_SUCCESS;
}

Result CreateComponentFile(ComponentFileInfo* out, SaveDataArchiveId saveDataArchiveId, const ClientArgument& clientArgument, ComponentFileType type, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForCreateComponentFile(url, sizeof(url), saveDataArchiveId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    adaptor::ComponentFileInfoMemoryOutputStream outputStream(out, 1); // 最大 1 つ
    adaptor::ComponentFileInfoAdaptor adaptor(&outputStream);

    char jsonString[256];
    l = nn::util::TSNPrintf(jsonString, sizeof(jsonString),
        "{"
             "\"index\": %llu,"
             "\"datatype\": \"%s\""
        "}"
        , clientArgument.chunkId, type == ComponentFileType::Meta ? "meta" : "save"
        );
    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, jsonString, curlHandle, workBuffer, workBufferSize, pCancelable));
    NN_RESULT_SUCCESS;
}

Result UpdateComponentFile(ComponentFileInfo* out, ComponentFileId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForUpdateComponentFile(url, sizeof(url), id);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    adaptor::ComponentFileInfoMemoryOutputStream outputStream(out, 1); // 最大 1 つ
    adaptor::ComponentFileInfoAdaptor adaptor(&outputStream);

    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, "", curlHandle, workBuffer, workBufferSize, pCancelable));

    NN_SDK_ASSERT(out->id == id);

    NN_RESULT_SUCCESS;
}

Result ExportComponentFile(size_t* pOutSize, SaveDataChunkDigest* pOutDigest, const char* url, fs::ISaveDataDivisionExporter* exporter, fs::SaveDataChunkId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // 空回ししてサイズとハッシュを取得
    size_t dataSize;
    SaveDataChunkDigest digest;
    NN_RESULT_DO(GetExportSizeByDryRun(&dataSize, &digest, exporter, id, workBuffer, workBufferSize));

    std::unique_ptr<fs::ISaveDataChunkExporter> chunkExporter;
    NN_RESULT_DO(exporter->OpenSaveDataChunkExporter(&chunkExporter, id));

    FsExporterUploader uploader(curlHandle, pCancelable);
    NN_RESULT_DO(uploader.Initialize(dataSize));
    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };
    headers = curl_slist_append(headers, "Expect:");
    char buffer[256];
    auto l = nn::util::SNPrintf(buffer, 256, "Content-Length: %lld", dataSize);
    NN_UNUSED(l);
    headers = curl_slist_append(headers, buffer);
    NN_RESULT_THROW_UNLESS(headers != nullptr, olsc::ResultInvalidArgument());
    // リクエスト生成
    NN_RESULT_DO(uploader.SetHeaders(headers));
    NN_RESULT_DO(uploader.SetTimeoutSeconds(static_cast<int32_t>(GetTransferTimeout(dataSize).GetSeconds())));
    NN_RESULT_DO(uploader.SetUrl(url));
    NN_RESULT_DO(uploader.Upload(chunkExporter.get(), workBuffer, workBufferSize));

    // アップロードに成功後出力値を設定
    *pOutSize = dataSize;
    *pOutDigest = digest;
    NN_RESULT_SUCCESS;
}

Result CompleteComponentFile(ComponentFileId id, size_t archiveDataSize, const SaveDataChunkDigest& saveDataChunkDigest, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForCompleteComponentFile(url, sizeof(url), id);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    ComponentFileInfo cf;

    adaptor::ComponentFileInfoMemoryOutputStream outputStream(&cf, 1); // 最大 1 つ
    adaptor::ComponentFileInfoAdaptor adaptor(&outputStream);

    static const size_t Base64Size = ((SaveDataChunkDigest::Size * 4 + (3 - 1) ) / 3 ) + 2;
    char b64Value[Base64Size];
    auto b64Error = nn::util::Base64::ToBase64String(b64Value, sizeof(b64Value), saveDataChunkDigest.data, sizeof(saveDataChunkDigest.data), nn::util::Base64::Mode_UrlSafe);
    NN_ABORT_UNLESS(b64Error == nn::util::Base64::Status_Success);

    char jsonString[256];
    l = nn::util::TSNPrintf(jsonString, sizeof(jsonString),
        "{"
             "\"archive_size\": %llu,"
             "\"encoded_archive_digest\": \"%s\""
        "}"
        , archiveDataSize, b64Value
        );
    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, jsonString, curlHandle, workBuffer, workBufferSize, pCancelable));

    // finish_upload でも cf がサーバーから返ってくるがこの cf は使用予定なしなので一応 id の一致確認だけして終わり
    NN_SDK_ASSERT(cf.id == id);

    NN_RESULT_SUCCESS;
}

}}}} //namespace nn::olsc::srv::transfer

