﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/olsc/srv/database/olsc_AutoUploadSettingDatabase.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/util/util_TFormatString.h>

namespace nn { namespace olsc { namespace srv { namespace database {

// ------------------------------------------------------------------------------
// Private
// ------------------------------------------------------------------------------
Result AutoUploadSettingDatabase::SetGlobalSettingImpl(bool isEnabled) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_GeneralSettingLock.IsLockedByCurrentThread());

    auto writeMount = m_MountManager.AcquireUserSettingsSaveForWrite(m_Uid);

    m_GlobalSetting.isEnabled = isEnabled;

    auto modifyFile = [this](const char* path) {
        return util::ModifyFile(path, &m_GlobalSetting, sizeof(m_GlobalSetting), 0);
    };

    char path[MaxPathLen];
    MakeGlobalSettingFilePath(path, MaxPathLen, writeMount.GetRootPath());
    NN_RESULT_TRY(modifyFile(path))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            NN_RESULT_DO(util::CreateFile(path, sizeof(m_GlobalSetting)));
            NN_RESULT_DO(modifyFile(path));
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

Result AutoUploadSettingDatabase::GetGlobalSettingImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_GeneralSettingLock.IsLockedByCurrentThread());

    auto readMount = m_MountManager.AcquireUserSettingsSaveForWrite(m_Uid);

    char path[MaxPathLen];
    MakeGlobalSettingFilePath(path, MaxPathLen, readMount.GetRootPath());

    NN_RESULT_DO(util::ReadFile(path, &m_GlobalSetting.isEnabled, sizeof(m_GlobalSetting.isEnabled), 0));
    NN_RESULT_SUCCESS;
}

Result AutoUploadSettingDatabase::ReadGlobalSetting() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_GeneralSettingLock)> lock(m_GeneralSettingLock);

    NN_RESULT_TRY(GetGlobalSettingImpl())
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            bool isEnabled;
            auto getSize = nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "olsc", "default_auto_upload_global_setting");
            NN_ABORT_UNLESS(getSize == sizeof(isEnabled));
            NN_RESULT_DO(SetGlobalSettingImpl(isEnabled));
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

const char* AutoUploadSettingDatabase::MakeGlobalSettingFilePath(char* out, int maxOutCount, const char* rootPath) const NN_NOEXCEPT
{
    nn::util::TSNPrintf(out, maxOutCount, "%s%s", rootPath, "auto_ul_global_setting");
    return out;
}

void AutoUploadSettingDatabase::SetApplicationSettingsAll(AutoUploadSettingDatabase::ApplicationAutoUploadSetting setting) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_SettingList)> lock(m_SettingList);

    for (int i = 0; i < m_SettingList.GetCount(); ++i)
    {
        auto kv = m_SettingList[i];
        ApplicationSetting appSetting = { setting };
        NN_ABORT_UNLESS(m_SettingList.Add(kv.key, appSetting));
    }
}

// ------------------------------------------------------------------------------
// Public
// ------------------------------------------------------------------------------

void AutoUploadSettingDatabase::Cleanup() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_SettingList)> lock(m_SettingList);

    m_SettingList.Cleanup();
}

void AutoUploadSettingDatabase::SetGlobalSetting(bool isEnabled) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_GeneralSettingLock)> lock(m_GeneralSettingLock);
    if (isEnabled == m_GlobalSetting.isEnabled)
    {
        return;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(SetGlobalSettingImpl(isEnabled));
    SetApplicationSettingsAll(isEnabled ? ApplicationAutoUploadSetting::Enabled : ApplicationAutoUploadSetting::Disabled);
}

bool AutoUploadSettingDatabase::GetGlobalSetting() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_GeneralSettingLock)> lock(m_GeneralSettingLock);
    return m_GlobalSetting.isEnabled;
}

void AutoUploadSettingDatabase::SetApplicationAutoUploadSetting(ApplicationId appId, AutoUploadSettingDatabase::ApplicationAutoUploadSetting setting) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_SettingList)> lock(m_SettingList);
    if (setting == ApplicationAutoUploadSetting::NotSet)
    {
        m_SettingList.Remove(appId);
        return;
    }

    ApplicationSetting appSetting = { setting };
    NN_ABORT_UNLESS(m_SettingList.Add(appId, appSetting));
}

AutoUploadSettingDatabase::ApplicationAutoUploadSetting AutoUploadSettingDatabase::GetApplicationAutoUploadSetting(ApplicationId appId) const NN_NOEXCEPT
{
    std::lock_guard<const decltype(m_SettingList)> lock(m_SettingList);
    auto setting = m_SettingList.Find(appId);
    return !setting ? AutoUploadSettingDatabase::ApplicationAutoUploadSetting::NotSet : (*setting).setting;
}

void AutoUploadSettingDatabase::RemoveApplicationAutoUploadSetting(ApplicationId appId) NN_NOEXCEPT
{
    std::lock_guard<const decltype(m_SettingList)> lock(m_SettingList);
    m_SettingList.Remove(appId);
}

void AutoUploadSettingDatabase::ForEach(ForEachPredicate pred) NN_NOEXCEPT
{
    std::lock_guard<const decltype(m_SettingList)> lock(m_SettingList);
    for (int i = 0; i < m_SettingList.GetCount(); ++i)
    {
        auto& kv = m_SettingList[i];
        pred(kv.key, kv.value.setting);
    }
}

}}}} //namespace nn::olsc::srv::database

