﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "olsc_PolicyInfoAdaptor.h"

#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/srv/olsc_Util.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_StringUtil.h>


namespace nn { namespace olsc { namespace srv { namespace adaptor {

namespace {
    const int MaxPropertyCount = PolicyInfoBuilder::MaxPropertyCount;
    enum FlagIndex : int
    {
        FlagIndex_PolicyType = 0,
        FlagIndex_PropertyCount // Property 数
    };

    const char FieldString[][PolicyInfoAdaptor::MaxPathLength] =
    {
        ".policy_type"
    };

    struct Flags
    {
        typedef typename nn::util::BitFlagSet<MaxPropertyCount>::template Flag<FlagIndex_PolicyType> PolicyType;
    };
} // namespace

PolicyInfoBuilder::PolicyInfoBuilder() NN_NOEXCEPT
{
    Reset();
}

void PolicyInfoBuilder::Reset() NN_NOEXCEPT
{
    m_Imported.Reset();
}

bool PolicyInfoBuilder::Validate() NN_NOEXCEPT
{
    // 必要な要素が全て設定されているかどうかの判定
    if(!m_Imported[FlagIndex_PolicyType])
    {
        NN_DETAIL_OLSC_WARN("[Error] PolicyInfo Validate Error\n");
        return false;
    }
    return true;
}

const PolicyInfo& PolicyInfoBuilder::GetPolicyInfo() const NN_NOEXCEPT
{
    return m_PolicyInfo;
}

template<typename FlagType>
void PolicyInfoBuilder::SetFlag() NN_NOEXCEPT
{
    m_Imported.template Set<FlagType>();
}

template<typename FlagType>
bool PolicyInfoBuilder::IsUnset() const NN_NOEXCEPT
{
    return !m_Imported.template Test<FlagType>();
}

void PolicyInfoBuilder::SetPolicyType(PolicyType type) NN_NOEXCEPT
{
    m_PolicyInfo.type = type;
    SetFlag<Flags::PolicyType>();
}

// ----------------------

bool PolicyInfoAdaptor::UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT
{
    NN_UNUSED(valueLength);
    if(!m_IsAcceptable)
    {
        return true;
    }

    size_t fullPathLength = 0;
    const char* FullPath = jsonPath.ToString(&fullPathLength);
    NN_SDK_ASSERT(fullPathLength > static_cast<size_t>(m_PathBufferLength), "%s %s pathLength Error", __FILE__, __FUNCTION__);

    if(m_Builder.IsUnset<Flags::PolicyType>())
    {
        if (nn::util::Strncmp(FullPath + m_PathBufferLength, FieldString[FlagIndex_PolicyType], MaxPathLength - m_PathBufferLength) == 0)
        {
            if (nn::util::Strncmp(value, "ALL_OK", valueLength + 1) == 0)
            {
                m_Builder.SetPolicyType(PolicyType::AllOk);
            }
            else if(nn::util::Strncmp(value, "ALL_NG", valueLength + 1) == 0)
            {
                m_Builder.SetPolicyType(PolicyType::AllNg);
            }
            // 無効な文字列はパース失敗
            else
            {
                NN_DETAIL_OLSC_WARN("PolicyInfo parse error. '%s' is invalid value\n", FieldString[FlagIndex_PolicyType]);
                m_IsAcceptable = false;
            }
            return true;
        }
    }
    return true;
} // NOLINT(impl/function_size)

bool PolicyInfoAdaptor::NotifyObjectBeginImpl(const JsonPathType& jsonPath) NN_NOEXCEPT
{
    if(MakeAndCheckObjectPath(m_PathBuffer, sizeof(m_PathBuffer), jsonPath))
    {
        m_Builder.Reset();
        m_PathBufferLength = nn::util::Strnlen(m_PathBuffer, MaxPathLength);
        // パス長さが許容範囲外なら、Validate できないだけで処理は続行
        m_IsAcceptable = m_PathBufferLength < MaxPathLength;
        if(!m_IsAcceptable)
        {
            NN_DETAIL_OLSC_WARN("PolicyInfo parse error. pathLength is invalid\n");
        }
    }
    return true;
}

bool PolicyInfoAdaptor::NotifyObjectEndImpl(const JsonPathType& jsonPath) NN_NOEXCEPT
{
    if (jsonPath.Match(m_PathBuffer))
    {
        m_PathBufferLength = 0;
        if(!(m_IsAcceptable && m_Builder.Validate()))
        {
            SetResult(ResultPolicyInfoUnacceptableContent());
            return false;
        }
        m_IsAcceptable = false;
    }
    return true;
}

bool PolicyInfoAdaptor::MakeAndCheckObjectPath(char* buffer, size_t bufferSize, const JsonPathType& jsonPath) NN_NOEXCEPT
{
    nn::util::TSNPrintf(buffer, bufferSize, "$");
    return jsonPath.Match(buffer);
}

Result PolicyInfoAdaptor::GetPolicyInfo(PolicyInfo* pOutValue) const NN_NOEXCEPT
{
    *pOutValue = m_Builder.GetPolicyInfo();
    NN_RESULT_SUCCESS;
}

}}}} //namespace nn::olsc::srv
