﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <functional>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_HipcSimpleClientSessionManager.h>

namespace nn { namespace olsc { namespace srv {

class IOlscServiceBase;
class IOlscServiceForSystemService;

}}}

namespace nn { namespace olsc { namespace detail {

const Bit32 InitializationManagerMagicNumber = 0x015c5413;
#define NN_OLSC_INITIALIZATION_MANAGER_INITIALIZER { \
    ::nn::olsc::detail::InitializationManagerMagicNumber, \
    0u, \
    NN_OS_MUTEX_INITIALIZER(false)}

/**
    @brief Shim ライブラリの初期化状態を管理する
 */
struct InitializationManager
{
    Bit32 _magicNumber;
    uint32_t _counter;
    os::MutexType _mutex;

    NN_EXPLICIT_OPERATOR bool() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == InitializationManagerMagicNumber);
        std::lock_guard<InitializationManager> lock(*this);
        return _counter > 0;
    }
    template <typename Initializer>
    void Initialize(const Initializer& initializer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == InitializationManagerMagicNumber);
        std::lock_guard<decltype(*this)> lock(*this);
        NN_ABORT_UNLESS(
            _counter < std::numeric_limits<uint32_t>::max(),
            "[nn::olsc] -----------------------------------------------\n"
            "  ABORT: Too much call for client initialization\n");
        if (_counter == 0)
        {
            initializer();
        }
        ++ _counter;
    }
    template <typename Finalizer>
    void Finalize(const Finalizer& finalizer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == InitializationManagerMagicNumber);
        std::lock_guard<decltype(*this)> lock(*this);
        NN_ABORT_UNLESS(
            _counter > 0,
            "[nn::olsc] -----------------------------------------------\n"
            "  ABORT: Too much call for client finalization\n");
        if (_counter == 1)
        {
            finalizer();
        }
        -- _counter;
    }
    void lock() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == InitializationManagerMagicNumber);
        os::LockMutex(&_mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == InitializationManagerMagicNumber);
        os::UnlockMutex(&_mutex);
    }
};


class ObjectHolder
{
public:
    ObjectHolder() NN_NOEXCEPT;
    ~ObjectHolder() NN_NOEXCEPT;

    Result AcquireForSystemService() NN_NOEXCEPT;
    void Release() NN_NOEXCEPT;
    nn::olsc::srv::IOlscServiceForSystemService& GetForSystemService() const NN_NOEXCEPT;

    void InitializeWithServiceObject(sf::SharedPointer<srv::IOlscServiceForSystemService> serviceObject) NN_NOEXCEPT;
private:
    struct ObjectHolderAllocatorTag {};
    static const size_t SizeOfHeapToAcquireObject = 1024 * 2;
    using AllocatorType = sf::ExpHeapStaticAllocator<SizeOfHeapToAcquireObject, ObjectHolderAllocatorTag>;
    nn::olsc::srv::IOlscServiceBase* m_ServiceBase;
    nn::olsc::srv::IOlscServiceForSystemService* m_ServiceForSystemService;

    static const int ClientSessionConcurrency = 1;
    nn::sf::HipcSimpleClientSessionManager m_HipcDomain;

    template <typename Interface>
    Result AcquireImpl(Interface** ppOutService, const char* serviceName) NN_NOEXCEPT;
};

}}} // namespace nn::olsc::detail
