﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "olsc_Initialization.h"

#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/olsc/srv/olsc_ServiceNames.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace olsc { namespace detail {

ObjectHolder::ObjectHolder() NN_NOEXCEPT
    : m_ServiceForSystemService(nullptr)
{
    AllocatorType::Initialize(lmem::CreationOption_NoOption);
}

ObjectHolder::~ObjectHolder() NN_NOEXCEPT
{
    Release();
}

Result ObjectHolder::AcquireForSystemService() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ServiceBase);
    NN_RESULT_DO(AcquireImpl(&m_ServiceForSystemService, nn::olsc::srv::ServiceNameForSystemService));
    m_ServiceBase = m_ServiceForSystemService;
    NN_RESULT_SUCCESS;
}

void ObjectHolder::InitializeWithServiceObject(sf::SharedPointer<srv::IOlscServiceForSystemService> serviceObject) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ServiceBase);
    m_ServiceForSystemService = serviceObject.Detach();
    m_ServiceBase = m_ServiceForSystemService;
}

void ObjectHolder::Release() NN_NOEXCEPT
{
    if (m_ServiceBase != nullptr)
    {
        sf::ReleaseSharedObject(m_ServiceBase);
        m_ServiceBase = m_ServiceForSystemService = nullptr;
        m_HipcDomain.Finalize();
    }
}

nn::olsc::srv::IOlscServiceForSystemService& ObjectHolder::GetForSystemService() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_ServiceForSystemService);
    return *m_ServiceForSystemService;
}

template<typename Interface>
Result ObjectHolder::AcquireImpl(Interface** outService, const char* serviceName) NN_NOEXCEPT
{
    nn::sf::SharedPointer<Interface> ptr;
    NN_RESULT_DO((m_HipcDomain.InitializeByName<Interface, AllocatorType::Policy>(&ptr, serviceName)));
    NN_RESULT_DO(m_HipcDomain.SetSessionCount(ClientSessionConcurrency));

    *outService = ptr.Detach();
    NN_ABORT_UNLESS(*outService != nullptr, "[nn::olsc] Initialization failure while connecting.\n");
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::olsc::detail
