﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_ScopeExit.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_SystemUpdateRequest.h>
#include "ns_InstallUtil.h"
#include "ns_SystemUpdateUtil.h"

namespace nn { namespace ns { namespace srv {
    Result SystemUpdateRequestList::RequestCheckLatest() NN_NOEXCEPT
    {
        m_IsCheckRequested = true;

        NN_RESULT_SUCCESS;
    }

    bool SystemUpdateRequestList::NeedsProcess(ProcessType type) NN_NOEXCEPT
    {
        if (type == ProcessType::All)
        {
            if (m_IsCheckRequested)
            {
                return true;
            }

            util::optional<SystemUpdateTaskIdInfo> idInfo;
            NN_ABORT_UNLESS_RESULT_SUCCESS(GetSystemUpdateTaskIdInfo(&idInfo));
            if ((idInfo && idInfo->info.progress.state != ncm::InstallProgressState::Downloaded)
                || (idInfo && idInfo->info.applyInfo == ncm::SystemUpdateTaskApplyInfo::RequireNoReboot))
            {
                return true;
            }
        }

        return false;
    }

    Result SystemUpdateRequestList::Prepare(ProcessType type) NN_NOEXCEPT
    {
        if (type == ProcessType::All)
        {
            NN_RESULT_DO(PrepareMetaRequestIfNeeded());
            NN_RESULT_DO(PrepareTaskRequestIfNeeded());
        }

        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateRequestList::PrepareMetaRequestIfNeeded() NN_NOEXCEPT
    {
        if (m_IsCheckRequested && !m_MetaRequest)
        {
            m_MetaRequest.emplace();
            NN_RESULT_TRY(nim::RequestSystemUpdateMeta(&(*m_MetaRequest)))
                NN_RESULT_CATCH_ALL
                {
                    m_MetaRequest = util::nullopt;
                    m_IsCheckRequested = false;
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
        }

        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateRequestList::PrepareTaskRequestIfNeeded() NN_NOEXCEPT
    {
        if (!m_TaskRequest)
        {
            util::optional<SystemUpdateTaskIdInfo> idInfo;
            NN_RESULT_DO(GetSystemUpdateTaskIdInfo(&idInfo));

            // 更新がダウンロード済みで再起動不要で適用できる場合は適用する
            if (idInfo && idInfo->info.progress.state == ncm::InstallProgressState::Downloaded && idInfo->info.applyInfo == ncm::SystemUpdateTaskApplyInfo::RequireNoReboot)
            {
                NN_RESULT_DO(m_SystemUpdateApplyManager->NotifyDownloadTaskCompleted(idInfo->id));
            }
            else if (idInfo && idInfo->info.progress.state != ncm::InstallProgressState::Downloaded)
            {
                m_TaskRequest.emplace();
                NN_RESULT_TRY(nim::RequestSystemUpdateTaskRun(&(*m_TaskRequest), idInfo->id))
                    NN_RESULT_CATCH_ALL
                    {
                        m_TaskRequest = util::nullopt;
                        NN_RESULT_RETHROW;
                    }
                NN_RESULT_END_TRY
            }
        }

        NN_RESULT_SUCCESS;
    }


    bool SystemUpdateRequestList::HasPreparedRequest() NN_NOEXCEPT
    {
        return m_MetaRequest || m_TaskRequest;
    }

    void SystemUpdateRequestList::WaitAlongWith(os::SystemEvent** events, int numEvents) NN_NOEXCEPT
    {
        MultiWaitSystemEvent multiWait;
        if (m_MetaRequest)
        {
            multiWait.Link(&(m_MetaRequest->GetEvent()));
        }
        if (m_TaskRequest)
        {
            multiWait.Link(&(m_TaskRequest->GetEvent()));
        }
        for (int i = 0; i < numEvents; ++i)
        {
            multiWait.Link(events[i]);
        }
        auto signaled = multiWait.WaitAny();
        NN_UNUSED(signaled);
    }

    bool SystemUpdateRequestList::HandleDone() NN_NOEXCEPT
    {
        bool allSuccess = true;
        if (m_MetaRequest && m_MetaRequest->TryWait())
        {
            auto result = HandleMetaRequestDone();
            if (result.IsFailure())
            {
                allSuccess = false;
                NN_DETAIL_NS_TRACE("[SystemUpdateRequestList] Failed to handle meta %08x\n", result.GetInnerValueForDebug());
            }
        }
        if (m_TaskRequest && m_TaskRequest->TryWait())
        {
            auto result = HandleTaskRequestDone();
            if (result.IsFailure())
            {
                allSuccess = false;
                NN_DETAIL_NS_TRACE("[SystemUpdateRequestList] Failed to handle task %08x\n", result.GetInnerValueForDebug());
            }
        }
        return allSuccess;
    }

    void SystemUpdateRequestList::Cleanup() NN_NOEXCEPT
    {
        m_MetaRequest = util::nullopt;
        m_TaskRequest = util::nullopt;
    }

    Result SystemUpdateRequestList::HandleMetaRequestDone() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[SystemUpdateRequestList] Cleanup meta request\n");
        NN_UTIL_SCOPE_EXIT{ m_MetaRequest = util::nullopt; };

        ncm::ContentMetaKey key;
        NN_RESULT_DO(GetLatestSystemUpdateMeta(&key, &(*m_MetaRequest)));

        bool needsUpdate;
        NN_RESULT_DO(NeedsUpdate(&needsUpdate, key, m_IsExFatDriverRequired));
        NN_DETAIL_NS_TRACE("[SystemUpdateRequestList] System update meta 0x%016llx version %d needs udpate %d\n", key.id, key.version, needsUpdate);
        if (needsUpdate)
        {
            // タスクを操作する前にリクエストを破棄する
            m_TaskRequest = util::nullopt;
            nim::SystemUpdateTaskId id;
            NN_RESULT_DO(CreateOrReuseSystemUpdateTask(&id, key, m_IsExFatDriverRequired));
        }

        m_IsCheckRequested = false;
        NN_RESULT_SUCCESS;
    }

    Result SystemUpdateRequestList::HandleTaskRequestDone() NN_NOEXCEPT
    {
        NN_DETAIL_NS_TRACE("[SystemUpdateRequestList] Cleanup task request\n");
        {
            NN_UTIL_SCOPE_EXIT{ m_TaskRequest = util::nullopt; };

            auto result = m_TaskRequest->Get();
            if (IsFatalSystemUpdate(result))
            {
                NN_DETAIL_NS_TRACE("[SystemUpdateRequestList] Destroy fatal system update task 0x%08x. Retry check\n", result.GetInnerValueForDebug());
                // タスクを操作する前にリクエストを破棄する
                m_TaskRequest = util::nullopt;
                NN_RESULT_DO(DestroyAllSystemUpdateTask());
                NN_RESULT_DO(RequestCheckLatest());
            }
            NN_RESULT_DO(result);
        }

        util::optional<SystemUpdateTaskIdInfo> info;
        NN_RESULT_DO(GetSystemUpdateTaskIdInfo(&info));
        if (info)
        {
            NN_RESULT_DO(m_SystemUpdateApplyManager->NotifyDownloadTaskCompleted(info->id));
        }

        NN_RESULT_SUCCESS;
    }

}}}
