﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/util/util_Optional.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Tick.h>
#include <nn/ns/ns_Result.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ncm/ncm_MaxCount.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ns/srv/ns_SystemUpdateInterfaceServer.h>
#include <nn/ns/srv/ns_ContentDeliveryManager.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/settings/system/settings_Boot.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_MountPrivate.h>
#include <nn/fs/fs_RegisteredUpdatePartition.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include "ns_AsyncContentDeliveryImpl.h"
#include "ns_AsyncImpl.h"
#include "ns_DebugUtil.h"
#include "ns_Config.h"
#include "ns_SystemUpdateUtil.h"
#include "ns_CleanupUtil.h"
#include "ns_TaskUtil.h"

namespace nn { namespace ns { namespace srv {

namespace
{
    typedef sf::ObjectFactory<sf::impl::StaticOneAllocationPolicy> StaticOneFactory;

    const char* UpdatePartitionMountName = "@upp";

    util::optional<nim::SystemUpdateTaskInfo> GetSystemUpdateTaskInfo() NN_NOEXCEPT
    {
        nim::SystemUpdateTaskId id;
        auto count = nim::ListSystemUpdateTask(&id, 1);
        if (count == 0)
        {
            return util::nullopt;
        }

        nim::SystemUpdateTaskInfo info;
        auto result = nim::GetSystemUpdateTaskInfo(&info, id);
        if (result.IsFailure() && result <= nim::ResultTaskNotFound())
        {
            return util::nullopt;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        return info;
    }

    bool HasDownloadedImpl()
    {
        auto info = GetSystemUpdateTaskInfo();
        return !info ? false : info->progress.state == ncm::InstallProgressState::Downloaded;
    }

    typedef kvdb::BoundedString<272> EulaDataFullPath;
    static const char EulaMountName[] = "euladata";

    Result MountDownloadedEulaData(const char* mountName) NN_NOEXCEPT
    {
        nim::SystemUpdateTaskId id;
        auto count = nim::ListSystemUpdateTask(&id, 1);
        NN_RESULT_THROW_UNLESS(count > 0, nim::ResultTaskNotFound());

        ncm::SystemDataId dataId = { NN_NS_SYSTEM_DATA_ID_OF_EULA };
        ncm::Path contentPath;
        NN_RESULT_TRY(nim::GetDownloadedSystemDataPath(&contentPath, dataId, id))
            NN_RESULT_CATCH(nim::ResultSystemDataNotFound)
            {
                NN_RESULT_THROW(ResultEulaSystemDataNotFound());
            }
        NN_RESULT_END_TRY

        NN_RESULT_DO(fs::MountContent(mountName, contentPath.string, dataId, fs::ContentType_Data));

        NN_RESULT_SUCCESS;
    }

    Result OpenEulaDataFile(fs::FileHandle* outValue, const char* mountName, const ns::detail::EulaDataPath& path) NN_NOEXCEPT
    {
        EulaDataFullPath fullPath;
        fullPath.AssignFormat("%s:/%s", mountName, path.str);
        NN_RESULT_TRY(fs::OpenFile(outValue, fullPath, fs::OpenMode_Read))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                NN_RESULT_THROW(ResultEulaDataPathNotFound());
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    Result GetEulaDataSize(size_t* outValue, const char* mountName, const ns::detail::EulaDataPath& path) NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(OpenEulaDataFile(&file, mountName, path));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, file));

        *outValue = static_cast<size_t>(fileSize);
        NN_RESULT_SUCCESS;
    }

    Result GetEulaData(size_t* outValue, void* buffer, size_t buffeSize, const char* mountName, const ns::detail::EulaDataPath& path) NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(OpenEulaDataFile(&file, mountName, path));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, file));
        auto dataSize = static_cast<size_t>(fileSize);
        NN_RESULT_THROW_UNLESS(dataSize <= buffeSize, ResultBufferNotEnough());

        NN_RESULT_DO(fs::ReadFile(file, 0, buffer, dataSize));

        *outValue = dataSize;
        NN_RESULT_SUCCESS;
    }

    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_Stack[4096];

    Result CleanupRedundant() NN_NOEXCEPT
    {
        static ncm::ContentMetaKey keyList[ncm::SystemMaxContentMetaCount];

        ncm::ContentMetaDatabase db;
        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInSystem));
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, ncm::StorageId::BuildInSystem));

        auto count = db.ListContentMeta(keyList, sizeof(keyList) / sizeof(keyList[0]));

        bool needsCommit = false;
        ncm::ContentMetaKey previousKey = {};
        for (int i = 0; i < count.listed; i++)
        {
            auto& key = keyList[i];
            if (previousKey.id == key.id)
            {
                NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Found redundant 0x%016llx version %u. Delete it.\n", previousKey.id, previousKey.version);
                ncm::ContentManagerAccessor accessor(&db, &storage);
                NN_RESULT_DO(accessor.DeleteRedundant(previousKey, &key));
                needsCommit = true;
            }

            previousKey = keyList[i];
        }

        NN_RESULT_DO(db.Commit());
        NN_RESULT_SUCCESS;
    }

    util::optional<nim::LocalCommunicationReceiveSystemUpdateTaskInfo> GetLocalCommunicationReceiveSystemUpdateTaskInfo() NN_NOEXCEPT
    {
        nim::LocalCommunicationReceiveSystemUpdateTaskId id;
        auto count = nim::ListLocalCommunicationReceiveSystemUpdateTask(&id, 1);
        if (count == 0)
        {
            return util::nullopt;
        }

        nim::LocalCommunicationReceiveSystemUpdateTaskInfo info;
        auto result = nim::GetLocalCommunicationReceiveSystemUpdateTaskInfo(&info, id);
        if (result.IsFailure() && result <= nim::ResultTaskNotFound())
        {
            return util::nullopt;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        return info;
    }

    bool HasReceivedImpl()
    {
        auto info = GetLocalCommunicationReceiveSystemUpdateTaskInfo();
        return !info ? false : info->progress.state == ncm::InstallProgressState::Downloaded;
    }

    Result MountReceivedEulaData(const char* mountName) NN_NOEXCEPT
    {
        nim::LocalCommunicationReceiveSystemUpdateTaskId id;
        auto count = nim::ListLocalCommunicationReceiveSystemUpdateTask(&id, 1);
        NN_RESULT_THROW_UNLESS(count > 0, nim::ResultTaskNotFound());

        ncm::SystemDataId dataId = { NN_NS_SYSTEM_DATA_ID_OF_EULA };
        ncm::Path contentPath;
        NN_RESULT_TRY(nim::GetReceivedSystemDataPath(&contentPath, dataId, id))
            NN_RESULT_CATCH(nim::ResultSystemDataNotFound)
            {
                NN_RESULT_THROW(ResultEulaSystemDataNotFound());
            }
        NN_RESULT_END_TRY

        NN_RESULT_DO(fs::MountContent(mountName, contentPath.string, dataId, fs::ContentType_Data));

        NN_RESULT_SUCCESS;
    }

    util::optional<nim::LocalCommunicationSendSystemUpdateTaskInfo> GetLocalCommunicationSendSystemUpdateTaskInfo() NN_NOEXCEPT
    {
        nim::LocalCommunicationSendSystemUpdateTaskId id;
        auto count = nim::ListLocalCommunicationSendSystemUpdateTask(&id, 1);
        if (count == 0)
        {
            return util::nullopt;
        }

        nim::LocalCommunicationSendSystemUpdateTaskInfo info;
        auto result = nim::GetLocalCommunicationSendSystemUpdateTaskInfo(&info, id);
        if (result.IsFailure() && result <= nim::ResultTaskNotFound())
        {
            return util::nullopt;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        return info;
    }
}

SystemUpdateControlServer::~SystemUpdateControlServer() NN_NOEXCEPT
{
    if (m_IsCardUpdateSetup)
    {
        fs::Unmount(UpdatePartitionMountName);
        m_CardUpdateTask->Cleanup();
        m_CardUpdateTransferMemory->Unmap();
    }

    m_Server->OnCloseSystemUpdateControl();
}

Result SystemUpdateControlServer::HasDownloaded(sf::Out<bool> outValue) NN_NOEXCEPT
{
    *outValue = HasDownloadedImpl();
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::RequestCheckLatestUpdate(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncValue>> outAsync) NN_NOEXCEPT
{
    NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Request check latest update\n");

    auto emplacedRef = StaticOneFactory::CreateSharedEmplaced <ns::detail::IAsyncValue, AsyncLatestSystemUpdateImpl>(m_ManagedStop.AddReference(), m_Server->IsExFatDriverRequired());
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

    NN_RESULT_DO(emplacedRef.GetImpl().Run());
    *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
    *outAsync = emplacedRef;

    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::RequestDownloadLatestUpdate(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync) NN_NOEXCEPT
{
    NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Request download latest update\n");

    NN_RESULT_THROW_UNLESS(!m_IsCardUpdateSetup, ResultCardUpdateAlreadySetup());

    auto emplacedRef = StaticOneFactory::CreateSharedEmplaced <ns::detail::IAsyncResult, AsyncDownloadLatestUpdateImpl>(m_ManagedStop.AddReference(), m_Server->IsExFatDriverRequired());
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

    NN_RESULT_DO(emplacedRef.GetImpl().Run());
    *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
    *outAsync = emplacedRef;

    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::GetDownloadProgress(sf::Out<SystemUpdateProgress> outValue) NN_NOEXCEPT
{
    auto info = GetSystemUpdateTaskInfo();
    if (!info)
    {
        SystemUpdateProgress progress = {};
        *outValue = progress;
        NN_RESULT_SUCCESS;
    }

    SystemUpdateProgress progress = { info->progress.installedSize, info->progress.totalSize };
    *outValue = progress;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::ApplyDownloadedUpdate() NN_NOEXCEPT
{
    auto hasDownloaded = HasDownloadedImpl();
    NN_RESULT_THROW_UNLESS(hasDownloaded, ResultInvalidSystemUpdateControlState());

    nim::SystemUpdateTaskId id;
    auto count = nim::ListSystemUpdateTask(&id, 1);
    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidSystemUpdateControlState());

    bool includesExFatDriver;
    NN_RESULT_DO(nim::IsExFatDriverIncluded(&includesExFatDriver, id));
    if (includesExFatDriver)
    {
        NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Notify exFAT driver downloaded\n");
        NN_RESULT_DO(m_Server->NotifyExFatDriverDownloaded());
    }

    NN_RESULT_DO(m_Server->ApplySystemUpdateTask(id));

    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::InitializeCardUpdateTask(sf::NativeHandle&& transferMemoryHandle, std::uint64_t transferMemorySize) NN_NOEXCEPT
{
    bool isSuccess = false;

    auto bufferSize = static_cast<size_t>(transferMemorySize);
    m_CardUpdateTransferMemory.emplace(bufferSize, transferMemoryHandle.GetOsHandle(), true);
    void* buffer;
    NN_RESULT_DO(m_CardUpdateTransferMemory->Map(&buffer, os::MemoryPermission_None));
    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            m_CardUpdateTransferMemory->Unmap();
            m_CardUpdateTransferMemory = util::nullopt;
        };
    };
    transferMemoryHandle.Detach();

    m_CardUpdateTask.emplace();

    char rootPath[32];
    util::SNPrintf(rootPath, sizeof(rootPath), "%s:/", UpdatePartitionMountName);
    char contextPath[32];
    util::SNPrintf(contextPath, sizeof(contextPath), "%s:/cup.ctx", SystemUpdateDataStoreMountName);
    NN_RESULT_DO(m_CardUpdateTask->Initialize(rootPath, contextPath, buffer, bufferSize, m_Server->IsExFatDriverDownloadedAtLeastOnce()));

    isSuccess = true;

    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::SetupCardUpdate(sf::NativeHandle&& transferMemoryHandle, std::uint64_t transferMemorySize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_IsCardUpdateSetup, ResultCardUpdateAlreadySetup());

    fs::GameCardHandle handle;
    NN_RESULT_DO(fs::GetGameCardHandle(&handle));
    fs::GameCardUpdatePartitionInfo info;
    NN_RESULT_DO(fs::GetGameCardUpdatePartitionInfo(&info, handle));

    // CUP 判定は exFAT ドライバを考慮しない
    bool needsUpdate;
    NN_RESULT_DO(NeedsUpdate(&needsUpdate, ncm::ContentMetaKey::Make(info.id, info.version, ncm::ContentMetaType::SystemUpdate), false));
    NN_RESULT_THROW_UNLESS(needsUpdate, ResultAlreadyUpToDate());

    nim::SystemUpdateTaskId id;
    auto count = nim::ListSystemUpdateTask(&id, 1);
    if (count > 0)
    {
        NN_RESULT_DO(nim::DestroySystemUpdateTask(id));
    }

    NN_RESULT_DO(fs::MountGameCardPartition(UpdatePartitionMountName, handle, fs::GameCardPartition::Update));
    NN_UTIL_SCOPE_EXIT
    {
        if (!m_IsCardUpdateSetup)
        {
            fs::Unmount(UpdatePartitionMountName);
        };
    };

    NN_RESULT_DO(InitializeCardUpdateTask(std::move(transferMemoryHandle), transferMemorySize));

    m_IsCardUpdateSetup = true;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::SetupCardUpdateViaSystemUpdater(sf::NativeHandle&& transferMemoryHandle, std::uint64_t transferMemorySize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_IsCardUpdateSetup, ResultCardUpdateAlreadySetup());

    // TODO: バージョンチェック

    nim::SystemUpdateTaskId id;
    auto count = nim::ListSystemUpdateTask(&id, 1);
    if (count > 0)
    {
        NN_RESULT_DO(nim::DestroySystemUpdateTask(id));
    }

    NN_RESULT_DO(fs::MountRegisteredUpdatePartition(UpdatePartitionMountName));
    NN_UTIL_SCOPE_EXIT
    {
        if (!m_IsCardUpdateSetup)
        {
            fs::Unmount(UpdatePartitionMountName);
        };
    };

    NN_RESULT_DO(InitializeCardUpdateTask(std::move(transferMemoryHandle), transferMemorySize));

    m_IsCardUpdateSetup = true;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::RequestPrepareCardUpdate(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsCardUpdateSetup, ResultCardUpdateNotSetup());
    NN_RESULT_THROW_UNLESS(!m_IsAlreadyCardUpdateRequested, ResultPrepareCardUpdateAlreadyRequested());

    auto emplacedRef = StaticOneFactory::CreateSharedEmplaced <ns::detail::IAsyncResult, AsyncPrepareCardUpdateImpl>(&(*m_CardUpdateTask), m_ManagedStop.AddReference());
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

    NN_RESULT_DO(emplacedRef.GetImpl().Run());
    *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
    *outAsync = emplacedRef;

    m_IsAlreadyCardUpdateRequested = true;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::GetPrepareCardUpdateProgress(sf::Out<SystemUpdateProgress> outValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsCardUpdateSetup, ResultCardUpdateNotSetup());

    auto installProgress = m_CardUpdateTask->GetProgress();
    SystemUpdateProgress progress = { installProgress.installedSize, installProgress.totalSize };
    *outValue = progress;

    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::HasPreparedCardUpdate(sf::Out<bool> outValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsCardUpdateSetup, ResultCardUpdateNotSetup());

    *outValue = m_CardUpdateTask->GetProgress().state == ncm::InstallProgressState::Downloaded;

    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::ApplyCardUpdate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsCardUpdateSetup, ResultCardUpdateNotSetup());
    NN_RESULT_THROW_UNLESS(m_CardUpdateTask->GetProgress().state == ncm::InstallProgressState::Downloaded, ResultCardUpdateNotPrepared());
    NN_RESULT_DO(m_Server->ApplyPackageSystemUpdateTask(&*m_CardUpdateTask));

    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::GetDownloadedEulaDataSize(sf::Out<std::uint64_t> outValue, const ns::detail::EulaDataPath& path) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(HasDownloadedImpl(), ResultSystemUpdateNotDownloaded());

    NN_RESULT_DO(MountDownloadedEulaData(EulaMountName));
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(EulaMountName); };

    size_t dataSize;
    NN_RESULT_DO(GetEulaDataSize(&dataSize, EulaMountName, path));

    *outValue = dataSize;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::GetDownloadedEulaData(sf::Out<std::uint64_t> outValue, const sf::OutBuffer& buffer, const ns::detail::EulaDataPath& path) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(HasDownloadedImpl(), ResultSystemUpdateNotDownloaded());

    NN_RESULT_DO(MountDownloadedEulaData(EulaMountName));
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(EulaMountName); };

    size_t dataSize;
    NN_RESULT_DO(GetEulaData(&dataSize, buffer.GetPointerUnsafe(), buffer.GetSize(), EulaMountName, path));

    *outValue = dataSize;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::GetPreparedCardUpdateEulaDataSize(sf::Out<std::uint64_t> outValue, const ns::detail::EulaDataPath& path) NN_NOEXCEPT
{
    NN_RESULT_DO(MountPreparedCardUpdateEulaData(EulaMountName));
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(EulaMountName); };

    size_t dataSize;
    NN_RESULT_DO(GetEulaDataSize(&dataSize, EulaMountName, path));

    *outValue = dataSize;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::GetPreparedCardUpdateEulaData(sf::Out<std::uint64_t> outValue, const sf::OutBuffer& buffer, const ns::detail::EulaDataPath& path) NN_NOEXCEPT
{
    NN_RESULT_DO(MountPreparedCardUpdateEulaData(EulaMountName));
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(EulaMountName); };

    size_t dataSize;
    NN_RESULT_DO(GetEulaData(&dataSize, buffer.GetPointerUnsafe(), buffer.GetSize(), EulaMountName, path));

    *outValue = dataSize;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::MountPreparedCardUpdateEulaData(const char* mountName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsCardUpdateSetup, ResultCardUpdateNotSetup());
    NN_RESULT_THROW_UNLESS(m_CardUpdateTask->GetProgress().state == ncm::InstallProgressState::Downloaded, ResultCardUpdateNotPrepared());

    ncm::Path placeHolderPath;
    NN_RESULT_DO(m_CardUpdateTask->GetPreparedPlaceHolderPath(&placeHolderPath, NN_NS_SYSTEM_DATA_ID_OF_EULA, ncm::ContentMetaType::SystemData, ncm::ContentType::Data));
    ncm::SystemDataId dataId = { NN_NS_SYSTEM_DATA_ID_OF_EULA };
    NN_RESULT_DO(fs::MountContent(mountName, placeHolderPath.string, dataId, fs::ContentType_Data));

    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::HasReceived(sf::Out<bool> outValue) NN_NOEXCEPT
{
    *outValue = HasReceivedImpl();
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::SetupToReceiveSystemUpdate() NN_NOEXCEPT
{
    nim::SystemUpdateTaskId id;
    auto count = nim::ListSystemUpdateTask(&id, 1);
    if (count > 0)
    {
        NN_RESULT_DO(nim::DestroySystemUpdateTask(id));
    }

    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::RequestReceiveSystemUpdate(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, uint32_t ipv4, uint16_t port, const SystemDeliveryInfo& info) NN_NOEXCEPT
{
    NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Request receive system update\n");

    NN_RESULT_THROW_UNLESS(!m_IsCardUpdateSetup, ResultCardUpdateAlreadySetup());

    ncm::ContentMetaKey key;
    NN_RESULT_DO(ContentDeliveryManager::GetSystemUpdateMeta(&key, info));

    auto emplacedRef = StaticOneFactory::CreateSharedEmplaced <ns::detail::IAsyncResult, AsyncReceiveSystemUpdateImpl>(m_ManagedStop.AddReference(), m_Server->IsExFatDriverDownloadedAtLeastOnce(), ipv4, port, key);
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

    NN_RESULT_DO(emplacedRef.GetImpl().Run());
    *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
    *outAsync = emplacedRef;

    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::GetReceiveProgress(sf::Out<SystemUpdateProgress> outValue) NN_NOEXCEPT
{
    auto info = GetLocalCommunicationReceiveSystemUpdateTaskInfo();
    if (!info)
    {
        SystemUpdateProgress progress = {};
        *outValue = progress;
        NN_RESULT_SUCCESS;
    }

    SystemUpdateProgress progress = { info->progress.installedSize, info->progress.totalSize };
    *outValue = progress;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::ApplyReceivedUpdate() NN_NOEXCEPT
{
    auto hasReceived = HasReceivedImpl();
    NN_RESULT_THROW_UNLESS(hasReceived, ResultInvalidSystemUpdateControlState());

    nim::LocalCommunicationReceiveSystemUpdateTaskId id;
    auto count = nim::ListLocalCommunicationReceiveSystemUpdateTask(&id, 1);
    NN_RESULT_THROW_UNLESS(count > 0, ResultInvalidSystemUpdateControlState());

    NN_RESULT_DO(m_Server->ApplyReceivedSystemUpdateTask(id));

    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::GetReceivedEulaDataSize(sf::Out<std::uint64_t> outValue, const ns::detail::EulaDataPath& path) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(HasReceivedImpl(), ResultSystemUpdateNotDownloaded());

    NN_RESULT_DO(MountReceivedEulaData(EulaMountName));
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(EulaMountName); };

    size_t dataSize;
    NN_RESULT_DO(GetEulaDataSize(&dataSize, EulaMountName, path));

    *outValue = dataSize;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateControlServer::GetReceivedEulaData(sf::Out<std::uint64_t> outValue, const sf::OutBuffer& buffer, const ns::detail::EulaDataPath& path) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(HasReceivedImpl(), ResultSystemUpdateNotDownloaded());

    NN_RESULT_DO(MountReceivedEulaData(EulaMountName));
    NN_UTIL_SCOPE_EXIT{ fs::Unmount(EulaMountName); };

    size_t dataSize;
    NN_RESULT_DO(GetEulaData(&dataSize, buffer.GetPointerUnsafe(), buffer.GetSize(), EulaMountName, path));

    *outValue = dataSize;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::Initialize(SystemReportManager* systemReportManager) NN_NOEXCEPT
{
    NN_RESULT_DO(CleanupRedundant());
    SaveDataInfo saveInfo = { SystemUpdateDataStoreMountName, SystemUpdateSaveDataId, SystemUpdateSaveDataSize, SystemUpdateSaveDataJournalSize, SystemUpdateSaveDataFlags };
    NN_RESULT_DO(m_DataStore.Initialize(saveInfo));
    NN_RESULT_DO(m_ExFatDriverManager.Initialize());

    {
        // SIGLO-71803
        // タスクが無い場合にクリーンナップ処理を行い余計なデータを削除するため、
        // 強制電源断と同じ処理をさせる
        nim::SystemUpdateTaskId id;
        auto count = nim::ListSystemUpdateTask(&id, 1);
        if (m_DataStore.DetectsForceShutdown() || count == 0)
        {
            NN_RESULT_DO(CleanupOnForceShutdown());
        }
    }

    m_RequestList.SetExFatDriverRequired(m_ExFatDriverManager.IsDriverRequired());

    m_RequestServer.Initialize(GetBgnupRetrySpan());
    m_SystemUpdateApplyManager.Initialize(systemReportManager, &m_ExFatDriverManager);

    //  修理中ならば RequestServer を止める
    if (settings::system::IsInRepairProcess())
    {
        m_Stopper = m_RequestServer.Stop();
    }

    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::CleanupOnForceShutdown() NN_NOEXCEPT
{
    NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Cleanup due to force shutdown\n");

    auto begin = os::GetSystemTick().ToTimeSpan();
    NN_UNUSED(begin);

    {
        nim::SystemUpdateTaskId id;
        auto count = nim::ListSystemUpdateTask(&id, 1);
        if (count > 0)
        {
            NN_RESULT_DO(nim::DestroySystemUpdateTask(id));
        }
    }

    ncm::ContentStorage storage;
    NN_RESULT_DO(ncm::OpenContentStorage(&storage, ncm::StorageId::BuildInSystem));
    NN_RESULT_DO(storage.CleanupAllPlaceHolder());

    NN_RESULT_DO(CleanupOrphanContents(ncm::StorageId::BuildInSystem));

    NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Took %lld ms to cleanup force shutdown\n", (os::GetSystemTick().ToTimeSpan() - begin).GetMilliSeconds());

    NN_RESULT_SUCCESS;
}

SystemUpdateInterfaceServer::SystemUpdateInterfaceServer() NN_NOEXCEPT :
    m_IsControlServerOccupied(false),
    m_RequestList(&m_SystemUpdateApplyManager),
    m_RequestServer(&m_RequestList, g_Stack, sizeof(g_Stack))
    {}

SystemUpdateInterfaceServer::~SystemUpdateInterfaceServer() NN_NOEXCEPT
{
    m_RequestServer.Finalize();
}

Result SystemUpdateInterfaceServer::OpenSystemUpdateControl(sf::Out<sf::SharedPointer<ns::detail::ISystemUpdateControl>> outValue) NN_NOEXCEPT
{
    NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Open system update control\n");

    std::lock_guard<os::Mutex> guard(m_OccupationMutex);

    if (m_IsControlServerOccupied)
    {
        NN_RESULT_THROW(ResultAlreadyOccupied());
    }

    *outValue = StaticOneFactory::CreateSharedEmplaced<ns::detail::ISystemUpdateControl, srv::SystemUpdateControlServer>(this, m_RequestServer.Stop());

    m_IsControlServerOccupied = true;

    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::GetBackgroundNetworkUpdateState(sf::Out<BackgroundNetworkUpdateState> outValue) NN_NOEXCEPT
{
    auto info = GetSystemUpdateTaskInfo();
    if (!info)
    {
        *outValue = BackgroundNetworkUpdateState::None;
        NN_RESULT_SUCCESS;
    }

    *outValue = info->progress.state == ncm::InstallProgressState::Downloaded ? BackgroundNetworkUpdateState::Ready : BackgroundNetworkUpdateState::InProgress;
    NN_RESULT_SUCCESS;
}

void SystemUpdateInterfaceServer::OnCloseSystemUpdateControl() NN_NOEXCEPT
{
    NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Close system update control\n");

    std::lock_guard<os::Mutex> guard(m_OccupationMutex);

    m_IsControlServerOccupied = false;
}

Result SystemUpdateInterfaceServer::NotifyExFatDriverRequired() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_OccupationMutex);

    if (m_IsControlServerOccupied)
    {
        NN_RESULT_THROW(ResultAlreadyOccupied());
    }

    {
        auto managedStop = m_RequestServer.Stop();

        nim::SystemUpdateTaskId id;
        auto count = nim::ListSystemUpdateTask(&id, 1);
        if (count > 0)
        {
            NN_RESULT_DO(nim::DestroySystemUpdateTask(id));
        }

        NN_RESULT_DO(m_ExFatDriverManager.NotifyDriverRequired());

        m_RequestList.SetExFatDriverRequired(m_ExFatDriverManager.IsDriverRequired());
        NN_RESULT_DO(m_RequestList.RequestCheckLatest());
    }

    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::NotifyBackgroundNetworkUpdate(const nn::ncm::ContentMetaKey& systemUpdateMetaKey) NN_NOEXCEPT
{
    bool needsUpdate;
    NN_RESULT_DO(NeedsUpdate(
        &needsUpdate, systemUpdateMetaKey, m_ExFatDriverManager.IsDriverRequired()));

    NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Notified bgnup.\n");
    if (needsUpdate)
    {
        NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Request bgnup.\n");
        NN_RESULT_DO(RequestBackgroundNetworkUpdate());
    }
    else
    {
        NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Skip bgnup.\n");
    }

    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::NotifyExFatDriverDownloaded() NN_NOEXCEPT
{
    return m_ExFatDriverManager.NotifyDriverDownloaded();
}

Result SystemUpdateInterfaceServer::ClearExFatDriverStatusForDebug() NN_NOEXCEPT
{
    return m_ExFatDriverManager.Clear();
}

Result SystemUpdateInterfaceServer::RequestBackgroundNetworkUpdate() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_OccupationMutex);

    if (m_IsControlServerOccupied)
    {
        NN_RESULT_THROW(ResultAlreadyOccupied());
    }

    {
        auto managedStop = m_RequestServer.Stop();

        NN_RESULT_DO(m_RequestList.RequestCheckLatest());
    }

    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::GetSystemUpdateNotificationEventForContentDelivery(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
{
    *outValue = sf::NativeHandle(m_EventForContentDelivery.GetReadableHandle(), false);
    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::NotifySystemUpdateForContentDelivery() NN_NOEXCEPT
{
    m_EventForContentDelivery.Signal();
    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::DestroySystemUpdateTask() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_OccupationMutex);
    NN_RESULT_THROW_UNLESS(!m_IsControlServerOccupied, ResultAlreadyOccupied());

    {
        auto managedStop = m_RequestServer.Stop();
        nim::SystemUpdateTaskId id;
        auto count = nim::ListSystemUpdateTask(&id, 1);
        if (count > 0)
        {
            NN_RESULT_DO(nim::DestroySystemUpdateTask(id));
        }
    }
    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::DestroyIncompleteSystemUpdateTask() NN_NOEXCEPT
{
    nim::SystemUpdateTaskId id;
    auto count = nim::ListSystemUpdateTask(&id, 1);
    if (count > 0)
    {
        nim::SystemUpdateTaskInfo info;
        NN_RESULT_DO(nim::GetSystemUpdateTaskInfo(&info, id));
        if (info.progress.state != ncm::InstallProgressState::Downloaded)
        {
            NN_RESULT_DO(nim::DestroySystemUpdateTask(id));
        }
    }
    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::PrepareShutdown() NN_NOEXCEPT
{
    // 誰かによって control が開かれている場合には、タスクを止めることはできない
    // そのため、control が開かれた状態ではシャットダウン処理は何も行わない
    // 何も行わないでおくと、強制電源断とみなされて次回起動時に途中のデータは破棄される
    // DestroyIncompleteSystemUpdateTask に失敗したときも強制電源断扱いにする
    if (!m_IsControlServerOccupied)
    {
        m_Stopper = m_RequestServer.Stop();

        auto result = DestroyIncompleteSystemUpdateTask();
        if (result.IsSuccess())
        {
            NN_RESULT_DO(nim::PrepareShutdownForSystemUpdate());
            NN_RESULT_DO(m_DataStore.PrepareShutdown());
        }
    }
    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::RequestSendSystemUpdate(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, uint32_t ipv4, uint16_t port, const SystemDeliveryInfo& info) NN_NOEXCEPT
{
    NN_DETAIL_NS_TRACE("[SystemUpdateInterfaceServer] Request receive system update\n");

    std::lock_guard<os::Mutex> guard(m_OccupationMutex);

    ncm::ContentMetaKey key;
    NN_RESULT_DO(ContentDeliveryManager::GetSystemUpdateMeta(&key, info));

    auto emplacedRef = StaticOneFactory::CreateSharedEmplaced <ns::detail::IAsyncResult, AsyncSendSystemUpdateImpl>(m_RequestServer.Stop(), ipv4, port, key);
    NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

    NN_RESULT_DO(emplacedRef.GetImpl().Run());
    *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
    *outAsync = emplacedRef;

    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::GetSendSystemUpdateProgress(sf::Out<SystemUpdateProgress> outValue) NN_NOEXCEPT
{
    auto info = GetLocalCommunicationSendSystemUpdateTaskInfo();
    if (!info)
    {
        SystemUpdateProgress progress = {};
        *outValue = progress;
        NN_RESULT_SUCCESS;
    }

    SystemUpdateProgress progress = { info->progress.sentSize, info->progress.totalSize };
    *outValue = progress;
    NN_RESULT_SUCCESS;
}

Result SystemUpdateInterfaceServer::ApplySystemUpdateTask(nim::SystemUpdateTaskId id) NN_NOEXCEPT
{
    return m_SystemUpdateApplyManager.ApplyTask(id, true, false);
}

Result SystemUpdateInterfaceServer::ApplyReceivedSystemUpdateTask(nim::LocalCommunicationReceiveSystemUpdateTaskId id) NN_NOEXCEPT
{
    return m_SystemUpdateApplyManager.ApplyReceivedTask(id);
}

Result SystemUpdateInterfaceServer::ApplyPackageSystemUpdateTask(ncm::PackageSystemUpdateTask* task) NN_NOEXCEPT
{
    return m_SystemUpdateApplyManager.ApplyPackageTask(task);
}

}}}
