﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include "ns_SystemSaveDataUtil.h"

namespace nn { namespace ns { namespace srv {
    typedef kvdb::BoundedString<32> Path;

    Path MakeSignaturePath(const char* mountName) NN_NOEXCEPT
    {
        Path path(mountName);
        return path.Append(":/signature");
    }
    SystemSaveDataSignature::SystemSaveDataSignature(const char* signature, uint32_t version) NN_NOEXCEPT : m_Data()
    {
        util::Strlcpy(m_Data.signature, signature, static_cast<int>(sizeof(m_Data)));
        m_Data.version = version;
    }

    Result SystemSaveDataSignature::IsValidOn(bool* outValue, const char* mountName) NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_TRY(fs::OpenFile(&file, MakeSignaturePath(mountName), fs::OpenMode_Read))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_END_TRY
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        Data data;
        NN_RESULT_DO(fs::ReadFile(file, 0, &data, sizeof(data)));

        *outValue = std::memcmp(&m_Data, &data, sizeof(m_Data)) == 0;
        NN_RESULT_SUCCESS;
    }

    Result SystemSaveDataSignature::WriteTo(const char* mountName) NN_NOEXCEPT
    {
        auto sigPath = MakeSignaturePath(mountName);
        fs::DeleteFile(sigPath);
        NN_RESULT_DO(fs::CreateFile(sigPath, sizeof(m_Data)));

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, sigPath, fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        NN_RESULT_DO(fs::WriteFile(file, 0, &m_Data, sizeof(m_Data), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }
}}}
