﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/npns.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/util/util_Span.h>
#include <nn/util/util_StringUtil.h>

#include <nn/ns/srv/ns_PushNotificationDisptcher.h>
#include "ns_PushNotificationReader.h"
#include "ns_JsonUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        const char NotificationTypeNup[] = "Nup";
        const char NotificationTypeDtl[] = "Dtl";
        const char NotificationTypeVersionList[] = "VersionList";
        const char NotificationTypeETicketAvailable[] = "ETicketAvailable";
        const char NotificationTypeSendRightsUsageStatusRequest[] = "SendRightsUsageStatusRequest";
        const char NotificationTypeSyncELicenseRequest [] = "SyncELicenseRequest";

        Result ParseWaitingPolicyCore(NotificationInfo::WaitingPolicy* out, nne::rapidjson::Document* document, const char* rootName) NN_NOEXCEPT
        {
            nne::rapidjson::Document::ValueType* policy;
            auto policyParseResult = JsonParser::FindObject(&policy, document, rootName, nne::rapidjson::kObjectType);
            if (policyParseResult.IsFailure())
            {
                NN_PN_TRACE("Failed to parse waiting policy: %08x\n", policyParseResult.GetInnerValueForDebug());
                out->type = NotificationInfo::WaitingPolicy::WaitingPolicyType::RandomWaiting;
                out->waitingLimit = 0;
                NN_RESULT_SUCCESS;
            }

            const char* typeString;
            size_t typeStringLength;
            auto waitingPolicyParseResult = JsonParser::FindString(&typeString, &typeStringLength, policy, "type");
            // パースに失敗したら待ち時間なしのランダムウェイト
            if (waitingPolicyParseResult.IsFailure())
            {
                NN_PN_TRACE("Failed to parse waiting policy: %08x\n", waitingPolicyParseResult.GetInnerValueForDebug());
                out->type = NotificationInfo::WaitingPolicy::WaitingPolicyType::RandomWaiting;
                out->waitingLimit = 0;
                NN_RESULT_SUCCESS;
            }

            const char RandomPolicy[] = "random";
            if (typeStringLength == sizeof(RandomPolicy) - 1 && util::Strncmp(typeString, RandomPolicy, static_cast<int>(typeStringLength)) == 0)
            {
                out->type = NotificationInfo::WaitingPolicy::WaitingPolicyType::RandomWaiting;
            }
            else
            {
                NN_PN_TRACE("Unexpected waiting policy type specified: \"%s\"\n", typeString);
                NN_PN_TRACE("Set waiting policy to \"random\"\n");
                out->type = NotificationInfo::WaitingPolicy::WaitingPolicyType::RandomWaiting;
            }

            uint32_t waitingLimit;
            NN_RESULT_DO(JsonParser::FindUint32(&waitingLimit, policy, "limit"));
            out->waitingLimit = waitingLimit;

            NN_RESULT_SUCCESS;
        }

        Result ParseWaitingPolicy(NotificationInfo* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            return ParseWaitingPolicyCore(&out->waitingPolicy, document, "wait_policy");
        }

        Result ParseVersionListWaitingPolicy(NotificationInfo* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            return ParseWaitingPolicyCore(&out->waitingPolicy, document, "wait_policy");
        }

        Result ParseAutoUpdateWaitingPolicy(NotificationInfo* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            return ParseWaitingPolicyCore(&out->versionListInfo.autoUpdateWaitingPolicy, document, "wait_policy");
        }

        Result ParseETag(nim::ETag* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            const char* pETagString;
            size_t eTagSize;
            NN_RESULT_DO(JsonParser::FindString(&pETagString, &eTagSize, document, "e_tag"));
            NN_RESULT_THROW_UNLESS(eTagSize <= sizeof(out->data), ns::ResultTooLargePushNotificationETag());

            std::memcpy(out->data, pETagString, std::min(eTagSize, sizeof(out->data)));
            NN_RESULT_SUCCESS;
        }

        Result ParseTitleInfo(NotificationInfo* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            uint32_t titleVersion;
            NN_RESULT_DO(JsonParser::FindUint32(&titleVersion, document, "title_version"));

            const char* pProgramIdString;
            NN_RESULT_DO(JsonParser::FindString(&pProgramIdString, nullptr, document, "title_id"));
            nn::Bit64 programId;
            JsonParser::MakeBit64FromString(&programId, pProgramIdString);

            uint32_t requiredTitleVersion;
            NN_RESULT_DO(JsonParser::FindUint32(&requiredTitleVersion, document, "required_title_version"));

            const char* pRequiredProgramIdString;
            NN_RESULT_DO(JsonParser::FindString(&pRequiredProgramIdString, nullptr, document, "required_title_id"));
            nn::Bit64 requiredProgramId;
            JsonParser::MakeBit64FromString(&requiredProgramId, pRequiredProgramIdString);

            out->nupInfo.titleId = programId;
            out->nupInfo.titleVersion = titleVersion;
            out->nupInfo.requiredTitleId = requiredProgramId;
            out->nupInfo.requiredVersion = requiredTitleVersion;

            NN_RESULT_SUCCESS;
        }

        Result ParseNupInfo(NotificationInfo* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            NN_RESULT_DO(ParseTitleInfo(out, document));
            NN_RESULT_DO(ParseWaitingPolicy(out, document));

            // TODO: サーバ側の変更が入ったら #if 0 の方のコードに置き換える
            {
#if 0
                NN_RESULT_DO(ParseETag(&out->eTag, document));
#else
                auto result = ParseETag(&out->eTag, document);
                if (result.IsFailure())
                {
                    out->eTag = nim::ETag::MakeEmpty();
                }
#endif
            }

            out->type = NotificationInfo::NotificationType::SystemUpdate;

            NN_RESULT_SUCCESS;
        }

        Result ParseDtlInfo(NotificationInfo* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            std::memset(out, 0, sizeof(*out));
            out->type = NotificationInfo::NotificationType::DownloadTaskList;

            NN_RESULT_DO(ParseETag(&out->eTag, document));
            NN_RESULT_DO(ParseWaitingPolicy(out, document));
            NN_RESULT_SUCCESS;
        }

        Result ParseVersionListInfo(NotificationInfo* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            std::memset(out, 0, sizeof(*out));
            out->type = NotificationInfo::NotificationType::VersionList;

            NN_RESULT_DO(ParseETag(&out->eTag, document));
            NN_RESULT_DO(ParseAutoUpdateWaitingPolicy(out, document));
            NN_RESULT_DO(ParseVersionListWaitingPolicy(out, document));
            NN_RESULT_SUCCESS;
        }

        Result ParseETicketAvailableInfo(NotificationInfo* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            std::memset(out, 0, sizeof(*out));
            out->type = NotificationInfo::NotificationType::ETicketAvailable;

            nne::rapidjson::Document::ValueType* type;
            NN_RESULT_DO(JsonParser::FindArray(&type, document, "title_ids"));
            NN_RESULT_THROW_UNLESS(!type->Empty(), ns::ResultCommunicationFailed());

            auto itr = type->Begin();
            NN_RESULT_THROW_UNLESS(itr->IsString(), ns::ResultCommunicationFailed());

            const char* pTitleIdString = itr->GetString();
            out->eTicketAvailableInfo.titleId = strtoull(pTitleIdString, nullptr, 16);

            NN_RESULT_SUCCESS;
        }

        Result ParseSendRightsUsageStatusRequest(NotificationInfo* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            NN_UNUSED(document);
            std::memset(out, 0, sizeof(*out));
            out->type = NotificationInfo::NotificationType::SendRightsUsageStatusRequest;

            NN_RESULT_SUCCESS;
        }

        Result ParseSyncELicenseRequest(NotificationInfo* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            std::memset(out, 0, sizeof(*out));
            out->type = NotificationInfo::NotificationType::SyncELicenseRequest;

            nne::rapidjson::Document::ValueType* type;
            NN_RESULT_DO(JsonParser::FindArray(&type, document, "nintendo_account_ids"));
            NN_RESULT_THROW_UNLESS(!type->Empty(), ns::ResultCommunicationFailed());
            NN_RESULT_THROW_UNLESS(std::distance(type->Begin(), type->End()) <= NN_ARRAY_SIZE(out->syncELicenseRequestInfo.accountIds), ns::ResultCommunicationFailed());
            int idCount = 0;
            for (auto& id : util::MakeSpan(type->Begin(), type->End()))
            {
                NN_RESULT_THROW_UNLESS(id.IsString(), ns::ResultCommunicationFailed());
                const char* pAccountIdString = id.GetString();
                out->syncELicenseRequestInfo.accountIds[idCount++] = { strtoull(pAccountIdString, nullptr, 16) };
            }
            out->syncELicenseRequestInfo.accountCount = idCount;
            NN_RESULT_DO(JsonParser::FindBool(&out->syncELicenseRequestInfo.syncAll, document, "sync_all_accounts"));

            NN_RESULT_SUCCESS;
        }

        // INFO:
        // http://spdlybra.nintendo.co.jp/confluence/display/SIGLO/%E3%83%A6%E3%83%BC%E3%82%B6%E3%83%BC%E3%82%BF%E3%82%A4%E3%83%88%E3%83%AB%E9%85%8D%E4%BF%A1%EF%BC%8F%E8%AD%B0%E4%BA%8B%E9%8C%B2%EF%BC%8F20160817
        Result ParseNotificationInfo(NotificationInfo* out, nne::rapidjson::Document* document) NN_NOEXCEPT
        {
            const char* pTypeString;
            size_t typeStringLength;
            NN_RESULT_DO(JsonParser::FindString(&pTypeString, &typeStringLength, document, "type"));

            struct {
                const char* typeName;
                std::function<Result(NotificationInfo*, nne::rapidjson::Document*)> parser;
            } parseMaps[] =
            {
                { NotificationTypeNup, ParseNupInfo },
                { NotificationTypeDtl, ParseDtlInfo },
                { NotificationTypeVersionList, ParseVersionListInfo },
                { NotificationTypeETicketAvailable, ParseETicketAvailableInfo },
                { NotificationTypeSendRightsUsageStatusRequest, ParseSendRightsUsageStatusRequest },
                { NotificationTypeSyncELicenseRequest, ParseSyncELicenseRequest },
            };

            for (auto& map : parseMaps)
            {
                if (util::Strncmp(map.typeName, pTypeString, static_cast<int>(typeStringLength)) == 0)
                {
                    NN_RESULT_DO(map.parser(out, document));
                    NN_RESULT_SUCCESS;
                }
            }

            NN_PN_TRACE("Invalid type is specified. type: '%s'\n", pTypeString);
            NN_RESULT_THROW(ns::ResultInvalidPushNotificationPayload());
        }
    }

    NotificationDataReader::NotificationDataReader() NN_NOEXCEPT
        : m_Initialized(false), m_Info({})
    {
    }

    Result NotificationDataReader::Initialize(const npns::NotificationData& notificationData) NN_NOEXCEPT
    {
        NN_PN_TRACE("Payload = %s\n", notificationData.GetPayload());

        // INFO: DOM 構築用のバッファを作成
        auto payloadSize = notificationData.GetPayloadSize();
        std::unique_ptr<char[]> domBuffer(new char[payloadSize + 1]);
        std::memcpy(domBuffer.get(), notificationData.GetPayload(), payloadSize);
        domBuffer.get()[payloadSize] = '\0';

        nne::rapidjson::Document document;
        NN_RESULT_DO(JsonParser::InitializeDocument(&document, domBuffer.get()));

        NN_RESULT_DO(ParseNotificationInfo(&m_Info, &document));
        time::PosixTime time;
        if (time::StandardNetworkSystemClock::GetCurrentTime(&time).IsSuccess())
        {
            m_Info.time = time;
        }

        m_Initialized = true;
        NN_RESULT_SUCCESS;
    }

    bool NotificationDataReader::IsSystemUpdateNotification() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Initialized);
        return m_Info.type == NotificationInfo::NotificationType::SystemUpdate;
    }

    bool NotificationDataReader::IsDownloadTaskListNotification() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Initialized);
        return m_Info.type == NotificationInfo::NotificationType::DownloadTaskList;
    }

    bool NotificationDataReader::IsVersionListNotification() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Initialized);
        return m_Info.type == NotificationInfo::NotificationType::VersionList;
    }

    bool NotificationDataReader::IsETicketAvailableNotification() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Initialized);
        return m_Info.type == NotificationInfo::NotificationType::ETicketAvailable;
    }

    bool NotificationDataReader::IsSendRightsUsageStatusRequest() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Initialized);
        return m_Info.type == NotificationInfo::NotificationType::SendRightsUsageStatusRequest;
    }

    bool NotificationDataReader::IsSyncELicenseRequest() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Initialized);
        return m_Info.type == NotificationInfo::NotificationType::SyncELicenseRequest;
    }
}}}
