﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_PseudoDeviceIdManager.h>
#include <nn/os/os_Random.h>
#include <nn/fs.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "ns_Config.h"

#include <nn/nn_SdkLog.h>

namespace nn { namespace ns { namespace srv {

    namespace {
        typedef kvdb::BoundedString<32> Path;
        Path GetSystemSeedPath() NN_NOEXCEPT
        {
            Path path;
            path.AssignFormat("%s:/%s", SystemSeedMountName, "systemseed.dat");
            return path;
        }

        Result CreateSystemSeed() NN_NOEXCEPT
        {
            ns::SystemSeedForPseudoDeviceId systemSeed;
            os::GenerateRandomBytes( &systemSeed, ns::SystemSeedSize );

            fs::FileHandle handle;
            auto path = GetSystemSeedPath();
            NN_RESULT_DO(fs::CreateFile( path, ns::SystemSeedSize ));
            NN_RESULT_DO(fs::OpenFile( &handle, path, fs::OpenMode_Write ));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };
            NN_RESULT_DO(fs::WriteFile( handle, 0, &systemSeed, ns::SystemSeedSize, fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush) ));

            NN_RESULT_SUCCESS;
        }

        Result LoadSystemSeed(nn::ns::SystemSeedForPseudoDeviceId* pSystemSeed) NN_NOEXCEPT
        {
            fs::FileHandle handle;
            NN_RESULT_DO(fs::OpenFile( &handle, GetSystemSeedPath(), fs::OpenMode_Read ));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(handle); };
            NN_RESULT_DO(fs::ReadFile( handle, 0, pSystemSeed, ns::SystemSeedSize ));

            NN_RESULT_SUCCESS;
        }
    }

    Result PseudoDeviceIdManager::Initialize() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lk(m_Mutex);
        NN_RESULT_TRY(fs::MountSystemSaveData( SystemSeedMountName, SystemSeedSaveDataId ));
        NN_RESULT_CATCH(fs::ResultTargetNotFound)
        {
            NN_RESULT_DO(fs::CreateSystemSaveData( SystemSeedSaveDataId,
                                                   SystemSeedSaveDataSize,
                                                   SystemSeedSaveDataJournalSize,
                                                   SystemSeedSaveDataFlags ));
            NN_RESULT_DO(fs::MountSystemSaveData( SystemSeedMountName, SystemSeedSaveDataId ));
            NN_UTIL_SCOPE_EXIT{ fs::Unmount( SystemSeedMountName ); };
            NN_RESULT_DO(CreateSystemSeed());
            NN_RESULT_DO(fs::CommitSaveData( SystemSeedMountName ));
            NN_RESULT_DO(LoadSystemSeed( &m_SystemSeed ));
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT{ fs::Unmount( SystemSeedMountName ); };
        NN_RESULT_DO(LoadSystemSeed( &m_SystemSeed ));

        NN_RESULT_SUCCESS;
    }

    Result PseudoDeviceIdManager::GetSystemSeed(SystemSeedForPseudoDeviceId* pOutSeed) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lk(m_Mutex);
        *pOutSeed = m_SystemSeed;

        NN_RESULT_SUCCESS;
    }

    Result PseudoDeviceIdManager::ResetSystemSeed() NN_NOEXCEPT
    {
        bool isEnabled;
        auto size = settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "ns.pseudodeviceid", "reset_pseudo_device_id");

        if (!(sizeof(isEnabled) == size))
        {
            return ns::ResultFailedToReadFirmwareDebugKey();
        }
        if (!isEnabled)
        {
            return ns::ResultNotPermittedOnProduction();
        }

        NN_RESULT_DO(fs::DeleteSaveData( SystemSeedSaveDataId ));
        return Initialize();
    }

}}}
