﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/fs.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_Logo.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/ns/ns_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include "ns_Config.h"
#include "ns_LogoFileUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        typedef kvdb::BoundedString<48> Path;

        Path MakeLogoDataPath(const char* mountName, const nn::ncm::Path& logoPath) NN_NOEXCEPT
        {
            return Path::MakeFormat("%s:/%s", mountName, logoPath.string);
        }
    }

    Result ReadLogoData(size_t* outValue, void* buffer, size_t bufferSize, ncm::ProgramId programId, const ncm::Path& contentPath, const ncm::Path& logoPath) NN_NOEXCEPT
    {
        NN_RESULT_TRY(fs::MountLogo(LogoMountName, contentPath.string, programId))
            NN_RESULT_CATCH(nn::fs::ResultPartitionNotFound)
            {
                NN_RESULT_THROW(nn::ns::ResultLogoPartitionNotFound());
            }
        NN_RESULT_END_TRY;
        NN_UTIL_SCOPE_EXIT{ fs::Unmount(LogoMountName); };

        fs::FileHandle file;
        auto fileName = MakeLogoDataPath(LogoMountName, logoPath);
        NN_RESULT_TRY(fs::OpenFile(&file, fileName.Get(), fs::OpenMode_Read))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_THROW(nn::ns::ResultLogoPathNotFound());
            }
        NN_RESULT_END_TRY;
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, file));
        auto logoDataSize = static_cast<size_t>(fileSize);

        NN_RESULT_THROW_UNLESS(logoDataSize <= bufferSize, ResultBufferNotEnough());

        NN_RESULT_DO(fs::ReadFile(file, 0, buffer, logoDataSize));

        *outValue = logoDataSize;
        NN_RESULT_SUCCESS;
    }
}}}
