﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/nim/nim_Config.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/ns/srv/ns_FactoryResetManager.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "ns_CleanupUtil.h"

namespace nn { namespace ns { namespace srv {
    Result FactoryResetManager::ResetToFactorySettings() NN_NOEXCEPT
    {
        StopRequestServerPermanently();
        NN_RESULT_DO(DeleteAllImages());
        NN_RESULT_DO(DeleteAllDownloadTask());
        NN_RESULT_DO(DeleteAllBuildInUserContents());
        NN_RESULT_DO(DeleteAllSaveDataAtomic(fs::SaveDataFlags_KeepAfterResettingSystemSaveData, true));
        NN_RESULT_SUCCESS;
    }

    Result FactoryResetManager::ResetToFactorySettingsWithoutUserSaveData() NN_NOEXCEPT
    {
        StopRequestServerPermanently();
        NN_RESULT_DO(DeleteAllDownloadTask());
        NN_RESULT_DO(DeleteAllBuildInUserContents());
        NN_RESULT_DO(DeleteAllSaveDataAtomic(
            fs::SaveDataFlags_KeepAfterResettingSystemSaveData |
            fs::SaveDataFlags_KeepAfterResettingSystemSaveDataWithoutUserSaveData, false));
        NN_RESULT_SUCCESS;
    }

    Result FactoryResetManager::ResetToFactorySettingsForRefurbishment() NN_NOEXCEPT
    {
        StopRequestServerPermanently();
        NN_RESULT_DO(DeleteAllImages());
        NN_RESULT_DO(DeleteAllDownloadTask());
        NN_RESULT_DO(DeleteAllBuildInUserContents());
        NN_RESULT_DO(DeleteAllSaveDataAtomic(fs::SaveDataFlags_KeepAfterRefurbishment, true));
        NN_RESULT_SUCCESS;
    }

    Result FactoryResetManager::DeleteAllDownloadTask() NN_NOEXCEPT
    {
        {
            nim::NetworkInstallTaskId idList[nim::MaxNetworkInstallTaskCount];
            auto count = nim::ListNetworkInstallTask(idList, sizeof(idList) / sizeof(idList[0]));
            for (int i = 0; i < count; i++)
            {
                char buffer[util::Uuid::StringSize];
                NN_UNUSED(buffer);
                NN_DETAIL_NS_TRACE("[FactoryResetManager] Destroy network install task %s\n", idList[i].uuid.ToString(buffer, sizeof(buffer)));
                NN_RESULT_DO(nim::DestroyNetworkInstallTask(idList[i]));
            }
        }

        {
            nim::ApplyDeltaTaskId idList[nim::MaxApplyDeltaTaskCount];
            auto count = nim::ListApplyDeltaTask(idList, sizeof(idList) / sizeof(idList[0]));
            for (int i = 0; i < count; i++)
            {
                char buffer[util::Uuid::StringSize];
                NN_UNUSED(buffer);
                NN_DETAIL_NS_TRACE("[FactoryResetManager] Destroy apply delta task %s\n", idList[i].uuid.ToString(buffer, sizeof(buffer)));
                NN_RESULT_DO(nim::DestroyApplyDeltaTask(idList[i]));
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result FactoryResetManager::DeleteAllImages() NN_NOEXCEPT
    {
        static const char MountName[] = "resetimages";
        NN_RESULT_DO(fs::MountImageDirectory(MountName, fs::ImageDirectoryId::Nand));
        NN_UTIL_SCOPE_EXIT{ fs::Unmount(MountName); };

        kvdb::BoundedString<768> path;
        path.AssignFormat("%s:/", MountName);

        fs::DirectoryHandle directory;
        NN_RESULT_DO(fs::OpenDirectory(&directory, path, fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT{ fs::CloseDirectory(directory); };

        while (NN_STATIC_CONDITION(true))
        {
            int64_t count;
            fs::DirectoryEntry entry;
            NN_RESULT_DO(fs::ReadDirectory(&count, &entry, directory, 1));
            if (count == 0)
            {
                break;
            }

            path.AssignFormat("%s:/%s", MountName, entry.name);
            switch (entry.directoryEntryType)
            {
            case fs::DirectoryEntryType_Directory:
                {
                    NN_RESULT_DO(fs::DeleteDirectoryRecursively(path));
                }
                break;
            case fs::DirectoryEntryType_File:
                {
                    NN_RESULT_DO(fs::DeleteFile(path));
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result FactoryResetManager::DeleteAllBuildInUserContents() NN_NOEXCEPT
    {
        ncm::ContentMetaDatabase db;
        ncm::ContentStorage storage;

        // TODO: ns が通常モードで起動時に built in user DB を Create しているので
        //       メンテナンスモードが２回連続で起動して、呼び出されると失敗してしまう
        //       とりあえず失敗を無視して対処する。
        //       DB が作成されていないということなので、将来的にもこのままかもしれないが
        //       ncm から返ってくる Result はちゃんとしたい。
        auto result = ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInUser);
        if (result.IsFailure())
        {
            NN_DETAIL_NS_TRACE("[FactoryResetManager] Failed to open built in user content meta database 0x%08x\n", result.GetInnerValueForDebug());
            NN_RESULT_SUCCESS;
        }
        result = ncm::OpenContentStorage(&storage, ncm::StorageId::BuildInUser);
        if (result.IsFailure())
        {
            NN_DETAIL_NS_TRACE("[FactoryResetManager] Failed to open built in user content storage 0x%08x\n", result.GetInnerValueForDebug());
            NN_RESULT_SUCCESS;
        }

        ncm::ContentMetaKey keyList[256];
        ncm::ContentManagerAccessor accessor(&db, &storage);
        while (NN_STATIC_CONDITION(true))
        {
            auto listCount = db.ListContentMeta(keyList, sizeof(keyList) / sizeof(keyList[0]),
                ncm::ContentMetaType::Unknown, ncm::ContentMetaDatabase::AnyApplicationId, 0x0, 0xffffffffffffffff,
                ncm::ContentInstallType::Unknown);
            if (listCount.listed == 0)
            {
                break;
            }

            for (int i = 0; i < listCount.listed; i++)
            {
                NN_DETAIL_NS_TRACE("[FactoryResetManager] Delete build-in user contents 0x%016llx\n", keyList[i].id);
                accessor.DeleteAll(keyList[i].id);
            }
        }

        NN_RESULT_DO(db.Commit());

        NN_RESULT_DO(storage.CleanupAllPlaceHolder());

        NN_RESULT_DO(CleanupOrphanContents(ncm::StorageId::BuildInUser));

        NN_RESULT_SUCCESS;
    }

    Result FactoryResetManager::DeleteAllSaveDataAtomic(uint32_t keepFlags, bool deleteUser) NN_NOEXCEPT
    {
        std::unique_ptr<fs::SaveDataIterator> systemIterator;
        NN_RESULT_DO(fs::OpenSaveDataIterator(&systemIterator, fs::SaveDataSpaceId::System));
        std::unique_ptr<fs::SaveDataIterator> userIterator;
        NN_RESULT_DO(fs::OpenSaveDataIterator(&userIterator, fs::SaveDataSpaceId::User));

        bool systemRemains = true;
        bool userRemains = true;

        // SD カードに入っているセーブデータは、そもそも SD カードが使えなくなって Cleanup を強要されるので、
        // わざわざここでクリーンナップする必要はない

        while (systemRemains || userRemains)
        {
            int countListed = 0;
            while (systemRemains && countListed < MaxSaveDataIdCountForAtomicReset)
            {
                fs::SaveDataInfo info;
                int64_t count;
                NN_RESULT_DO(systemIterator->ReadSaveDataInfo(&count, &info, 1));
                if (count == 0)
                {
                    systemRemains = false;
                    break;
                }

                uint32_t flags;
                auto result = fs::GetSaveDataFlags(&flags, info.saveDataId);
                if (result.IsFailure() || (flags & keepFlags) == 0)
                {
                    NN_DETAIL_NS_TRACE("[FactoryResetManager] System save data 0x%016llx (0x%016llx) listed\n", info.saveDataId, info.systemSaveDataId);
                    if (result.IsFailure())
                    {
                        NN_DETAIL_NS_TRACE("[FactoryResetManager]     Failed to get flags as 0x%08x\n", result.GetInnerValueForDebug());
                    }
                    m_SaveDataIdBuffer[countListed] = info.saveDataId;
                    countListed++;
                }
                else
                {
                    NN_DETAIL_NS_TRACE("[FactoryResetManager] System save data 0x%016llx ignored\n", info.saveDataId);
                }
            }

            while (userRemains && countListed < MaxSaveDataIdCountForAtomicReset)
            {
                fs::SaveDataInfo info;
                int64_t count;
                NN_RESULT_DO(userIterator->ReadSaveDataInfo(&count, &info, 1));
                if (count == 0)
                {
                    userRemains = false;
                    break;
                }

                if (info.saveDataType == fs::SaveDataType::Bcat ||
                    info.saveDataType == fs::SaveDataType::Cache ||
                    deleteUser)
                {
                    NN_DETAIL_NS_TRACE("[FactoryResetManager] User save data 0x%016llx listed\n", info.saveDataId);

                    m_SaveDataIdBuffer[countListed] = info.saveDataId;
                    countListed++;
                }
            }

            NN_RESULT_DO(fs::RegisterSaveDataAtomicDeletion(m_SaveDataIdBuffer, countListed));
            NN_DETAIL_NS_TRACE("[FactoryResetManager] %d save data registered to atomic deletion\n", countListed);
        }

        NN_RESULT_SUCCESS;
    }
}}}
