﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/fs.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/ns/srv/ns_BlackListReader.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace ns { namespace srv {
namespace {

    const int PathLengthMax = 32;
    const char* MountName = "blacklist";
    const char* BlackListFileName = "blacklist.dat";

} // ~namespace nn::ns::srv::<anonymous>

BlackListReader::BlackListReader() NN_NOEXCEPT :
    m_IsMounted(false)
{
}

BlackListReader::~BlackListReader() NN_NOEXCEPT
{
    if (m_IsMounted)
    {
        Unmount();
    }
}

void BlackListReader::Mount() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!m_IsMounted);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    size_t cacheSize;
    ncm::SystemDataId dataId = { NN_NS_SYSTEM_DATA_ID_OF_APPLICATIONBLACKLIST };
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::QueryMountSystemDataCacheSize(&cacheSize, dataId));

    m_MountCache.reset(new Bit8[cacheSize]);
    NN_ABORT_UNLESS_NOT_NULL(m_MountCache);

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::MountSystemData(MountName, dataId, m_MountCache.get(), cacheSize));
#endif
    m_IsMounted = true;
}

void BlackListReader::Unmount() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_IsMounted);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    fs::Unmount(MountName);
    m_MountCache.reset(nullptr);
#endif
    m_IsMounted = false;
}

void BlackListReader::Read(BlackListData* buffer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_IsMounted);
    NN_ABORT_UNLESS_NOT_NULL(buffer);

    size_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadAll(&size, buffer->records, sizeof(buffer->records), BlackListFileName));
    NN_ABORT_UNLESS((size % sizeof(BlackListRecord)) == 0);
    buffer->count = static_cast<int>(size / sizeof(BlackListRecord));
}

Result BlackListReader::ReadAll(size_t* readSize, void* readBuffer, size_t bufferSize, const char* fileName) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(readSize);
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);
    NN_ABORT_UNLESS_NOT_NULL(fileName);

    *readSize = 0u;

    char filePath[PathLengthMax];
    NN_ABORT_UNLESS(util::SNPrintf(filePath, sizeof(filePath), "%s:/%s", MountName, fileName) < PathLengthMax);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    fs::FileHandle fileHandle;
    NN_RESULT_DO(fs::OpenFile(&fileHandle, filePath, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(fileHandle); };

    int64_t fileSize;
    NN_RESULT_DO(fs::GetFileSize(&fileSize, fileHandle));
    NN_ABORT_UNLESS(fileSize <= static_cast<int64_t>(bufferSize));
    NN_RESULT_DO(fs::ReadFile(readSize, fileHandle, 0, readBuffer, std::min(static_cast<size_t>(fileSize), bufferSize)));
#else
    NN_UNUSED(bufferSize);
#endif

    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::ns::srv
