﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/ec/system/ec_TicketSystemApi.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/os/os_Thread.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_ScopedTransaction.h>

#include "ns_AsyncPreInstallImpl.h"
#include "ns_AsyncThreadAllocator.h"

namespace nn { namespace ns { namespace srv {

    //--------------------------------------------------------------------------
    //  AsyncCleanupPreInstalledApplicationImpl
    //--------------------------------------------------------------------------
    AsyncCleanupPreInstalledApplicationImpl::~AsyncCleanupPreInstalledApplicationImpl() NN_NOEXCEPT
    {
        if (m_ThreadInfo)
        {
            CancelImpl();
            os::WaitThread(m_ThreadInfo->thread);
            os::DestroyThread(m_ThreadInfo->thread);
            GetAsyncThreadAllocator()->Free(*m_ThreadInfo);
        }
    }

    Result AsyncCleanupPreInstalledApplicationImpl::Run() NN_NOEXCEPT
    {
        ThreadInfo info;
        NN_RESULT_DO(GetAsyncThreadAllocator()->Allocate(&info));
        bool isSuccess = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccess)
            {
                GetAsyncThreadAllocator()->Free(info);
            }
        };
        info.priority = NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncCleanupPreInstallTask);
        NN_RESULT_DO(os::CreateThread(info.thread, [](void* p)
            {
                auto t = reinterpret_cast<AsyncCleanupPreInstalledApplicationImpl*>(p);
                t->m_Result = t->Execute();
                t->m_Event.Signal();
            }, this, info.stack, info.stackSize, info.priority));
        os::SetThreadNamePointer(info.thread, NN_SYSTEM_THREAD_NAME(nssrv, AsyncCleanupPreInstallTask));
        os::StartThread(info.thread);
        m_ThreadInfo = info;
        isSuccess = true;

        NN_RESULT_SUCCESS;
    }

    Result AsyncCleanupPreInstalledApplicationImpl::Execute() NN_NOEXCEPT
    {
        // チケット同期は起動シーケンスでも行うため、先にフラグを折る
        if (m_AppId)
        {
            NN_RESULT_DO(m_pRecordDb->ClearPreInstalledApplicationFlag(*m_AppId));
        }
        else
        {
            NN_RESULT_DO(m_pRecordDb->ClearAllPreInstalledApplicationFlag());
        }

        {
            util::ScopedTransaction transaction;
            NN_UTIL_LOCK_GUARD(m_Mutex);
            m_Async.emplace();
            NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
            {
                m_Async = util::nullopt;
            };

            NN_RESULT_DO(ec::system::RequestSyncTicketForSystem(&(*m_Async)));

            transaction.Commit();
        }
        NN_RESULT_DO(m_Async->Get());

        NN_RESULT_SUCCESS;
    }

    Result AsyncCleanupPreInstalledApplicationImpl::GetImpl() NN_NOEXCEPT
    {
        return m_Result;
    }

    void AsyncCleanupPreInstalledApplicationImpl::CancelImpl() NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Mutex);
        if (m_Async)
        {
            m_Async->Cancel();
        }
    }

    Result AsyncCleanupPreInstalledApplicationImpl::GetErrorContext(sf::Out<err::ErrorContext> outValue) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Mutex);
        if (m_Async)
        {
            m_Async->GetErrorContext(&(*outValue));
        }
        else
        {
            *outValue = ErrorContextHolder::GetErrorContextImpl();
        }
        NN_RESULT_SUCCESS;
    }

}}} // namespace nn::ns::srv
