﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <cstring>
#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_Application.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_Mount.h>
#include <nn/nn_Result.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringView.h>

#include "ns_ApplicationPathRedirectionUtil.h"
#include "ns_ProgramIndexUtil.h"
#include "ns_SystemUpdateUtil.h"

using namespace nn;

namespace nn { namespace ns { namespace srv {

namespace {
    constexpr nn::ncm::ProgramId DummyId = { 0xFF00000000000000 }; // TORIAEZU
    constexpr int MountNameLength = fs::MountNameLengthMax + 1;

    bool IsMatchTail(const char* str, const char* tail) NN_NOEXCEPT
    {
        auto strLength = strlen(str);
        auto tailLength = strlen(tail);
        if (tailLength > strLength)
        {
            return false;
        }
        return (util::string_view(str).substr(strLength - tailLength)
                == util::string_view(tail));
    }

    void MakeUniqueMountName(char* mountName, size_t bufferSize) NN_NOEXCEPT
    {
        static std::atomic_uint s_Count;
        util::SNPrintf(mountName, bufferSize, "aput%08x", s_Count.fetch_add(1));
    }

    Result ReadApplicationControlPropertyFromHostFile(::std::unique_ptr<ApplicationControlProperty>* outBuffer, const char* pPath) NN_NOEXCEPT
    {
        // Open .nacp file
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, pPath, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        ::std::unique_ptr<ApplicationControlProperty> property(new ApplicationControlProperty);
        NN_RESULT_THROW_UNLESS(property, ResultBufferNotEnough());

        NN_RESULT_DO(fs::ReadFile(file, 0, property.get(), sizeof(ApplicationControlProperty)));

        *outBuffer = ::std::move(property);

        NN_RESULT_SUCCESS;
    }

    class ApplicationPackageReader
    {
        NN_DISALLOW_COPY(ApplicationPackageReader);

    public:
        ApplicationPackageReader() = default;
        ~ApplicationPackageReader() NN_NOEXCEPT
        {
            if (m_IsMounted)
            {
                fs::Unmount(m_MountName);
            }
        }

        Result Initialize(const char* path) NN_NOEXCEPT
        {
            m_OriginalPath = path;
            m_ExtensionType = GetExtensionType(path);

            NN_RESULT_THROW_UNLESS(m_ExtensionType != ExtensionType::None, fs::ResultPathNotFound());

            if (m_ExtensionType == ExtensionType::Nca)
            {
                m_ProgramId = DummyId;
                m_ProgramIndex = 0;

                NN_RESULT_SUCCESS;
            }

            MakeUniqueMountName(m_MountName, MountNameLength);
            NN_RESULT_DO(fs::MountApplicationPackage(m_MountName, path));

            util::SNPrintf(m_RootDirectoryPath, NN_ARRAY_SIZE(m_RootDirectoryPath), "%s:/", m_MountName);

            NN_RESULT_DO(SetContentMetaBuffer());

            NN_RESULT_DO(GetProgramIdAndProgramIndex(&m_ProgramId, &m_ProgramIndex));

            m_IsMounted = true;

            NN_RESULT_SUCCESS;
        }

        Result SearchContent(bool* outValue, lr::Path* outPath, const char* tailString) const NN_NOEXCEPT
        {
            fs::DirectoryHandle directoryHandle;
            NN_RESULT_DO(fs::OpenDirectory(&directoryHandle, m_RootDirectoryPath, m_ExtensionType == Nsp ? fs::OpenDirectoryMode_File : fs::OpenDirectoryMode_All));
            NN_UTIL_SCOPE_EXIT
            {
                fs::CloseDirectory(directoryHandle);
            };

            while (NN_STATIC_CONDITION(true))
            {
                fs::DirectoryEntry entry;
                int64_t outRead;
                NN_RESULT_DO(fs::ReadDirectory(&outRead, &entry, directoryHandle, 1));
                if (outRead == 0)
                {
                    break;
                }

                if (IsMatchTail(entry.name, tailString))
                {
                    *outValue = true;
                    if (outPath)
                    {
                        auto length = util::SNPrintf(outPath->string, sizeof(outPath->string), "%s/%s", m_OriginalPath, entry.name);
                        NN_RESULT_THROW_UNLESS(length + 1 < sizeof(outPath->string), ResultBufferNotEnough());
                        if (entry.directoryEntryType == fs::DirectoryEntryType_Directory)
                        {
                            outPath->string[length] = '/';
                            outPath->string[length + 1] = '\0';
                        }
                    }

                    NN_RESULT_SUCCESS;
                }
            }

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        ExtensionType GetExtension() const NN_NOEXCEPT
        {
            return m_ExtensionType;
        }

        Result GetContentPath(lr::Path* outValue, ncm::ContentType type, util::optional<uint8_t> index) const NN_NOEXCEPT
        {
            if (m_ExtensionType == ExtensionType::Nca && type == ncm::ContentType::Program)
            {
                auto length = util::SNPrintf(outValue->string, sizeof(outValue->string), "%s", m_OriginalPath);
                NN_RESULT_THROW_UNLESS(length + 1 == sizeof(outValue->string), ResultBufferNotEnough());
            }
            else if (m_ExtensionType == ExtensionType::Nsp)
            {
                NN_RESULT_DO(GetContentPathInNsp(outValue, type, index));
            }
            else if(m_ExtensionType == ExtensionType::Nspd)
            {
                NN_RESULT_DO(GetContentPathInNspd(outValue, type, index));
            }
            else
            {
                NN_RESULT_THROW(ResultApplicationContentNotFound());
            }

            NN_RESULT_SUCCESS;
        }

        util::optional<ncm::ApplicationId> GetApplicationId() const NN_NOEXCEPT
        {
            if (!m_ContentMetaBuffer.Get())
            {
                return util::nullopt;
            }

            auto metaReader = ncm::PackagedContentMetaReader(m_ContentMetaBuffer.Get(), m_ContentMetaBuffer.GetSize());
            return metaReader.GetApplicationId();
        }

        uint32_t GetVersion() const NN_NOEXCEPT
        {
            if (m_ContentMetaBuffer.Get())
            {
                auto metaReader = ncm::PackagedContentMetaReader(m_ContentMetaBuffer.Get(), m_ContentMetaBuffer.GetSize());
                return metaReader.GetKey().version;
            }
            else
            {
                return 0;
            }
        }

        bool IsApplication() const NN_NOEXCEPT
        {
            if (m_ContentMetaBuffer.Get())
            {
                auto metaReader = ncm::PackagedContentMetaReader(m_ContentMetaBuffer.Get(), m_ContentMetaBuffer.GetSize());
                return metaReader.GetHeader()->type == ncm::ContentMetaType::Application;
            }
            else
            {
                return false;
            }
        }

// Generic では SystemVersion が存在しないので、VerifyRequiredSystemVersion() をビルドしない
#if !defined(NN_BUILD_CONFIG_SPEC_GENERIC)
        Result VerifyRequiredSystemVersion() const NN_NOEXCEPT
        {
            auto metaReader = ncm::PackagedContentMetaReader(m_ContentMetaBuffer.Get(), m_ContentMetaBuffer.GetSize());
            auto contentMetaType = metaReader.GetHeader()->type;
            NN_RESULT_THROW_UNLESS(contentMetaType == ncm::ContentMetaType::Application, ncm::ResultInvalidContentMetaKey());

            uint32_t requiredSystemVersion = metaReader.GetExtendedHeader<ncm::ApplicationMetaExtendedHeader>()->requiredSystemVersion;
            auto systemUpdateMetaVersion = GetSystemUpdateVersion();
            NN_RESULT_THROW_UNLESS(systemUpdateMetaVersion >= requiredSystemVersion, ResultSystemUpdateRequired());
            NN_RESULT_SUCCESS;
        }
#endif

        Result HasContent(bool* outValue, ncm::ContentType type, util::optional<uint8_t> index = util::nullopt) const NN_NOEXCEPT
        {
            lr::Path path;
            NN_RESULT_TRY(GetContentPath(&path, type, index))
                NN_RESULT_CATCH(ResultApplicationContentNotFound)
                {
                    *outValue = false;
                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY

            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        ncm::ProgramId GetProgramId() const NN_NOEXCEPT
        {
            return m_ProgramId;
        }

        uint8_t GetProgramIndex() const NN_NOEXCEPT
        {
            return m_ProgramIndex;
        }

    private:
        Result SetContentMetaBuffer() NN_NOEXCEPT
        {
            const char* contentMetaExtension = ".cnmt.nca";
            const char* contentMetaDirectoryExtension = "meta0.ncd";

            NN_RESULT_THROW_UNLESS(m_ExtensionType == ExtensionType::Nsp || m_ExtensionType == ExtensionType::Nspd, ResultContentMetaNotFound());

            bool hasContent;
            lr::Path metaPath;
            auto extension = m_ExtensionType == ExtensionType::Nsp ? contentMetaExtension : contentMetaDirectoryExtension;
            NN_RESULT_DO(SearchContent(&hasContent, &metaPath, extension));
            NN_RESULT_THROW_UNLESS(hasContent, ResultContentMetaNotFound());

            ncm::AutoBuffer autoBuffer;
            NN_RESULT_DO(ncm::ReadContentMetaPath(&m_ContentMetaBuffer, metaPath.string));

            NN_RESULT_SUCCESS;
        }

        Result GetContentPathInNsp(lr::Path* outValue, ncm::ContentType type, util::optional<uint8_t> index) const NN_NOEXCEPT
        {
            auto metaReader = ncm::PackagedContentMetaReader(m_ContentMetaBuffer.Get(), m_ContentMetaBuffer.GetSize());
            const ncm::PackagedContentInfo* contentInfo {};
            if (index)
            {
                contentInfo = metaReader.GetContentInfo(type, *index);
            }
            else
            {
                contentInfo = metaReader.GetContentInfo(type);
            }
            NN_RESULT_THROW_UNLESS(contentInfo, ResultApplicationContentNotFound());

            auto contentId = contentInfo->GetId();
            ncm::ContentIdString idString;
            ncm::GetStringFromContentId(idString.data, sizeof(idString.data), contentId);
            char fileName[ncm::ContentIdStringLength + 5];
            auto length = util::SNPrintf(fileName, sizeof(fileName), "%s.nca", idString.data);
            NN_RESULT_THROW_UNLESS(length + 1 == sizeof(fileName), ResultBufferNotEnough());

            bool hasContent;
            NN_RESULT_DO(SearchContent(&hasContent, outValue, fileName));
            NN_RESULT_THROW_UNLESS(hasContent, ResultApplicationContentNotFound());

            NN_RESULT_SUCCESS;
        }

        Result GetContentPathInNspd(lr::Path* outValue, ncm::ContentType type, util::optional<uint8_t> index) const NN_NOEXCEPT
        {
            NN_UNUSED(index);
            // TORIAEZU: nspd の場合は必ず 0 として扱う
            const uint8_t ProgramIndex = 0;
            const char* contentName = nullptr;
            switch (type)
            {
            case ncm::ContentType::Program:
                contentName = "program";
                break;
            case ncm::ContentType::Control:
                contentName = "control";
                break;
            case ncm::ContentType::HtmlDocument:
                contentName = "htmlDocument";
                break;
            case ncm::ContentType::LegalInformation:
                contentName = "legalInformation";
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }

            char fileName[32];
            auto length = util::SNPrintf(fileName, sizeof(fileName), "%s%d.ncd", contentName, ProgramIndex);
            NN_RESULT_THROW_UNLESS(length + 1 <= sizeof(fileName), ResultBufferNotEnough());

            bool hasContent;
            NN_RESULT_DO(SearchContent(&hasContent, outValue, fileName));
            NN_RESULT_THROW_UNLESS(hasContent, ResultApplicationContentNotFound());

            NN_RESULT_SUCCESS;
        }

        Result GetProgramIdAndProgramIndex(ncm::ProgramId* outId, uint8_t* outIndex) const NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(m_ContentMetaBuffer.Get(), ResultContentMetaNotFound());

            auto metaReader = ncm::PackagedContentMetaReader(m_ContentMetaBuffer.Get(), m_ContentMetaBuffer.GetSize());
            auto contentMetaId = metaReader.GetKey().id;
            auto appId = metaReader.GetApplicationId();

            if (m_ExtensionType == ExtensionType::Nspd)
            {
                if (!appId)
                {
                    ncm::ProgramId programId = { contentMetaId };
                    *outId = programId;
                    *outIndex = 0;
                }
                else
                {
                    bool hasContent;
                    NN_RESULT_DO(HasContent(&hasContent, ncm::ContentType::Control));
                    NN_RESULT_THROW_UNLESS(hasContent, ResultApplicationContentNotFound());

                    // TORIAEZU: 決め打ちでアプリケーション管理データを読む
                    lr::Path path;
                    util::SNPrintf(path.string, sizeof(path.string), "%s/control0.ncd/data/control.nacp", m_RootDirectoryPath);

                    std::unique_ptr<ApplicationControlProperty> property;
                    NN_RESULT_DO(ReadApplicationControlPropertyFromHostFile(&property, path.string));
                    *outId = srv::GetProgramId(*appId, property->programIndex);
                    *outIndex = property->programIndex;
                }
                NN_RESULT_SUCCESS;
            }

            auto programContentInfo = metaReader.GetContentInfo(ncm::ContentType::Program);
            NN_RESULT_THROW_UNLESS(programContentInfo, ResultApplicationContentNotFound());
            if (!appId)
            {
                ncm::ProgramId programId = { contentMetaId };
                *outId = programId;
                NN_RESULT_THROW_UNLESS(programContentInfo->GetIdOffset() == 0, ResultUnexpectedProgramIndex());
                *outIndex = 0;
                NN_RESULT_SUCCESS;
            }

            *outId = srv::GetProgramId(*appId, programContentInfo->GetIdOffset());
            *outIndex = programContentInfo->GetIdOffset();
            NN_RESULT_SUCCESS;
        }

        const char* m_OriginalPath;
        char m_MountName[MountNameLength];
        char m_RootDirectoryPath[MountNameLength + 2];
        ncm::AutoBuffer m_ContentMetaBuffer;
        ncm::ProgramId m_ProgramId{};
        uint8_t m_ProgramIndex;
        ExtensionType m_ExtensionType;
        bool m_IsMounted {};
    };

    void RedirectApplicationContentPath(
        const lr::Path& path,
        ncm::ProgramId programId,
        ncm::ContentType contentType,
        lr::LocationResolver* pLocationResolver) NN_NOEXCEPT
    {
        switch(contentType)
        {
        case ncm::ContentType::Program:
            pLocationResolver->RedirectApplicationProgramPath(programId, path);
            break;
        case ncm::ContentType::Control:
            pLocationResolver->RedirectApplicationControlPath(programId, path);
            break;
        case ncm::ContentType::HtmlDocument:
            pLocationResolver->RedirectApplicationHtmlDocumentPath(programId, path);
            break;
        case ncm::ContentType::LegalInformation:
            pLocationResolver->RedirectApplicationLegalInformationPath(programId, path);
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    Result RedirectContentIfExist(
        ApplicationPackageReader* pPackage,
        ncm::ContentType contentType,
        lr::LocationResolver* pLocationResolver,
        util::optional<uint8_t> index = util::nullopt) NN_NOEXCEPT
    {
        ncm::ProgramId programId = pPackage->GetProgramId();

        if (!pPackage->IsApplication())
        {
            if (contentType == ncm::ContentType::Program)
            {
                lr::Path path;
                NN_RESULT_DO(pPackage->GetContentPath(&path, contentType, index));
                pLocationResolver->RedirectProgramPath(programId, path);
            }
            NN_RESULT_SUCCESS;
        }

        bool hasContent;
        NN_RESULT_DO(pPackage->HasContent(&hasContent, contentType, index));
        if (hasContent)
        {
            lr::Path path;
            NN_RESULT_DO(pPackage->GetContentPath(&path, contentType, index));
            RedirectApplicationContentPath(path, programId, contentType, pLocationResolver);
        }
        NN_RESULT_SUCCESS;
    }
}

Result LoadHostProgram(ncm::ProgramId* outProgramId, uint8_t* outIndex, util::optional<ncm::ApplicationId>* outApplicationId, uint32_t* outVersion, const char* pPath, lr::LocationResolver* pLocationResolver) NN_NOEXCEPT
{
    ApplicationPackageReader package;
    NN_RESULT_DO(package.Initialize(pPath));

    auto programId = package.GetProgramId();
    if (package.IsApplication())
    {
// Generic では SystemVersion が存在しないので、VerifyRequiredSystemVersion() を呼ばない
#if !defined(NN_BUILD_CONFIG_SPEC_GENERIC)
        NN_RESULT_DO(package.VerifyRequiredSystemVersion());
#endif
        NN_RESULT_DO(pLocationResolver->ClearApplicationRedirection());
    }
    else
    {
        NN_RESULT_DO(pLocationResolver->EraseProgramRedirection(programId));
    }

    NN_RESULT_DO(RedirectContentIfExist(&package, ncm::ContentType::Program, pLocationResolver));
    NN_RESULT_DO(RedirectContentIfExist(&package, ncm::ContentType::Control, pLocationResolver));
    NN_RESULT_DO(RedirectContentIfExist(&package, ncm::ContentType::HtmlDocument, pLocationResolver));
    NN_RESULT_DO(RedirectContentIfExist(&package, ncm::ContentType::LegalInformation, pLocationResolver));

    *outProgramId = programId;
    *outIndex = package.GetProgramIndex();
    *outApplicationId = package.GetApplicationId();
    *outVersion = package.GetVersion();

    NN_RESULT_SUCCESS;
}

ExtensionType GetExtensionType(const char* fileName) NN_NOEXCEPT
{
    if (IsMatchTail(fileName, ".nca"))
    {
        return ExtensionType::Nca;
    }
    else if (IsMatchTail(fileName, ".nsp"))
    {
        return ExtensionType::Nsp;
    }
    else if (IsMatchTail(fileName, ".nspd/") || IsMatchTail(fileName, ".nspd"))
    {
        return ExtensionType::Nspd;
    }
    else
    {
        return ExtensionType::None;
    }
}
}}}
