﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/es/es_Api.h>
#include <nn/es/es_Result.h>
#include <nn/es/es_Types.h>
#include <nn/fs/fs_RightsId.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_ApplicationCopyIdentifierManager.h>
#include <nn/result/result_HandlingUtility.h>
#include "ns_ProgramIndexUtil.h"

namespace nn { namespace ns { namespace srv {
    Result ApplicationCopyIdentifierManager::GetGameCardApplicationCopyIdentifier(sf::Out<ns::GameCardApplicationCopyIdentifier> outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        GameCardInfo info;
        NN_RESULT_DO(m_GameCardManager->GetGameCardInfo(&info));

        auto begin = info.idList;
        auto end = &info.idList[info.idCount];
        NN_RESULT_THROW_UNLESS(std::find(begin, end, id) != end, ResultApplicationCopyIdentifierNotFound());

        std::memcpy(outValue->deviceId, info.deviceId, sizeof(outValue->deviceId));
        std::memcpy(outValue->certificate, info.cert, sizeof(outValue->certificate));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationCopyIdentifierManager::GetInstalledApplicationCopyIdentifier(sf::Out<ns::InstalledApplicationCopyIdentifier> outValue, ncm::ApplicationId id, ncm::StorageId storage) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_UNUSED(storage);

        NN_RESULT_THROW(ResultNotImplemented());
#else
        es::RightsIdIncludingKeyId rightsIdKeyId;
        {
            fs::RightsId rightsId;
            NN_RESULT_DO(fs::GetRightsId(&rightsId, GetProgramId(id, 0), storage));

            fs::RightsId zeroRightsId = {};
            NN_RESULT_THROW_UNLESS(std::memcmp(rightsId.data, zeroRightsId.data, sizeof(rightsId.data)) != 0, ResultApplicationCopyIdentifierRightsIdNotDescribed());

            rightsIdKeyId = es::RightsIdIncludingKeyId::Construct(rightsId);
        }

        size_t outSize;
        NN_RESULT_TRY(es::GetEncryptedTicketData(&outValue->ticketId, &outSize, outValue->ticket, sizeof(outValue->ticket), outValue->key, sizeof(outValue->key), rightsIdKeyId))
            NN_RESULT_CATCH_CONVERT(es::ResultRightsIdNotFound, ResultApplicationCopyIdentifierTicketNotFound())
        NN_RESULT_END_TRY
        outValue->ticketSize = outSize;

        NN_RESULT_SUCCESS;
#endif
    }
}}}
