﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/lr/lr_RegisteredLocationResolver.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_ApplicationControlDataManager.h>
#include <nn/ns/srv/ns_ApplicationRecordDatabase.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include "ns_ApplicationControlFileUtil.h"
#include "ns_AsyncDownloadApplicationControlDataImpl.h"
#include "ns_AsyncImpl.h"
#include "ns_Config.h"
#include "ns_ProgramIndexUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        typedef sf::ObjectFactory<sf::impl::StaticOneAllocationPolicy> StaticOneFactory;

        Result GetRedirectedControlPath(lr::Path* outValue, ncm::ProgramId id) NN_NOEXCEPT
        {
            lr::LocationResolver hostLocationResolver;
            NN_RESULT_DO(lr::OpenLocationResolver(&hostLocationResolver, ncm::StorageId::Host));

            NN_RESULT_TRY(hostLocationResolver.ResolveApplicationControlPath(outValue, id))
                NN_RESULT_CATCH_CONVERT(lr::ResultControlNotFound, ResultApplicationControlDataNotFound())
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        Result GetGameCardControlPath(ncm::Path* outValue) NN_NOEXCEPT
        {
            constexpr auto StorageId = ncm::StorageId::GameCard;
            ncm::ContentMetaDatabase db;
            NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, StorageId));
            ncm::ContentStorage storage;
            NN_RESULT_DO(ncm::OpenContentStorage(&storage, StorageId));

            ncm::ContentMetaKey appKey;
            auto count = db.ListContentMeta(&appKey, 1, ncm::ContentMetaType::Application);
            NN_RESULT_THROW_UNLESS(count.total == 1, ResultApplicationContentNotFound());

            ncm::ContentMetaKey patchKey;
            count = db.ListContentMeta(&patchKey, 1, ncm::ContentMetaType::Patch);
            bool hasPatch = count.total == 1;

            auto mainContentKey = hasPatch ? patchKey : appKey;

            // 記録がないので、ContentMetaDatabase と ContentStorage に問い合わせて Control へのパスを取得して arp に渡す
            ncm::ContentId contentId;
            NN_RESULT_DO(db.GetContentIdByType(&contentId, mainContentKey, ncm::ContentType::Control));

            storage.GetPath(outValue, contentId);

            NN_RESULT_SUCCESS;
        }
    }

    Result ApplicationControlDataManager::Initialize(ApplicationRecordDatabase* recordDb, IntegratedContentManager* integrated, RequestServer* requestServer) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Cache.Initialize(ApplicationControlCacheSaveDataId, NsaCacheMountName, m_CacheBufferBuffer, sizeof(m_CacheBufferBuffer), ApplicationControlCacheSaveDataSize, ApplicationControlCacheSaveDataJournalSize, ApplicationControlCacheSaveDataFlags));
        m_RecordDb = recordDb;
        m_Integrated = integrated;
        m_RequestServer = requestServer;

        NN_RESULT_SUCCESS;
    }

    void ApplicationControlDataManager::Finalize() NN_NOEXCEPT
    {
        m_Cache.Finalize();
    }

    Result ApplicationControlDataManager::GetApplicationControlData(sf::Out<std::uint32_t> outValue, const sf::OutBuffer& buffer, ns::ApplicationControlSource source, ncm::ApplicationId id) NN_NOEXCEPT
    {
        size_t size;
        NN_RESULT_DO(Get(&size, buffer.GetPointerUnsafe(), buffer.GetSize(), source, id));

        *outValue = static_cast<uint32_t>(size);
        NN_RESULT_SUCCESS;
    }

    Result ApplicationControlDataManager::GetApplicationControlProperty(ApplicationControlProperty* outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT
    {
        if ((info.launchInfoFlags & static_cast<Bit8>(ApplicationLaunchInfoFlag::AutoBootByGameCard)) == static_cast<Bit8>(ApplicationLaunchInfoFlag::AutoBootByGameCard))
        {
            NN_RESULT_THROW_UNLESS(info.applicationStorageId == ncm::StorageId::GameCard, ResultApplicationControlDataNotFound());
            ncm::Path path = {};
            NN_RESULT_DO(GetGameCardControlPath(&path));
            NN_RESULT_DO(ReadApplicationControlProperty(outValue, info.id, programIndex, path.string));
        }
        else if (info.applicationStorageId == ncm::StorageId::Host)
        {
            NN_RESULT_THROW_UNLESS(info.patchStorageId == ncm::StorageId::None, ResultApplicationControlDataNotFound());
            lr::Path controlPath = {};
            NN_RESULT_DO(GetRedirectedControlPath(&controlPath, GetProgramId(info.id, programIndex)));
            NN_RESULT_DO(ReadApplicationControlProperty(outValue, info.id, programIndex, controlPath.string));
        }
        else
        {
            NN_RESULT_DO(GetByProgramIndex(outValue, info.id, programIndex));
        }
        NN_RESULT_SUCCESS;
    }

    Result ApplicationControlDataManager::InvalidateAllApplicationControlCache() NN_NOEXCEPT
    {
        return m_Cache.DeleteAll();
    }

    Result ApplicationControlDataManager::InvalidateApplicationControlCache(ncm::ApplicationId id) NN_NOEXCEPT
    {
        return m_Cache.DeleteById(id);
    }

    Result ApplicationControlDataManager::RequestDownloadApplicationControlData(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncResult, AsyncDownloadApplicationControlDataImpl>(this, m_RequestServer->Stop());
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(id));
        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;

        NN_RESULT_SUCCESS;
    }

    Result ApplicationControlDataManager::GetMaxApplicationControlCacheCount(sf::Out<std::int32_t> outValue) NN_NOEXCEPT
    {
        *outValue = ApplicationControlKeyValueCache::MaxEntryCount;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationControlDataManager::Get(size_t* outValue, void* buffer, size_t bufferSize, ns::ApplicationControlSource source, ncm::ApplicationId id, bool alongWithIcon) NN_NOEXCEPT
    {
        util::optional<ncm::ContentMetaKey> cacheKey;
        util::optional<ncm::ContentMetaKey> storageKey;
        if (m_RecordDb->Has(id))
        {
            NN_RESULT_DO(m_RecordDb->FindControl(&cacheKey, &storageKey, id));
        }

        // storageKey があるときは必ず cacheKey がある
        if (cacheKey)
        {
            settings::LanguageCode language;
            settings::GetLanguageCode(&language);

            size_t size;
            auto result = GetByKey(&size, buffer, bufferSize, source, ApplicationControlKey::Make(id, cacheKey->version, language), storageKey, alongWithIcon);
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(ResultApplicationControlDataNotFound) {}
            NN_RESULT_END_TRY

            if (result.IsSuccess())
            {
                *outValue = size;
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_THROW_UNLESS(source != ApplicationControlSource::StorageOnly, ResultApplicationControlDataNotFound());

        size_t size;
        NN_RESULT_DO(GetLatestFromCache(&size, buffer, bufferSize, id));
        *outValue = size;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationControlDataManager::GetByProgramIndex(ApplicationControlProperty* outValue, ncm::ApplicationId id, uint8_t programIndex) NN_NOEXCEPT
    {
        util::optional<ncm::ContentMetaKey> cacheKey;
        util::optional<ncm::ContentMetaKey> storageKey;
        NN_RESULT_THROW_UNLESS(m_RecordDb->Has(id), ResultApplicationControlDataNotFound());
        NN_RESULT_DO(m_RecordDb->FindControl(&cacheKey, &storageKey, id));
        NN_RESULT_THROW_UNLESS(storageKey, ResultApplicationControlDataNotFound());

        settings::LanguageCode language;
        settings::GetLanguageCode(&language);

        size_t size;
        NN_RESULT_DO(GetFromStorage(&size, outValue, sizeof(ApplicationControlProperty), *storageKey, ApplicationControlKey::Make(id, storageKey->version, language), programIndex, false));
        NN_ABORT_UNLESS(size == sizeof(ApplicationControlProperty));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationControlDataManager::PutPath(ncm::ApplicationId id, uint32_t version, uint8_t programIndex, const ncm::Path& path, bool notifyParentalControl) NN_NOEXCEPT
    {
        settings::LanguageCode language;
        settings::GetLanguageCode(&language);

        {
            std::lock_guard<NonRecursiveMutex> guard(m_ControDataBufferMutex);

            size_t size;
            NN_RESULT_DO(ReadApplicationControlData(&size, m_ControlDataBuffer, sizeof(m_ControlDataBuffer), id, programIndex, path.string, language));
            NN_RESULT_DO(m_Cache.Put(ApplicationControlKey::Make(id, version, language), m_ControlDataBuffer, size));

            if (notifyParentalControl)
            {
#if !defined NN_BUILD_CONFIG_OS_WIN
                ApplicationControlDataAccessor accessor(m_ControlDataBuffer, size);
                pctl::AddToFreeCommunicationApplicationList(id, accessor.GetProperty());
#endif
            }
        }

        NN_RESULT_SUCCESS;
    }

    bool ApplicationControlDataManager::Has(ncm::ApplicationId id, uint32_t version) NN_NOEXCEPT
    {
        settings::LanguageCode language;
        settings::GetLanguageCode(&language);

        return m_Cache.Has(ApplicationControlKey::Make(id, version, language));
    }

    bool ApplicationControlDataManager::HasAny(ncm::ApplicationId id) NN_NOEXCEPT
    {
        return m_Cache.FindKeyById(id);
    }

    Result ApplicationControlDataManager::ListApplicationControlCacheEntryInfo(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::ApplicationControlCacheEntryInfo>& outList) NN_NOEXCEPT
    {
        return m_Cache.ListEntryInfo(outCount.GetPointer(), outList.GetData(), static_cast<int>(outList.GetLength()));
    }

    Result ApplicationControlDataManager::GetByKey(size_t* outValue, void* buffer, size_t bufferSize, ApplicationControlSource source, ApplicationControlKey cacheKey, util::optional<ncm::ContentMetaKey> storageKey, bool alongWithIcon) NN_NOEXCEPT
    {
        size_t size;
        switch (source)
        {
        case ApplicationControlSource::CacheOnly:
            {
                NN_RESULT_DO(GetFromCache(&size, buffer, bufferSize, cacheKey));
            }
            break;
        case ApplicationControlSource::Storage:
            {
                NN_RESULT_TRY(GetFromCache(&size, buffer, bufferSize, cacheKey))
                    NN_RESULT_CATCH(ResultApplicationControlDataNotFound)
                    {
                        NN_RESULT_THROW_UNLESS(storageKey, ResultApplicationControlDataNotFound());
                        uint8_t programIndex;
                        NN_RESULT_DO(m_Integrated->GetMinimumProgramIndex(&programIndex, *storageKey));
                        NN_RESULT_DO(GetFromStorage(&size, buffer, bufferSize, *storageKey, ApplicationControlKey::Make(cacheKey.GetId(), storageKey->version, cacheKey.GetLanguageCode()), programIndex, alongWithIcon));
                    }
                NN_RESULT_END_TRY
            }
            break;
        case ApplicationControlSource::StorageOnly:
            {
                NN_RESULT_THROW_UNLESS(storageKey, ResultApplicationControlDataNotFound());
                uint8_t programIndex;
                NN_RESULT_DO(m_Integrated->GetMinimumProgramIndex(&programIndex, *storageKey));
                NN_RESULT_DO(GetFromStorage(&size, buffer, bufferSize, *storageKey, ApplicationControlKey::Make(cacheKey.GetId(), storageKey->version, cacheKey.GetLanguageCode()), programIndex, alongWithIcon));
            }
            break;
        default: NN_RESULT_THROW(ResultUnsupportedStorage());
        }

        *outValue = size;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationControlDataManager::GetFromStorage(size_t* outValue, void* buffer, size_t bufferSize, const ncm::ContentMetaKey& key, const ApplicationControlKey& appKey, uint8_t programIndex, bool alongWithIcon) NN_NOEXCEPT
    {
        ncm::ContentId contentId;
        NN_RESULT_TRY(m_Integrated->GetContentId(&contentId, key, ncm::ContentType::Control, programIndex))
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound) { NN_RESULT_THROW(ResultApplicationControlDataNotFound()); }
        NN_RESULT_END_TRY

        size_t size;
        ncm::Path path;
        NN_RESULT_DO(m_Integrated->GetContentPath(&path, contentId));
        if (alongWithIcon)
        {
            NN_RESULT_DO(ReadApplicationControlData(&size, buffer, bufferSize, appKey.GetId(), programIndex, path.string, appKey.GetLanguageCode()));
            // TODO: id につきキャッシュが一つ、であればここで消してもいい
            NN_RESULT_DO(m_Cache.Put(appKey, buffer, size));
        }
        else
        {
            size = sizeof(ApplicationControlProperty);
            NN_RESULT_THROW_UNLESS(bufferSize >= size, ResultBufferNotEnough());
            NN_RESULT_DO(ReadApplicationControlProperty(reinterpret_cast<ApplicationControlProperty*>(buffer), appKey.GetId(), programIndex, path.string));

            // アイコンを取得してない状態では、キャッシュは更新しない(アイコンが消えるため)
        }

#if !defined NN_BUILD_CONFIG_OS_WIN
        ApplicationControlDataAccessor accessor(buffer, size);
        pctl::AddToFreeCommunicationApplicationList(appKey.GetId(), accessor.GetProperty());
#endif

        *outValue = size;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationControlDataManager::GetLatestFromCache(size_t* outValue, void* buffer, size_t bufferSize, ncm::ApplicationId id) NN_NOEXCEPT
    {
        NN_RESULT_TRY(m_Cache.GetLatest(outValue, buffer, bufferSize, id))
            NN_RESULT_CATCH_CONVERT(kvdb::ResultKeyNotFound, ResultApplicationControlDataNotFound())
            NN_RESULT_CATCH_CONVERT(kvdb::ResultBufferNotEnough, ResultBufferNotEnough())
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    Result ApplicationControlDataManager::GetFromCache(size_t* outValue, void* buffer, size_t bufferSize, const ApplicationControlKey& appKey) NN_NOEXCEPT
    {
        size_t size;
        NN_RESULT_TRY(m_Cache.Get(&size, buffer, bufferSize, appKey))
            NN_RESULT_CATCH_CONVERT(kvdb::ResultKeyNotFound, ResultApplicationControlDataNotFound())
            NN_RESULT_CATCH_CONVERT(kvdb::ResultBufferNotEnough, ResultBufferNotEnough())
        NN_RESULT_END_TRY

        *outValue = size;
        NN_RESULT_SUCCESS;
    }
}}}
