﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Config.h>
#include <nn/ns/ns_Result.h>
#include "ns_InstallUtil.h"
#include "ns_ApplicationApplyDeltaTask.h"
#include <nn/ns/detail/ns_Log.h>

namespace nn { namespace ns { namespace srv {
namespace
{
ApplicationApplyDeltaState ConvertState(nim::ApplyDownloadedDeltaProgressState state) NN_NOEXCEPT
{
    switch(state)
    {
    case nim::ApplyDownloadedDeltaProgressState::NotRunning:
        return ApplicationApplyDeltaState::WaitApply;

    case nim::ApplyDownloadedDeltaProgressState::Applying:
    case nim::ApplyDownloadedDeltaProgressState::InitialPrepared:
    case nim::ApplyDownloadedDeltaProgressState::Suspended:       // いずれ Applying になるので、Suspended は観測させない
        return ApplicationApplyDeltaState::Applying;

    case nim::ApplyDownloadedDeltaProgressState::AllApplied:
    case nim::ApplyDownloadedDeltaProgressState::Commited:
        return ApplicationApplyDeltaState::Applied;

    case nim::ApplyDownloadedDeltaProgressState::NotEnoughSpace:
        return ApplicationApplyDeltaState::NotEnoughSpace;

    default:
        return ApplicationApplyDeltaState::Fatal;
    }
}
bool IsRunning(nim::ApplyDownloadedDeltaProgressState state) NN_NOEXCEPT
{
    return state == nim::ApplyDownloadedDeltaProgressState::Applying;
}

TimeSpan CalculateRemainingTime(const nim::ApplyDeltaTaskInfo& info) NN_NOEXCEPT
{
    // タスクが実行中かつ、1sec 以上経過しているかつ、1% 以上処理が進んでいる場合に残り時間を出す
    if (IsRunning(info.progress.state) && info.throughput.elapsedTime.GetSeconds() >= 1 && info.throughput.processedSize >= 100)
    {
        return TimeSpan::FromMilliSeconds((info.progress.total - info.progress.applied) * info.throughput.elapsedTime.GetMilliSeconds() / info.throughput.processedSize);
    }
    else
    {
        return 0;
    }
}

}

    ApplicationApplyDeltaTask::ApplicationApplyDeltaTask(ncm::ApplicationId appId) NN_NOEXCEPT
    {
        m_IsValid = nim::ListApplicationApplyDeltaTask(&m_Id, 1, appId) > 0;
    }
    ApplicationApplyDeltaTask::ApplicationApplyDeltaTask(nim::ApplyDeltaTaskId id) NN_NOEXCEPT
    {
        // TODO: id の正しさは検証できていない
        m_Id = id;
        m_IsValid = true;
    }
    Result ApplicationApplyDeltaTask::StartApplyDeltaTask() NN_NOEXCEPT
    {
        nim::AsyncResult result;
        NN_RESULT_DO(nim::RequestApplyDeltaTaskRun(&result, m_Id));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationApplyDeltaTask::GetProgress(ApplicationApplyDeltaProgress* outValue) NN_NOEXCEPT
    {
        nim::ApplyDeltaTaskInfo info;
        NN_RESULT_DO(nim::GetApplyDeltaTaskInfo(&info, m_Id));

        ApplicationApplyDeltaProgress progress = {};
        progress.applied = info.progress.applied;
        progress.total = info.progress.total;
        progress.state = ConvertState(info.progress.state);
        progress.lastResult = util::Get(info.progress.lastResult);
        progress.isRunning = IsRunning(info.progress.state);
        progress.remainingTime = CalculateRemainingTime(info);

        *outValue = progress;

        NN_RESULT_SUCCESS;
    }

    Result ApplicationApplyDeltaTask::IsCommittable(bool* outValue) NN_NOEXCEPT
    {
        nim::ApplyDeltaTaskInfo info;
        NN_RESULT_DO(nim::GetApplyDeltaTaskInfo(&info, m_Id));

        if (info.progress.state == nim::ApplyDownloadedDeltaProgressState::AllApplied)
        {
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationApplyDeltaTask::Commit() NN_NOEXCEPT
    {
        NN_RESULT_DO(nim::CommitApplyDeltaTask(m_Id));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationApplyDeltaTask::ListKey(int* outCount, ncm::StorageContentMetaKey outList[], int count, int offset) NN_NOEXCEPT
    {
        NN_RESULT_DO(nim::ListApplyDeltaTaskContentMeta(outCount, outList, count, offset, m_Id));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationApplyDeltaTask::NeedsCleanup(bool* outValue) NN_NOEXCEPT
    {
        nim::ApplyDeltaTaskInfo info;
        NN_RESULT_DO(nim::GetApplyDeltaTaskInfo(&info, m_Id));
        if (info.progress.state == nim::ApplyDownloadedDeltaProgressState::Commited)
        {
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    Result ApplicationApplyDeltaTask::Destroy() NN_NOEXCEPT
    {
        NN_RESULT_DO(nim::DestroyApplyDeltaTask(m_Id));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationApplyDeltaTask::Resume() NN_NOEXCEPT
    {
        nim::ApplyDeltaTaskInfo info;
        NN_RESULT_DO(nim::GetApplyDeltaTaskInfo(&info, m_Id));
        NN_RESULT_THROW_UNLESS(info.progress.state == nim::ApplyDownloadedDeltaProgressState::NotEnoughSpace, ResultApplyDeltaTaskNotResumable());
        NN_RESULT_DO(nim::ClearNotEnoughSpaceStateOfApplyDeltaTask(m_Id));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationApplyDeltaTask::CalculateRequiredSize(int64_t* outValue) NN_NOEXCEPT
    {
        NN_RESULT_DO(nim::CalculateApplyDeltaTaskRequiredSize(outValue, m_Id));
        NN_RESULT_SUCCESS;
    }
    Result ApplicationApplyDeltaTask::GetRequiredStorage(ncm::StorageId* outValue) NN_NOEXCEPT
    {
        NN_RESULT_DO(nim::GetApplyDeltaTaskRequiredStorage(outValue, m_Id));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationApplyDeltaTask::NeedsProcess(bool* outValue) NN_NOEXCEPT
    {
        nim::ApplyDeltaTaskInfo info;
        NN_RESULT_DO(nim::GetApplyDeltaTaskInfo(&info, m_Id));

        if (info.progress.state == nim::ApplyDownloadedDeltaProgressState::InitialPrepared ||
            info.progress.state == nim::ApplyDownloadedDeltaProgressState::Applying ||
            info.progress.state == nim::ApplyDownloadedDeltaProgressState::Suspended)
        {
            *outValue = true;
        }
        else
        {
            *outValue = false;
        }
        NN_RESULT_SUCCESS;
    }
    Result ApplicationApplyDeltaTask::NeedsRequestToRun(bool* outValue) NN_NOEXCEPT
    {
        nim::ApplyDeltaTaskInfo info;
        NN_RESULT_DO(nim::GetApplyDeltaTaskInfo(&info, m_Id));

        if (info.progress.state == nim::ApplyDownloadedDeltaProgressState::InitialPrepared ||
            info.progress.state == nim::ApplyDownloadedDeltaProgressState::Suspended)
        {
            *outValue = true;
        }
        else
        {
            *outValue = false;
        }
        NN_RESULT_SUCCESS;
    }
    Result ApplicationApplyDeltaTask::CalculateOccupiedSize(int64_t* outValue, ncm::StorageId storageId) NN_NOEXCEPT
    {
        NN_RESULT_DO(nim::CalculateApplyDeltaTaskOccupiedSize(outValue, m_Id, storageId));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationApplyDeltaTask::ResumeAll() NN_NOEXCEPT
    {
        // 16byte * 64 = 1KiByte
        nim::ApplyDeltaTaskId ids[nim::MaxApplyDeltaTaskCount];
        auto count = nim::ListApplyDeltaTask(ids, nim::MaxApplyDeltaTaskCount);

        for (int i = 0; i < count; ++i)
        {
            nim::ApplyDeltaTaskInfo info;
            NN_RESULT_DO(nim::GetApplyDeltaTaskInfo(&info, ids[i]));
            if (info.progress.state == nim::ApplyDownloadedDeltaProgressState::NotEnoughSpace)
            {
                NN_RESULT_DO(nim::ClearNotEnoughSpaceStateOfApplyDeltaTask(ids[i]));
            }
        }

        NN_RESULT_SUCCESS;
    }

}}}
