﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/util/util_ScopeExit.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/ncm/ncm_ProgramLocation.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_Shell.h>
#include <nn/ns/srv/ns_AppletLauncher.h>
#include <nn/ns/detail/ns_Log.h>

namespace nn { namespace ns { namespace srv {
    namespace {
        ncm::ProgramId GetProgramIdFromSettingsItemValue(const char* name, const char* key) NN_NOEXCEPT
        {
            char idStr[32];
            char* endPtr;
            settings::fwdbg::GetSettingsItemValue(idStr, sizeof(idStr), name, key);
            ncm::ProgramId programId = { std::strtoull(idStr, &endPtr, 16) };
            NN_ABORT_UNLESS(*endPtr == '\0');

            return programId;
        }

        Result GetSystemAppletProgramId(ncm::ProgramId* outValue, pm::BootMode bootMode) NN_NOEXCEPT
        {
            if (bootMode == pm::BootMode_Maintenance)
            {
                ncm::ProgramId id = { NN_NS_PROGRAM_ID_OF_MAINTENANCEMENU };
                *outValue = id;
                NN_RESULT_SUCCESS;
            }

            *outValue = GetProgramIdFromSettingsItemValue("ns.applet", "system_applet_id");
            NN_RESULT_SUCCESS;
        }

        Result GetOverlayAppletProgramId(ncm::ProgramId* outValue) NN_NOEXCEPT
        {
            *outValue = GetProgramIdFromSettingsItemValue("ns.applet", "overlay_applet_id");
            NN_RESULT_SUCCESS;
        }
    }

    Result AppletLauncher::GetApplicationShellEvent(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
    {
        *outValue = sf::NativeHandle(m_Observer.GetEvent().GetReadableHandle(), false);
        NN_RESULT_SUCCESS;
    }

    Result AppletLauncher::PopApplicationShellEventInfo(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::ApplicationShellEventInfo>& outList) NN_NOEXCEPT
    {
        *outCount = m_Observer.PopApplicationShellEventInfo(outList.GetData(), static_cast<int>(outList.GetLength()));
        NN_RESULT_SUCCESS;
    }

    Result AppletLauncher::LaunchLibraryApplet(sf::Out<os::ProcessId> outValue, ncm::SystemProgramId id) NN_NOEXCEPT
    {
        os::ProcessId processId;
        NN_RESULT_DO(LaunchProgram(&processId, ncm::MakeProgramLocation(ncm::StorageId::BuildInSystem, id), LaunchProgramFlags_NotifyDebug | LaunchProgramFlags_NotifyStart | LaunchProgramFlags_NotifyExit));

        *outValue = processId;
        NN_RESULT_SUCCESS;
    }

    Result AppletLauncher::TerminateLibraryApplet(os::ProcessId pid) NN_NOEXCEPT
    {
        return srv::TerminateProcess(pid);
    }

    Result AppletLauncher::LaunchSystemApplet(sf::Out<os::ProcessId> outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_IsLaunchSystemAppletEnabled, ResultLaunchSystemAppletDisabled());

        ncm::ProgramId programId;
        NN_RESULT_DO(GetSystemAppletProgramId(&programId, m_BootMode));

        os::ProcessId processId;
        NN_RESULT_DO(LaunchProgram(&processId, ncm::MakeProgramLocation(ncm::StorageId::BuildInSystem, programId), LaunchProgramFlags_NotifyDebug | LaunchProgramFlags_NotifyStart | LaunchProgramFlags_NotifyExit));

        *outValue = processId;
        NN_RESULT_SUCCESS;
    }

    Result AppletLauncher::TerminateSystemApplet(os::ProcessId pid) NN_NOEXCEPT
    {
        return srv::TerminateProcess(pid);
    }

    Result AppletLauncher::LaunchOverlayApplet(sf::Out<os::ProcessId> outValue) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_IsLaunchOverlayAppletEnabled, ResultLaunchOverlayAppletDisabled());

        ncm::ProgramId programId;
        NN_RESULT_DO(GetOverlayAppletProgramId(&programId));

        os::ProcessId processId;
        NN_RESULT_DO(LaunchProgram(&processId, ncm::MakeProgramLocation(ncm::StorageId::BuildInSystem, programId), LaunchProgramFlags_NotifyDebug | LaunchProgramFlags_NotifyStart | LaunchProgramFlags_NotifyExit));

        *outValue = processId;
        NN_RESULT_SUCCESS;
    }

    Result AppletLauncher::TerminateOverlayApplet(os::ProcessId pid) NN_NOEXCEPT
    {
        return srv::TerminateProcess(pid);
    }

    void AppletLauncher::RequestReboot() NN_NOEXCEPT
    {
        m_Observer.RequestReboot();
    }

    void AppletLauncher::NotifyStartCardUpdate() NN_NOEXCEPT
    {
        m_Observer.NotifyStartCardUpdate();
    }

    void AppletLauncher::RequestShowCardUpdateProcessing() NN_NOEXCEPT
    {
        m_Observer.RequestShowCardUpdateProcessing();
    }

    void AppletLauncher::NotifyEndCardUpdate() NN_NOEXCEPT
    {
        m_Observer.NotifyEndCardUpdate();
    }
}}}
