﻿<#
    Copyright (C)2014 Nintendo Co., Ltd.  All rights reserved.

    These coded instructions, statements, and computer programs contain
    proprietary information of Nintendo of America Inc. and/or Nintendo
    Company Ltd., and are protected by Federal copyright law.  They may
    not be disclosed to third parties or copied or duplicated in any form,
    in whole or in part, without the prior written consent of Nintendo.
#>

<#
    .SYNOPSIS
        Generate headers for NSPR builds.

    .DESCRIPTION
        Determine the version of NSPR we are building, get the Siglo
        specific platform configuration header and autogenerate the
        version header.

    .INPUTS
        The directory where NSPR is being built.
        The directory containing the NSPR sources.
        The base name of the NSPR library output.
#>
Param(
    [Parameter(Mandatory=$true)]
    [string]$BuildDir,

    [Parameter(Mandatory=$true)]
    [string]$SourceDir,

    [Parameter(Mandatory=$true)]
    [string]$LibName
)

#trap [Exception]
#{
#    Write-ErrorRecord $_
#    exit 1
#}

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

$SigloCpuCfg = "_nnsdk.cfg"
$PrIncMdPath = Join-Path -Path $SourceDir -ChildPath "pr\include\md"
$SigloCpuCfgPath = Join-Path -Path $PrIncMdPath -ChildPath $SigloCpuCfg
$SiglogCpuCfgDestPath = Join-Path -Path $BuildDir -ChildPath "prcpucfg.h"

# Copy the Siglo configuration header to be in the build directory so it
# will get picked up by the build tools.
Copy-Item -Path $SigloCpuCfgPath -Destination $SiglogCpuCfgDestPath


# Auto-generate the _pr_bld.h file which has the build date, time
# and library name embedded in it.
# Note that the easiest way to get time as needed by NSPR is to get
# the current time in seconds since 12:00AM Jan. 1, 1970 (UNIX epoch)
# and multiply to get it to usecs.  This way the C compiler can handle
# creating the constant rather than us parsing and converting in Powershell.
$curDate = Get-Date -UFormat "%Y-%m-%d %T"
$now = ([double]::Parse((Get-Date -UFormat "%s"))).ToString("0",[System.Globalization.CultureInfo]::InvariantCulture)

$buildHdr = Join-Path -Path $BuildDir -ChildPath "_pr_bld.h"
$buildDateDef = '#define _BUILD_STRING  "{0}"' -f $curDate
$buildTimeDef = '#define _BUILD_TIME    ({0}LL * 1000000)' -f $now
$buildProdDef = '#define _PRODUCTION    "{0}"' -f $Libname

$buildDateDef | Out-File -FilePath $buildHdr -Encoding ascii
$buildTimeDef | Out-File -FilePath $buildHdr -append -Encoding ascii
$buildProdDef | Out-File -FilePath $buildHdr -append -Encoding ascii

# The Portable Library components are just included in the NSPR library
# for Siglo insteal of separate plc and plds libraries.  So generate
# _pl_bld.h with the same content.
$plBuildHdr = Join-Path -Path $BuildDir -ChildPath "_pl_bld.h"
$buildDateDef | Out-File -FilePath $plBuildHdr -Encoding ascii
$buildTimeDef | Out-File -FilePath $plBuildHdr -append -Encoding ascii
$buildProdDef | Out-File -FilePath $plBuildHdr -append -Encoding ascii
