﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <nn/sf/sf_ProxyObjectAllocator.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerInitializeApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_EcAppletApi.h>
#include <nn/ns/ns_WebAppletApi.h>
#include <nn/ns/ns_VulnerabilityApi.h>
#include <nn/ns/detail/ns_ISystemUpdateInterface.sfdl.h>
#include <nn/ns/detail/ns_IVulnerabilityManagerInterface.sfdl.h>
#include <nn/ns/detail/ns_Log.h>

#include "ns_ServiceObjectHolder.h"
#include "ns_Initialize.h"

#include <nn/ns/srv/ns_ApplicationManagerInterfaceServer.h>
#include <nn/ns/srv/ns_SystemUpdateInterfaceServer.h>
#include <nn/ns/srv/ns_ServiceGetterInterfaceServer.h>
#include <nn/ns/srv/ns_SystemReportManager.h>

namespace nn {
    namespace ns {
        namespace
        {
            template<typename ImplType>
            struct Storage {
                static const size_t ImplSize = sizeof(ImplType);
                static const size_t ImplAlign = NN_ALIGNOF(ImplType);
                typedef typename std::aligned_storage<ImplSize, ImplAlign>::type StorageType;
            };

            Storage<srv::ApplicationManagerInterfaceServer>::StorageType g_AppManagerStorage;
            Storage<srv::SystemUpdateInterfaceServer>::StorageType g_SystemUpdateStorage;
            Storage<srv::ServiceGetterInterfaceServer>::StorageType g_ServiceGetterInterfaceStorage;

            class ApplicationManagerInstance
            {
            public:
                ApplicationManagerInstance() : m_pAppManager(nullptr) {}
                Result Initialize(void* storage, srv::SystemReportManager* systemReportManager) NN_NOEXCEPT
                {
                    m_pAppManager = new (storage) srv::ApplicationManagerInterfaceServer();
                    NN_RESULT_DO(m_pAppManager->Initialize(nullptr, systemReportManager));
                    NN_RESULT_SUCCESS;
                }
                void Finalize() NN_NOEXCEPT
                {
                    m_pAppManager->~ApplicationManagerInterfaceServer();
                    m_pAppManager = nullptr;
                }
                srv::ApplicationManagerInterfaceServer* Get() NN_NOEXCEPT
                {
                    return m_pAppManager;
                }
            private:
                srv::ApplicationManagerInterfaceServer* m_pAppManager;
            };

            ApplicationManagerInstance g_AppManager;

            srv::SystemReportManager g_SystemReportManager;

            Result CreateSystemUpdateInterface(sf::SharedPointer<detail::ISystemUpdateInterface>*outValue, void* storage) NN_NOEXCEPT
            {
                auto deleter = [](srv::SystemUpdateInterfaceServer* p)
                {
                    p->~SystemUpdateInterfaceServer();
                };
                auto p = std::unique_ptr<srv::SystemUpdateInterfaceServer, decltype(deleter)>(new (storage) srv::SystemUpdateInterfaceServer(), deleter);
                NN_RESULT_DO(p->Initialize(&g_SystemReportManager));
                *outValue = sf::CreateSharedObject<detail::ISystemUpdateInterface>(std::move(p));

                NN_RESULT_SUCCESS;
            }

            Result CreateServiceGetterInterface(sf::SharedPointer<detail::IServiceGetterInterface>*outValue, void* storage, void* applicationManagerImpl) NN_NOEXCEPT
            {
                auto deleter = [](srv::ServiceGetterInterfaceServer* p)
                {
                    p->~ServiceGetterInterfaceServer();
                };
                const Bit32 FullCapability = 0xffffffff;
                auto p = std::unique_ptr<srv::ServiceGetterInterfaceServer, decltype(deleter)>(new (storage) srv::ServiceGetterInterfaceServer(reinterpret_cast<srv::ApplicationManagerInterfaceServer*>(applicationManagerImpl), FullCapability), deleter);
                *outValue = sf::CreateSharedObject<detail::IServiceGetterInterface>(std::move(p));

                NN_RESULT_SUCCESS;
            }

        } // namespace


        void InitializeServiceObjectsImpl(bool abortIfApplicationManagerInitializeFailed, ServiceObjectHolders* holders) NN_NOEXCEPT
        {
            NN_UNUSED(abortIfApplicationManagerInitializeFailed);

            g_SystemReportManager.Initialize();

            {
                sf::SharedPointer<detail::ISystemUpdateInterface> p;
                NN_ABORT_UNLESS_RESULT_SUCCESS(CreateSystemUpdateInterface(&p, &g_SystemUpdateStorage));
                holders->systemUpdateInterfaceHolder.InitializeDirectly(p);
            }

            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(g_AppManager.Initialize(&g_AppManagerStorage, &g_SystemReportManager));
            }

            {
                holders->vulnerabilityManagerInterfaceHolder.InitializeWithImplObject(g_AppManager.Get(), sf::GetCurrentMemoryResource());
            }

            {
                sf::SharedPointer<detail::IServiceGetterInterface> p;
                NN_ABORT_UNLESS_RESULT_SUCCESS(CreateServiceGetterInterface(&p, &g_ServiceGetterInterfaceStorage, g_AppManager.Get()));
                holders->serviceGetterInterfaceHolder.InitializeDirectly(p);
            }
        }

        void FinalizeServiceObjectsImpl(ServiceObjectHolders* holders) NN_NOEXCEPT
        {
            holders->serviceGetterInterfaceHolder.Finalize();
            holders->vulnerabilityManagerInterfaceHolder.Finalize();
            holders->systemUpdateInterfaceHolder.Finalize();
            g_AppManager.Finalize();
        }

        void InitializeMemoryResourceImpl() NN_NOEXCEPT
        {
            // デフォルトの MemoryResource 使うので初期化不要
        }

        void FinalizeMemoryResourceImpl() NN_NOEXCEPT
        {
            // デフォルトの MemoryResource 使うので終了処理不要
        }
    }
}
