﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_TransferMemory.h>

#include <nn/ns/ns_ApplicationVerificationApi.h>

#include "ns_Initialize.h"

namespace nn { namespace ns {
    namespace
    {
        Result RequestVerifyApplicationImpl(AsyncVerifyApplicationResult* outValue, ncm::ApplicationId id, VerifyContentFlag flags, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
        {
            uintptr_t workBufferAddress = reinterpret_cast<uintptr_t>(workBuffer);
            NN_UNUSED(workBufferAddress);
            NN_SDK_REQUIRES_NOT_NULL(workBuffer);
            NN_SDK_REQUIRES_ALIGNED(workBufferAddress, 0x1000u);
            NN_SDK_REQUIRES_GREATER(workBufferSize, 0u);
            NN_SDK_REQUIRES_ALIGNED(workBufferSize, 0x1000u);
            NN_SDK_REQUIRES_GREATER(workBufferAddress + workBufferSize, workBufferAddress);

            sf::NativeHandle nativeHandle;
            sf::SharedPointer<detail::IProgressAsyncResult> sp;

            os::TransferMemory transfer(workBuffer, workBufferSize, os::MemoryPermission_None);
            auto transferHandle = transfer.Detach();
            NN_RESULT_DO(GetApplicationManagerInterface()->RequestVerifyApplication(&nativeHandle, &sp, id, flags._storage[0], sf::NativeHandle(transferHandle, true), workBufferSize));
            outValue->Initialize(sp, nativeHandle);
            nativeHandle.Detach();

            NN_RESULT_SUCCESS;
        }
    }

    Result RequestVerifyApplication(AsyncVerifyApplicationResult* outValue, ncm::ApplicationId id, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        NN_RESULT_DO(RequestVerifyApplicationImpl(outValue, id, VerifyContentFlag_All, workBuffer, workBufferSize));

        NN_RESULT_SUCCESS;
    }

    Result RequestVerifyApplication(AsyncVerifyApplicationResult* outValue, ncm::ApplicationId id, VerifyContentFlag flags, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        NN_RESULT_DO(RequestVerifyApplicationImpl(outValue, id, flags, workBuffer, workBufferSize));

        NN_RESULT_SUCCESS;
    }

    Result CorruptApplicationForDebug(ncm::ApplicationId id, CorruptContentFlag flags, ncm::StorageId storageId) NN_NOEXCEPT
    {
        NN_RESULT_DO(GetApplicationManagerInterface()->CorruptApplicationForDebug(id, flags._storage[0], storageId));

        NN_RESULT_SUCCESS;
    }

    Result CorruptContentForDebug(const ncm::ContentMetaKey& key, ncm::StorageId storageId) NN_NOEXCEPT
    {
        NN_RESULT_DO(GetApplicationManagerInterface()->CorruptContentForDebug(key, storageId));

        NN_RESULT_SUCCESS;
    }

    Result RequestVerifyAddOnContentsRights(AsyncVerifyAddOnContentsResult* outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
        sf::NativeHandle nativeHandle;
        sf::SharedPointer<detail::IProgressAsyncResult> sp;

        NN_RESULT_DO(GetApplicationManagerInterface()->RequestVerifyAddOnContentsRights(&nativeHandle, &sp, id));
        outValue->Initialize(sp, nativeHandle);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }
}}
