﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_Mount.h>
#include <nn/ncm/ncm_InstallTaskData.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_TaskDataStore.h>

namespace nn { namespace nim { namespace srv {

    util::optional<util::Uuid> IdIterator::Next() NN_NOEXCEPT
    {
        if (m_Count == m_MaxCount)
        {
            return util::nullopt;
        }

        auto uuid = m_List[m_Count];
        m_Count++;
        return uuid;
    }

    Result TaskDataStore::Initialize(const char* mountName, fs::SystemSaveDataId id, int64_t saveDataSize, int64_t saveDataJournalSize, int saveDataFlags) NN_NOEXCEPT
    {
        m_Id = id;
        nn::fs::DisableAutoSaveDataCreation();
        NN_RESULT_TRY(fs::MountSystemSaveData(mountName, m_Id))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(nn::fs::CreateSystemSaveData(m_Id, saveDataSize, saveDataJournalSize, saveDataFlags));
                NN_RESULT_DO(nn::fs::MountSystemSaveData(mountName, m_Id));
            }
        NN_RESULT_END_TRY;

        m_MoundName.emplace();
        m_MoundName->Assign(mountName);

        NN_RESULT_TRY(ReadList())
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_DO(fs::CreateFile(MakeTaskListFilePath(), sizeof(m_List)));
                ClearList();
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    TaskDataStore::~TaskDataStore() NN_NOEXCEPT
    {
        if (m_MoundName)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::CommitSaveData(*m_MoundName));
            fs::Unmount(*m_MoundName);
        }
    }

    Result TaskDataStore::Create(const util::Uuid& uuid, const void* meta, size_t metaSize) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(HasSpace(), ResultOutOfMaxTask());

        NN_RESULT_DO(fs::CreateDirectory(MakeTaskDirectoryPath(uuid)));
        bool success = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                fs::DeleteDirectoryRecursively(MakeTaskDirectoryPath(uuid));
            }
        };

        NN_RESULT_DO(ncm::FileInstallTaskData::Create(MakeTaskDataFilePath(uuid), 2048));
        NN_RESULT_DO(fs::CreateFile(MakeTaskMetaFilePath(uuid), metaSize));

        if (meta)
        {
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, MakeTaskMetaFilePath(uuid), fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
            NN_RESULT_DO(fs::WriteFile(file, 0, meta, metaSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
        }

        NN_RESULT_DO(AddToList(uuid));

        success = true;
        NN_RESULT_SUCCESS;
    }

    Result TaskDataStore::Destroy(const util::Uuid& uuid) NN_NOEXCEPT
    {
        NN_RESULT_DO(RemoveFromList(uuid));
        NN_RESULT_DO(fs::DeleteDirectoryRecursively(MakeTaskDirectoryPath(uuid)));
        NN_RESULT_SUCCESS;
    }

    Result TaskDataStore::GetMetaSize(size_t* outValue, const util::Uuid& uuid) NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, MakeTaskMetaFilePath(uuid), fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, file));

        *outValue = static_cast<size_t>(fileSize);
        NN_RESULT_SUCCESS;
    }

    Result TaskDataStore::GetMeta(size_t* outValue, const util::Uuid& uuid, void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, MakeTaskMetaFilePath(uuid), fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

        int64_t fileSize;
        NN_RESULT_DO(fs::GetFileSize(&fileSize, file));

        auto metaSize = static_cast<size_t>(fileSize);
        NN_RESULT_THROW_UNLESS(metaSize <= bufferSize, ResultBufferNotEnough());
        NN_RESULT_DO(fs::ReadFile(file, 0, buffer, metaSize));

        *outValue = metaSize;
        NN_RESULT_SUCCESS;
    }

    Result TaskDataStore::GetMetaFilePath(Path* outValue, const util::Uuid& uuid) NN_NOEXCEPT
    {
        *outValue = MakeTaskMetaFilePath(uuid);
        NN_RESULT_SUCCESS;
    }

    Result TaskDataStore::GetDataFilePath(Path* outValue, const util::Uuid& uuid) NN_NOEXCEPT
    {
        *outValue = MakeTaskDataFilePath(uuid);
        NN_RESULT_SUCCESS;
    }

    IdIterator TaskDataStore::CreateIdIterator() NN_NOEXCEPT
    {
        IdIterator iter;
        iter.m_List = m_List;
        iter.m_MaxCount = Count();
        return iter;
    }

    Result TaskDataStore::Commit() NN_NOEXCEPT
    {
        return fs::CommitSaveData(*m_MoundName);
    }

    Result TaskDataStore::DestroySaveData() NN_NOEXCEPT
    {
        if (m_MoundName)
        {
            fs::Unmount(*m_MoundName);
            m_MoundName = util::nullopt;

            // m_MoundName が設定されている時点で、m_Id も設定済み
            NN_RESULT_DO(DeleteSaveData(m_Id));
        }
        NN_RESULT_SUCCESS;
    }

    Result TaskDataStore::DeleteSaveData(fs::SystemSaveDataId id) NN_NOEXCEPT
    {
        NN_RESULT_TRY(fs::DeleteSaveData(id))
            NN_RESULT_CATCH(fs::ResultTargetNotFound) {}
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }

    TaskDataStore::Path TaskDataStore::MakeRootDirecotryPath() NN_NOEXCEPT
    {
        Path path;
        path.AssignFormat("%s:/", m_MoundName->Get());

        return path;
    }

    TaskDataStore::Path TaskDataStore::MakeTaskDirectoryPath(const util::Uuid& uuid) NN_NOEXCEPT
    {
        char uuidString[util::Uuid::StringSize];
        uuid.ToString(uuidString, sizeof(uuidString));

        Path path;
        path.AssignFormat("%s:/%s", m_MoundName->Get(), uuidString);

        return path;
    }

    TaskDataStore::Path TaskDataStore::MakeTaskDataFilePath(const util::Uuid& uuid) NN_NOEXCEPT
    {
        char uuidString[util::Uuid::StringSize];
        uuid.ToString(uuidString, sizeof(uuidString));

        Path path;
        path.AssignFormat("%s:/%s/data", m_MoundName->Get(), uuidString);

        return path;
    }

    TaskDataStore::Path TaskDataStore::MakeTaskMetaFilePath(const util::Uuid& uuid) NN_NOEXCEPT
    {
        char uuidString[util::Uuid::StringSize];
        uuid.ToString(uuidString, sizeof(uuidString));

        Path path;
        path.AssignFormat("%s:/%s/meta", m_MoundName->Get(), uuidString);

        return path;
    }

    TaskDataStore::Path TaskDataStore::MakeTaskListFilePath() NN_NOEXCEPT
    {
        Path path;
        path.AssignFormat("%s:/list", m_MoundName->Get());

        return path;
    }

    void TaskDataStore::ClearList() NN_NOEXCEPT
    {
        for (auto& id : m_List)
        {
            id = util::InvalidUuid;
        }
    }

    Result TaskDataStore::ReadList() NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, MakeTaskListFilePath(), fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
        NN_RESULT_DO(fs::ReadFile(file, 0, m_List, sizeof(m_List)));
        NN_RESULT_SUCCESS;
    }

    Result TaskDataStore::WriteList() NN_NOEXCEPT
    {
        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, MakeTaskListFilePath(), fs::OpenMode_Write | fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
        NN_RESULT_DO(fs::WriteFile(file, 0, m_List, sizeof(m_List), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
        NN_RESULT_SUCCESS;
    }

    Result TaskDataStore::AddToList(const util::Uuid& uuid) NN_NOEXCEPT
    {
        for (auto& id : m_List)
        {
            if (id == util::InvalidUuid)
            {
                id = uuid;
                break;
            }
        }

        NN_RESULT_DO(WriteList());
        NN_RESULT_SUCCESS;
    }

    Result TaskDataStore::RemoveFromList(const util::Uuid& uuid) NN_NOEXCEPT
    {
        for (int i = 0; i < MaxTaskCount; i++)
        {
            if (m_List[i] == uuid)
            {
                // i == MaxTaskCount - 1 の時は、memmove に size 0 が渡されるので
                // 不正なアドレス（&m_List[i + i]）は参照されない
                std::memmove(&m_List[i], &m_List[i + 1], (MaxTaskCount - (i + 1)) * sizeof(util::Uuid));
                m_List[MaxTaskCount - 1] = util::InvalidUuid;

                break;
            }
        }

        NN_RESULT_DO(WriteList());
        NN_RESULT_SUCCESS;
    }

    int TaskDataStore::Count() NN_NOEXCEPT
    {
        int count = 0;
        for (auto& id : m_List)
        {
            if (id == util::InvalidUuid)
            {
                break;
            }

            count++;
        }

        return count;
    }

    bool TaskDataStore::HasSpace() NN_NOEXCEPT
    {
        // TODO: m_List は std::array を使うべき
        return std::any_of(m_List, &m_List[MaxTaskCount],
            [](const util::Uuid& uuid) { return uuid == util::InvalidUuid; });
    }

}}}
