﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include "nim_Json.h"
#include "nim_NetworkInstallUtil.h"

#include <mutex>

namespace nn { namespace nim { namespace srv {
    namespace {

        ncm::ContentMetaType GetContentMetaType(const char* value, size_t valueSize) NN_NOEXCEPT
        {
            const char Application[] = "Application";
            if (util::Strncmp(value, Application, static_cast<int>(valueSize)) == 0 && sizeof(Application) - 1 == valueSize)
            {
                return ncm::ContentMetaType::Application;
            }

            const char Patch[] = "Patch";
            if (util::Strncmp(value, Patch, static_cast<int>(valueSize)) == 0 && sizeof(Patch) - 1 == valueSize)
            {
                return ncm::ContentMetaType::Patch;
            }

            const char AddOnContent[] = "AddOnContent";
            if (util::Strncmp(value, AddOnContent, static_cast<int>(valueSize)) == 0 && sizeof(AddOnContent) - 1 == valueSize)
            {
                return ncm::ContentMetaType::AddOnContent;
            }

            return ncm::ContentMetaType::Unknown;
        }

        Result ParseLatestVersionsResponse(ncm::ContentMetaKey* out, int* outCount, int maxCount, char* readBuffer) NN_NOEXCEPT
        {
            nne::rapidjson::Document document;
            NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseLatestVersionParseError());

            NN_RESULT_THROW_UNLESS(document.IsArray(), ResultUnexpectedResponseLatestVersionParseError());

            auto entryCount = static_cast<int>(document.Size());
            NN_RESULT_THROW_UNLESS(entryCount <= maxCount, ResultUnexpectedResponseLatestVersionTooManyEntries());

            for (int i = 0; i < entryCount; ++i)
            {
                auto& entry = document[i];

                auto titleTypeObj = entry.FindMember("title_type");
                NN_RESULT_THROW_UNLESS(titleTypeObj != entry.MemberEnd() && titleTypeObj->value.IsString(), ResultUnexpectedResponseLatestVersionParseError());
                auto titleType = GetContentMetaType(titleTypeObj->value.GetString(), titleTypeObj->value.GetStringLength());
                if (titleType == ncm::ContentMetaType::Unknown) { continue; }

                auto versionObj = entry.FindMember("version");
                NN_RESULT_THROW_UNLESS(versionObj != entry.MemberEnd() && versionObj->value.IsUint(), ResultUnexpectedResponseLatestVersionParseError());
                auto version = versionObj->value.GetUint();

                auto idObj = entry.FindMember("title_id");
                NN_RESULT_THROW_UNLESS(idObj != entry.MemberEnd() && idObj->value.IsString(), ResultUnexpectedResponseLatestVersionParseError());
                auto id = std::strtoll(idObj->value.GetString(), nullptr, 16);

                out[i] = ncm::ContentMetaKey::Make(id, version, titleType);
            }

            *outCount = entryCount;
            NN_RESULT_SUCCESS;
        }

        Result ParseLatestVersionResponse(uint32_t* outValue, char* readBuffer) NN_NOEXCEPT
        {
            nne::rapidjson::Document document;
            NN_RESULT_THROW_UNLESS(!document.ParseInsitu(readBuffer).HasParseError(), ResultUnexpectedResponseLatestVersionParseError());

            auto versionObj = document.FindMember("version");
            NN_RESULT_THROW_UNLESS(versionObj != document.MemberEnd() && versionObj->value.IsUint(), ResultUnexpectedResponseLatestVersionMemberNotFound());
            auto version = versionObj->value.GetUint();

            *outValue = version;
            NN_RESULT_SUCCESS;
        }

        char g_LatestContentMetaKeyBuffer[160 * 1024]; // Json で 1 タイトルあたり 80バイト程度 * 2048 でとりあえず 160KiB
        nn::os::Mutex g_LatestContentMetaKeyBufferLock(false);
    }

    Result GetLatestContentMetaKey(ncm::ContentMetaKey* out, int* outCount, int maxCount, HttpConnection* connection, Bit64 id) NN_NOEXCEPT
    {
        Url url;
        MakeLatestVersionArrayUrl(&url, id);

        std::lock_guard<nn::os::Mutex> lock(g_LatestContentMetaKeyBufferLock);

        size_t readSize = 0;
        NN_RESULT_TRY(connection->Get(url,
            [&readSize](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(g_LatestContentMetaKeyBuffer), ResultUnexpectedResponseLatestVersionTooLong());
                std::memcpy(&g_LatestContentMetaKeyBuffer[readSize], buffer, size);

                readSize += size;

                NN_RESULT_SUCCESS;
            }))
            NN_RESULT_CATCH(ResultHttpStatus404NotFound)
            {
                NN_RESULT_THROW(ResultLatestVersionNotFound());
            }
        NN_RESULT_END_TRY;
        g_LatestContentMetaKeyBuffer[readSize] = '\0';
        NN_DETAIL_NIM_TRACE("json: %s\n", g_LatestContentMetaKeyBuffer);

        NN_RESULT_DO(ParseLatestVersionsResponse(out, outCount, maxCount, g_LatestContentMetaKeyBuffer));

        NN_RESULT_SUCCESS;
    }

    Result GetLatestVersion(util::optional<uint32_t>* outValue, HttpConnection* connection, Bit64 id) NN_NOEXCEPT
    {
        Url url;
        MakeLatestVersionUrl(&url, id);

        char readBuffer[1024];
        size_t readSize = 0;
        NN_RESULT_TRY(connection->Get(url,
            [&readSize, &readBuffer](const void* buffer, size_t size) -> Result
        {
            NN_RESULT_THROW_UNLESS(readSize + size < sizeof(readBuffer), ResultUnexpectedResponseLatestVersionTooLong());
            std::memcpy(&readBuffer[readSize], buffer, size);

            readSize += size;

            NN_RESULT_SUCCESS;
        }))
            NN_RESULT_CATCH(ResultHttpStatus404NotFound)
            {
                *outValue = util::nullopt;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;
        readBuffer[readSize] = '\0';
        NN_DETAIL_NIM_TRACE("json: %s\n", readBuffer);

        uint32_t latestVersion;
        NN_RESULT_DO(ParseLatestVersionResponse(&latestVersion, readBuffer));

        *outValue = latestVersion;
        NN_RESULT_SUCCESS;
    }

    Result GetAndInstallCommonTicket(HttpConnection* connection, const es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        Url url;
        MakeDownloadCommonTicketUrl(&url, rightsId);

        char readBuffer[4096];
        size_t readSize = 0;

        NN_RESULT_TRY(connection->Get(url,
            [&readSize, &readBuffer](const void* buffer, size_t size) -> Result
            {
                NN_RESULT_THROW_UNLESS(readSize + size < sizeof(readBuffer), ResultUnexpectedResponseCommonTicketTooLong());
                std::memcpy(&readBuffer[readSize], buffer, size);

                readSize += size;

                NN_RESULT_SUCCESS;
            }))
            NN_RESULT_CATCH(ResultHttpStatus404NotFound)
            {
                NN_RESULT_THROW(ResultCommonTicketNotFound());
            }
        NN_RESULT_END_TRY;

        // 204:NoContent のステータスコードは ResultCommonTicketNotFound を返す
        if (connection->GetLastStatusCode() == 204)
        {
            NN_RESULT_THROW(ResultCommonTicketNotFound());
        }

        // チケットをインポートする
        NN_RESULT_DO(nn::es::ImportTicketCertificateSet(readBuffer, readSize));
#else
        NN_UNUSED(connection);
        NN_UNUSED(rightsId);

#endif

        NN_RESULT_SUCCESS;
    }

}}}
