﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/nim/srv/nim_LocalCommunicationSendSystemUpdateTask.h>
#include <nn/result/result_HandlingUtility.h>

#include "nim_LocalCommunicationSendUtil.h"

#include <nn/nn_SdkLog.h>

namespace nn { namespace nim { namespace srv {

namespace {
    template <class FuncT>
    Result ForEachContentMetaInfo(const ncm::ContentMetaKey& metaKey, FuncT func) NN_NOEXCEPT
    {
        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuiltInSystem));

        int offset = 0;
        while (NN_STATIC_CONDITION(true))
        {
            const int ListCount = 16;
            ncm::ContentMetaInfo infoList[ListCount];

            int count;
            NN_RESULT_DO(db.ListContentMetaInfo(&count, infoList, ListCount, metaKey, offset));

            for (int i = 0; i < count; i++)
            {
                bool isBreak = false;
                NN_RESULT_DO(func(&isBreak, infoList[i]));
                if (isBreak)
                {
                    NN_RESULT_SUCCESS;
                }
            }

            if (count != ListCount)
            {
                break;
            }

            offset += count;
        }

        NN_RESULT_SUCCESS;
    }

    Result IncludesContentMetaInSystemUpdateMeta(bool* outValue, const ncm::ContentMetaKey& key, const ncm::ContentMetaKey& metaKey) NN_NOEXCEPT
    {
        NN_RESULT_DO(ForEachContentMetaInfo(metaKey, [&outValue, &key](bool* outLoopEnd, const ncm::ContentMetaInfo& info) -> Result
        {
            if (info.ToKey() == key)
            {
                *outValue = true;
                *outLoopEnd = true;
            }
            else
            {
                *outValue = false;
                *outLoopEnd = false;
            }
            NN_RESULT_SUCCESS;
        }));

        NN_RESULT_SUCCESS;
    }

}

    Result LocalCommunicationSendSystemUpdateTask::Initialize(uint32_t ipv4, uint16_t port, const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(key.type == ncm::ContentMetaType::SystemUpdate, ResultInvalidContentMetaKey());

        NN_RESULT_DO(LocalCommunicationSendContentTaskBase::Initialize(ipv4, port));
        m_SystemUpdateKey = key;

        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationSendSystemUpdateTask::GetPackagedContentInfo(ncm::PackagedContentInfo* outValue, const ncm::ContentMetaKey& key) NN_NOEXCEPT
    {
        bool hasKey;
        NN_RESULT_DO(IncludesContentMetaInSystemUpdateMeta(&hasKey, key, m_SystemUpdateKey));
        NN_RESULT_THROW_UNLESS(hasKey || key == m_SystemUpdateKey, ResultLocalCommunicationInvalidContentMetaKey());

        NN_RESULT_DO(GetPackagedContentInfoImpl(outValue, key, ncm::StorageId::BuiltInSystem));

        NN_RESULT_SUCCESS;
    }

    Result LocalCommunicationSendSystemUpdateTask::GetStorage(ncm::StorageId* outValue, const ncm::ContentId& contentId) NN_NOEXCEPT
    {
        NN_UNUSED(contentId);
        *outValue = ncm::StorageId::BuiltInSystem;
        NN_RESULT_SUCCESS;
    }
}}}
