﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>

namespace nn { namespace nim { namespace srv {

namespace HeapUtil {

/**
 * @brief       標準ヒープアロケータ。
 * @details     スレッドセーフです。
 */
void* AllocateHeap(size_t beginAlignment, size_t size) NN_NOEXCEPT;

/**
 * @brief       標準ヒープデアロケータ。
 * @details     スレッドセーフです。
 */
void DeallocateHeap(void* address) NN_NOEXCEPT;

/**
 * @brief       一時ヒープセッション( 広域ヒープ経由 )。
 */
class OnetimeHeapSession
{
public:
    void*   pTop;
    size_t  size;

    OnetimeHeapSession() NN_NOEXCEPT;
    ~OnetimeHeapSession() NN_NOEXCEPT;

    Result Acquire(size_t beginAlignment, size_t capacity) NN_NOEXCEPT;

    Result Expand(size_t beginAlignment, size_t capacity) NN_NOEXCEPT;

    NN_FORCEINLINE bool IsAvailable() const NN_NOEXCEPT
    {
        return (nullptr != pTop && size > 0);
    }

private:
    void Release() NN_NOEXCEPT;
};

/**
 * @brief   サービスフレームワークサービスオブジェクトヒープ定義。
 */
template<size_t CapacityByteSize>
class ServiceFactoryHeap
{
    NN_DISALLOW_COPY(ServiceFactoryHeap);
    NN_DISALLOW_MOVE(ServiceFactoryHeap);

public:
    typedef ::nn::sf::ExpHeapAllocator AllocatorType;
    typedef ::nn::sf::ObjectFactory<AllocatorType::Policy> Factory;

    NN_IMPLICIT ServiceFactoryHeap() NN_NOEXCEPT
    {
        m_HeapHandle = ::nn::lmem::CreateExpHeap(&::nn::util::Get(m_HeapBuffer), sizeof(m_HeapBuffer), ::nn::lmem::CreationOption_NoOption);
        m_Allocator.Attach(m_HeapHandle);
    }

    ~ServiceFactoryHeap() NN_NOEXCEPT
    {
        m_Allocator.Detach();
        ::nn::lmem::DestroyExpHeap(m_HeapHandle);
    }

    NN_FORCEINLINE AllocatorType* GetAllocator() NN_NOEXCEPT
    {
        return &m_Allocator;
    }

private:
    struct NN_ALIGNAS(std::alignment_of<std::max_align_t>::value) HeapCapacity
    {
        Bit8 heap[CapacityByteSize];
    };
    typedef ::nn::util::TypedStorage<HeapCapacity, sizeof(HeapCapacity), NN_ALIGNOF(HeapCapacity)> StorageType;

    AllocatorType           m_Allocator;
    StorageType             m_HeapBuffer;
    ::nn::lmem::HeapHandle  m_HeapHandle;
};

}   // ~HeapUtil

}}}
