﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nim/nim_Result.h>
#include <nn/nim/detail/nim_Log.h>
#include <nn/util/util_Base64.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>

#include "nim_DynamicRightsSyncELicenses.h"
#include "nim_StringUtil.h"

namespace nn { namespace nim { namespace srv {
namespace DynamicRights {

//-----------------------------------------------------------------------------
namespace {
//-----------------------------------------------------------------------------
// デバッグコード
#if !defined(NN_SDK_BUILD_RELEASE)
#define DEBUG_TRACE(...) NN_DETAIL_NIM_TRACE( "[DynamicRights::SyncELicenses] " __VA_ARGS__ )
#else
#define DEBUG_TRACE(...)    static_cast<void>(0)
#endif
//-----------------------------------------------------------------------------

/**
 * @brief       デバイス証明書取得ユーティリティ。
 */
namespace DeviceCertification {

    //! @brief      BASE64エンコードされたデバイス証明書の論理最大文字列長。
    constexpr int LengthMaxAsBase64 = sizeof(::nn::settings::factory::Rsa2048DeviceCertificate) * 4 / 3 + 1;     // 理論値( 133% )

    //! @brief      BASE64エンコードされたデバイス証明書の取得。
    //! @param[out] pOutLength                  BASE64エンコードされたデバイス証明書文字列長( null 終端文字は含まない )。@n
    //! @param[out] pOutCertification           BASE64エンコードされたデバイス証明書文字列の出力バッファ。@n
    //!                                         @ref LengthMaxAsBase64 以上の容量が必要です。@n
    //!                                         末尾に null 終端文字が追加されます。
    //! @param[in]  availableOutValueCapacity   @a pOutCertification が示すメモリ領域の容量。 @ref LengthMaxAsBase64 以上の容量が必要です。
    //! @details    内部で証明書取得のためのバッファ( 576 byte )をスタック領域から消費します。
    Result GetBase64(size_t* pOutLength, char* pOutCertification, int availableOutValueCapacity) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutLength);
        NN_SDK_ASSERT_NOT_NULL(pOutCertification);
        NN_ABORT_UNLESS_GREATER_EQUAL(availableOutValueCapacity, LengthMaxAsBase64);

        ::nn::settings::factory::Rsa2048DeviceCertificate deviceCertificate;
        ::nn::settings::factory::GetEticketDeviceCertificate(&deviceCertificate);
        const auto status = ::nn::util::Base64::ToBase64String(pOutCertification, availableOutValueCapacity
            , deviceCertificate.data
            , sizeof(::nn::settings::factory::Rsa2048DeviceCertificate)
            , ::nn::util::Base64::Mode_NormalNoLinefeed);
        NN_RESULT_THROW_UNLESS(status == ::nn::util::Base64::Status_Success, ResultConvertBase64Failed());

        *pOutLength = ::nn::util::Strnlen(pOutCertification, LengthMaxAsBase64);
        NN_RESULT_SUCCESS;
    }

}   // ~DeviceCertification


//! @brief      ライセンスアーカイブ JSON 用インポートアダプタ。
//! @tparam     ConstantBufferCapacity  内部バッファ。4の倍数である必要があります。
//! @details    内部で BASE64 変換を行う必要があるため、ConstantBufferCapacity + ( ConstantBufferCapacity * 3 / 4 ) バイトが確保されます。
template<size_t ConstantBufferCapacity = 8 * 1024>
class ELicenseImportAdaptor : public HttpJson::Canceler
{
    NN_DISALLOW_COPY(ELicenseImportAdaptor);
    NN_DISALLOW_MOVE(ELicenseImportAdaptor);

public:
    typedef ::nn::es::ELicenseImportContext ImportContext;
    typedef HttpJson::EntryLookup<4, 48>    EntryLookup;
    typedef EntryLookup::JsonPathType       JsonPathType;

    static constexpr size_t BufferCapacity = ConstantBufferCapacity;
    NN_STATIC_ASSERT(0 == (BufferCapacity % 4));
    NN_STATIC_ASSERT(BufferCapacity > 0u);

    //! @brief  コンストラクタ。
    explicit ELicenseImportAdaptor(const ImportContext* pContext) NN_NOEXCEPT
        : m_pContext(pContext), m_Result(::nn::ResultSuccess()), m_FilledSize(0u), m_CanImport(false) {}

    //! @brief  デストラクタ。( -Wdelete-non-virtual-dtor で警告が出るため実装 )
    virtual ~ELicenseImportAdaptor() NN_NOEXCEPT {}

    //! @brief  rapidjson::GenericReader 用
    char* PutBegin() NN_NOEXCEPT
    {
        m_Result = nn::ResultSuccess();
        m_FilledSize = 0u;
        return m_Buffer;
    }

    //! @brief  rapidjson::GenericReader 用
    void Put(char value) NN_NOEXCEPT
    {
        if (m_FilledSize >= BufferCapacity)
        {
            // オーバーしたらインポート
            Import(m_Buffer, m_FilledSize);
            m_FilledSize = 0u;
        }
        m_Buffer[m_FilledSize++] = value;
    }

    //! @brief  rapidjson::GenericReader 用
    size_t PutEnd(char* pValue) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(pValue == m_Buffer);

        // 最終インポートチェック。
        const auto filledSize = m_FilledSize;
        if (filledSize > 0u && Import(m_Buffer, filledSize))
        {
            // インポートした( 値を処理した )なら、値は終端文字のみの文字列として rapidjson へ通知。( 判定効率化 )
            // rapidjson::GenericReader::ParseString() 内で ASSERT((PutEnd() - 1) < 0xFFFFFFFF) しているため。
            m_CanImport = false;
            m_FilledSize = 0u;
            pValue[0] = '\0';
            return 1u;
        }
        return filledSize;
    }

    //! @brief  rapidjson::GenericReader 用
    void Flush() NN_NOEXCEPT {}

    //! @brief  HackedEventHandler クラス経由のキー検出用
    void UpdateImportableCondition(bool enable) NN_NOEXCEPT
    {
        m_CanImport = enable;
    }

    //! @brief      内部異常値の取得。
    //!
    //! @details    本クラスインスタンスをキャンセラ登録する事で内部異常が発生した場合には、@ref ::nn::http::json::ImportJsonByRapidJson() の返値に http::ResultCanceled が返ります。@n
    //!             その場合、本メソッドで異常値を取得し適切に処理してください。@n
    //!             異常状態が発生していない場合は、ResultSuccess が返されます。
    Result GetFailure() const NN_NOEXCEPT
    {
        return m_Result;
    }

    // 内部処理で失敗があったらパースをキャンセルする。
    virtual bool IsCancelled() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Result.IsFailure();
    }

    // 以下未使用
    void Update(const JsonPathType&, const char*, int) NN_NOEXCEPT {}
    void NotifyObjectBegin(const JsonPathType&) NN_NOEXCEPT {}
    void NotifyObjectEnd(const JsonPathType&) NN_NOEXCEPT {}
    void Update(const JsonPathType&, bool) NN_NOEXCEPT {}
    void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, double) NN_NOEXCEPT {}
    void Update(const JsonPathType&, int64_t) NN_NOEXCEPT {}
    void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT {}

private:
    bool Import(const char* pSourceBase64, size_t sourceSize) NN_NOEXCEPT
    {
        if (m_CanImport && m_Result.IsSuccess())
        {
            // 末尾終端文字チェック ( rapidjson は文字列要素値の終端 `"` 検出時に '\0' 入力を行う )
            // 末尾終端はインポート対象外なので除去調整。
            // 4 倍数違反は util::Base64 側で判定している。
            const auto importSize = ('\0' == pSourceBase64[sourceSize - 1]) ? sourceSize - 1 : sourceSize;
            m_Result = ImportImpl(pSourceBase64, importSize);
            return true;
        }
        return false;
    }

    Result ImportImpl(const char* pSourceBase64, size_t sourceSize) NN_NOEXCEPT
    {
        // sourceSize が4の倍数である事を確認する。
        NN_RESULT_THROW_UNLESS(0 == (sourceSize % 4), ResultConvertBase64Failed());

        // BASE64 デコード。
        size_t outBytes;
        const auto status = ::nn::util::Base64::FromBase64String(&outBytes, m_Binary, sizeof(m_Binary), pSourceBase64, sourceSize, ::nn::util::Base64::Mode_NormalNoLinefeed);
        NN_RESULT_THROW_UNLESS(status == ::nn::util::Base64::Status_Success, ResultConvertBase64Failed());

        // インポート実施。
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        NN_RESULT_DO(::nn::es::ImportELicenseArchive(*m_pContext, m_Binary, outBytes));
#else
        NN_UNUSED(outBytes);
#endif

        DEBUG_TRACE("ELicense archive imported( base64: %zu byte, decoded binary: %zu byte )\n", sourceSize, outBytes);
        NN_RESULT_SUCCESS;
    }

    const ImportContext* const  m_pContext;
    Result                      m_Result;
    size_t                      m_FilledSize;
    char                        m_Buffer[BufferCapacity];
    char                        m_Binary[BufferCapacity * 3 / 4];   // FromBase64String() の出力に null 終端付与はないので +1 不要.
    bool                        m_CanImport;
};

//-----------------------------------------------------------------------------
//! @brief  長大要素値を持つ JSON ストリーム用パーサーイベントハンドラ。
template <typename JsonAdaptorType, typename CancelableType>
class HackedEventHandler : public ::nn::http::json::EventHandlerForRapidJson<JsonAdaptorType, CancelableType>
{
private:
    typedef ::nn::http::json::EventHandlerForRapidJson<JsonAdaptorType, CancelableType> BaseType;
    JsonAdaptorType* const pAdaptor;

public:
    HackedEventHandler(JsonAdaptorType* pJsonAdaptor, const CancelableType* pCancelable) NN_NOEXCEPT
        : BaseType(*pJsonAdaptor, pCancelable)
        , pAdaptor(pJsonAdaptor)
    {
        NN_SDK_ASSERT_NOT_NULL(pJsonAdaptor);
    }

    bool Key(const char* pStringValue, ::nne::rapidjson::SizeType length, bool copy) NN_NOEXCEPT
    {
        const auto accept = BaseType::Key(pStringValue, length, copy);
        if (accept)
        {
            // true なら、eLicense アーカイブバイナリ要素のキーである事が確定。
            constexpr char keyName[] = "elicense_archive";
            const auto findKey = ((sizeof(keyName) - 1) == length && 0 == ::nn::util::Strncmp(pStringValue, keyName, static_cast<int>(length)));
            pAdaptor->UpdateImportableCondition(findKey);
        }
        return accept;
    }
};

//-----------------------------------------------------------------------------
//! @brief  長大要素値を持つ JSON ストリーム用パーサー。
template <typename JsonErrorMap, typename JsonAdaptorType, typename InputStreamType, typename CancelableType>
Result ImportJson(JsonAdaptorType* pAdaptor, InputStreamType* pStream, const CancelableType* pCancelable) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pAdaptor);
    NN_SDK_ASSERT_NOT_NULL(pStream);

    typedef ::nn::http::json::NoAllocatorForRapidJson NoAllocatorForRapidJson;
    typedef ::nne::rapidjson::GenericReader<::nne::rapidjson::UTF8<>, ::nne::rapidjson::UTF8<>, NoAllocatorForRapidJson> Reader;
    HackedEventHandler<JsonAdaptorType, CancelableType> handler(pAdaptor, pCancelable);
    NoAllocatorForRapidJson allocator;

    auto s = Reader(&allocator, 1u).Parse<
        ::nne::rapidjson::kParseInsituFlag |
        ::nne::rapidjson::kParseStopWhenDoneFlag |
        ::nne::rapidjson::kParseValidateEncodingFlag>(*pStream, handler);
    const auto result = ::nn::http::json::HandleParseResult<JsonErrorMap>(s);
    if (!result.IsSuccess())
    {
        if (pCancelable && pCancelable->IsCancelled())
        {
            // キャンセルが最優先
            NN_RESULT_THROW(nn::http::ResultCancelled());
        }
        NN_RESULT_DO(pStream->GetResult()); // IO エラーは次点
        NN_RESULT_THROW(result); // JSON パーサのエラーは最低優先度
    }
    NN_RESULT_SUCCESS;
}



//-----------------------------------------------------------------------------
}   // ~unnamed
//-----------------------------------------------------------------------------

//!============================================================================
//! @name ELicenseArchiveProfile 実装
//-----------------------------------------------------------------------------
ELicenseArchiveProfile::ELicenseArchiveProfile(const AccountId& accountId, ImportContext* pContext) NN_NOEXCEPT
    :  m_pImportContext(pContext), m_AccountId(accountId)
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
}

Result ELicenseArchiveProfile::OnQueryAccessProfile(AccessProfile* pOut, char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);
    constexpr char base[] = "/v1/elicense_archives/publish";
    NN_ABORT_UNLESS(sizeof(base) <= availablePathUrlCapacity && availablePathUrlCapacity <= static_cast<size_t>(std::numeric_limits<int>::max()));

    const auto capacity = static_cast<int>(availablePathUrlCapacity);
    NN_ABORT_UNLESS(::nn::util::Strlcpy(pOutPathUrl, base, capacity) < capacity);

    pOut->uid = ::nn::account::InvalidUid;
    pOut->naId = m_AccountId;
    pOut->timeout = 60 * 10; // アーカイブの大きさ次第だけども、暫定 10 min.
    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveProfile::OnSetupRequestBody(ConnectionType* pConnection) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConnection);

    // ライセンスアーカイブ同期用チャレンジ取得。
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    const auto challenge = m_pImportContext->GetChallenge().value;
#else
    const ::nn::es::ELicenseImportContext::Challenge challenge = {};
#endif

    // ポストデータボディ作成。
    char data[128 + DeviceCertification::LengthMaxAsBase64];
    const char PostDataPrefix[] = "{\"challenge\":\"%016llx\",\"certificate\":\"";
    const size_t prefixLength = ::nn::util::SNPrintf(data, sizeof(data), PostDataPrefix, challenge);

    size_t certificationLength = 0;
    int capacity = static_cast<int>(sizeof(data) - prefixLength);
    NN_RESULT_DO(DeviceCertification::GetBase64(&certificationLength, &data[prefixLength], capacity));
    capacity -= static_cast<int>(certificationLength);

    const char PostDataSuffix[] = "\"}";
    const size_t postLength = prefixLength + certificationLength + (sizeof(PostDataSuffix) - 1);
    NN_ABORT_UNLESS(::nn::util::Strlcpy(&data[prefixLength + certificationLength], PostDataSuffix, capacity) < capacity);
    DEBUG_TRACE("Post filed generate complete, (length: %zu)=>`%s`\n", postLength, data);
    NN_RESULT_DO(pConnection->SetTransactionPostFields(data, postLength, true));    // クローンコピーするので CURL 経由でヒープ使います。

    // "Content-Length", "Content-Type" ヘッダ。
    NN_ABORT_UNLESS(::nn::util::SNPrintf(data, sizeof(data), "Content-Length:%zu", postLength) < sizeof(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader(data));
    NN_RESULT_DO(pConnection->SetTransactionHeader("Content-Type:application/json"));
    NN_RESULT_SUCCESS;
}

Result ELicenseArchiveProfile::OnResolveResponse(InputStream* pInputStream) NN_NOEXCEPT
{
    // 8 KiB + 6 KiB の BASE64 アーカイブデータ受信アダプタ。( Global ヒープからインスタンスを獲得 )
    typedef ELicenseImportAdaptor<8 * 1024> ImportAdaptorClass;
    std::unique_ptr<ImportAdaptorClass> adaptor(new ImportAdaptorClass(m_pImportContext));
    auto pAdaptor = adaptor.get();

    // JSON パース。
    HttpJson::Stream::ProxyStream<InputStream, ImportAdaptorClass> stream(pInputStream, pAdaptor);
    NN_RESULT_TRY(ImportJson<::nn::http::json::DefaultJsonErrorMap>(pAdaptor, &stream, static_cast<HttpJson::Canceler*>(pAdaptor)))
        NN_RESULT_CATCH_CONVERT(::nn::http::ResultCanceled, pAdaptor->GetFailure())
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

//!============================================================================
//! @name ELicenseSyncDoneProfile 実装
//-----------------------------------------------------------------------------
ELicenseSyncDoneProfile::ELicenseSyncDoneProfile(const AccountId& accountId) NN_NOEXCEPT
    : m_AccountId(accountId)
{
}

Result ELicenseSyncDoneProfile::BeginImport(ImportContext** pOutContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutContext);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_RESULT_DO(::nn::es::BeginImportELicenseArchive(&m_ImportContext, m_AccountId));
    *pOutContext = &m_ImportContext;
#else
    *pOutContext = nullptr;
#endif

    NN_RESULT_SUCCESS;
}

Result ELicenseSyncDoneProfile::EndImport() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    return ::nn::es::EndImportELicenseArchive(&m_ArchiveId, m_ImportContext);
#else
    NN_RESULT_SUCCESS;
#endif
}

Result ELicenseSyncDoneProfile::OnQueryAccessProfile(AccessProfile* pOut, char* pOutPathUrl, size_t availablePathUrlCapacity) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    constexpr char base[] = "/v1/elicense_archives/%s/report";
    char stringForId[::nn::es::ELicenseArchiveId::StringSize + 1] = {};
    const size_t length = ::nn::util::SNPrintf(pOutPathUrl, availablePathUrlCapacity, base, m_ArchiveId.ToString(stringForId, sizeof(stringForId)));
    NN_ABORT_UNLESS(length < availablePathUrlCapacity);

    pOut->uid = ::nn::account::InvalidUid;
    pOut->naId = m_AccountId;
    NN_RESULT_SUCCESS;
}

Result ELicenseSyncDoneProfile::OnSetupRequestBody(ConnectionType* pConnection) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConnection);
    NN_FUNCTION_LOCAL_STATIC(char, s_Empty, [] = "");
    NN_RESULT_DO(pConnection->SetTransactionPostFields(s_Empty));
    NN_RESULT_DO(pConnection->SetTransactionCustomMethod("PUT"));
    NN_RESULT_SUCCESS;
}

Result ELicenseSyncDoneProfile::OnResolveResponse(InputStream* pInputStream) NN_NOEXCEPT
{
    // 報告系 API は解析すべきレスポンスストリームはなし。
    NN_UNUSED(pInputStream);
    NN_RESULT_SUCCESS;
}

//!============================================================================
//! @name AsyncSyncELicensesImpl 実装
//
// NOTE:
//  Shimレイヤの nim::AsyncResult の Get() アクセスにおいて Wait() される事を前提想定した実装です。
//  非同期に Get() を呼び出す要件になった場合は排他が必要です。
//-----------------------------------------------------------------------------
AsyncSyncELicensesImpl::AsyncSyncELicensesImpl() NN_NOEXCEPT
    : AsyncBase(this, GetSharedThreadAllocator()), Executor()
    , m_AccountId(::nn::account::InvalidNintendoAccountId)
{
}

//-----------------------------------------------------------------------------
AsyncSyncELicensesImpl::~AsyncSyncELicensesImpl() NN_NOEXCEPT
{
    Join();
}

//-----------------------------------------------------------------------------
Result AsyncSyncELicensesImpl::Initialize(DeviceContext* pDeviceContext, const AccountId& accountId) NN_NOEXCEPT
{
    NN_RESULT_DO(Executor::Initialize(pDeviceContext));
    m_AccountId = accountId;
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
Result AsyncSyncELicensesImpl::Execute() NN_NOEXCEPT
{
    return DragonsAsyncAccessTaskBase::RequestTaskFinished(OnExecute());
}

//-----------------------------------------------------------------------------
Result AsyncSyncELicensesImpl::OnExecute() NN_NOEXCEPT
{
    ELicenseSyncDoneProfile doneProfile(m_AccountId);

    ELicenseSyncDoneProfile::ImportContext* pContext = nullptr;
    NN_RESULT_DO(doneProfile.BeginImport(&pContext));

    // ライセンス同期要求.
    ELicenseArchiveProfile archiveProfile(m_AccountId, pContext);
    NN_RESULT_DO(Request(&archiveProfile));

    NN_RESULT_DO(doneProfile.EndImport());

    // ライセンス同期完了通知.
    NN_RESULT_DO(Request(&doneProfile));
    NN_RESULT_SUCCESS;
}


}   // ~DynamicRights
}}} // ~nn::nim::srv
